import logging
import requests
import waffle

from typing import Optional

from constance import config
from django.conf import settings
from django.core.files.base import ContentFile
from django.utils.timezone import datetime

from intranet.femida.src.core.workflow import WorkflowError
from intranet.femida.src.oebs.fake_data import get_fake_bp
from intranet.femida.src.utils.tvm2_client import (
    get_service_ticket,
)


logger = logging.getLogger(__name__)


# Note: мы с OeBS иногда общаемся человекочитаемыми ошибками :/
OEBS_PERSON_NOT_FOUND_ERROR = 'Персона не найдена'
OEBS_PERSON_EXISTS_ERROR = 'Человек с такими данными уже существует.'


class BudgetPositionError(WorkflowError):

    def __init__(self, message='budget_position_error', response=None):
        self.message = message
        self.response = response


class EmptyBudgetPositionError(BudgetPositionError):
    pass


class OebsPersonError(WorkflowError):

    def __init__(self, message='oebs_person_error'):
        self.message = message


class OebsPersonExistsError(OebsPersonError):
    pass


class OebsFormulaError(WorkflowError):
    pass


class BaseOebsAPI:

    _url = None
    error_class = None
    tvm_client_id = settings.TVM_OEBS_API_CLIENT_ID

    @classmethod
    def _raise_for_status(cls, response: requests.Response, method: str):
        if response.status_code == 500:
            try:
                raise cls.error_class(response.json()['message'])
            except (ValueError, KeyError):
                pass

        # Если OeBS в ручке bpInfo отдаёт пустоту с кодом 200,
        # значит БП есть, но просто в невалидном для Фемиды состоянии (EMPTY) :/
        if method == 'bpInfo' and response.ok and not response.content:
            raise EmptyBudgetPositionError

        response.raise_for_status()

    @classmethod
    def _request(cls, http_method: str, method: str, is_json_response: bool, params: dict = None,
                 data: dict = None):
        headers = {
            'X-Ya-Service-Ticket': get_service_ticket(cls.tvm_client_id),
        }

        try:
            response = requests.request(
                method=http_method,
                url=cls._url + method,
                params=params,
                json=data,
                headers=headers,
                verify=settings.YANDEX_INTERNAL_CERT_PATH,
                timeout=config.OEBS_TIMEOUT,
            )
            cls._raise_for_status(response, method)
        except requests.Timeout:
            logger.exception('Failed to get data from OEBS (timeout), %s', data)
            raise cls.error_class('oebs_timeout')
        except requests.RequestException:
            logger.exception('Failed to get data from OEBS, %s', data)
            raise cls.error_class('oebs_request_failed')

        try:
            result = response.json() if is_json_response else response.content
        except ValueError:
            logger.exception('OEBS response is not a valid json, %s', data)
            raise cls.error_class('oebs_invalid_response', response=response)

        return result

    @classmethod
    def _get(cls, method: str, is_json_response=True, **params):
        return cls._request('GET', method, is_json_response, params=params)

    @classmethod
    def _post(cls, method: str, is_json_response=True, **data):
        return cls._request('POST', method, is_json_response, data=data)


class OebsAPI(BaseOebsAPI):

    _url = f'{settings.OEBS_API_URL}/rest/'
    error_class = BudgetPositionError

    @classmethod
    def get_budget_position(cls, bp_id, date=None):
        logger.info('Getting info for budget position %s', bp_id)

        if waffle.switch_is_active('enable_fake_oebs'):
            data = get_fake_bp(bp_id)
        else:
            data = cls._post('bpInfo', bpNumber=bp_id, dateFrom=date)

        data['id'] = bp_id
        data['status'] = data.get('hiring')
        data['wage_rate'] = data.get('rate')
        data['is_headcount'] = data.get('headcount', '').lower() == 'да'
        return data

    @classmethod
    def get_budget_position_form_data(cls, bp_id, date=None):
        # Note: этот метод нигде сейчас явно в бизнес-логике не используется,
        # но он полезен для диагностики проблем
        return cls._post('getPosition', positionNumber=bp_id, effectiveDate=date)

    @classmethod
    def reset_budget_position_status(cls, bp_id):
        """
        Переводит БП в состояние VACANCY
        """
        if waffle.switch_is_active('enable_fake_oebs'):
            return True

        logger.info('Resetting status for budget position %s', bp_id)
        data = cls._post('updateBpHiring', bpNumber=bp_id)
        return data.get('status') == 'OK'

    @classmethod
    def push_bank_details(cls, username, full_name, **params):
        """
        Запись банковских реквизитов нового сотрудника в OeBS.
        :return: bool - флаг успеха, unicode - описание
        """
        params['login'] = username
        params['emp_fio'] = full_name
        params['source'] = 'femida'

        # БИК и номер счёта - обязательные параметры.
        # Ошибку не рейзим, просто пишем в лог,
        # чтобы таск зря не перезапускался.
        if not params.get('bic') or not params.get('bank_account'):
            logger.warning('OEBS push_bank_details(%s): no bic or bank_account', username)
            return False, 'no_bic_or_bank_account'

        # Статус заявления: не отправлено (1), отправлено (2), подписано (3)
        params['statement_status'] = '2'

        result = cls._post('bankdetails', params=params)

        if result.get('state') != 'OK':
            logger.error('OEBS push_bank_details(%s): %s', username, result)
            raise cls.error_class('oebs_error')

        return True, 'success'

    @classmethod
    def update_bank_login(cls, old_login, new_login):
        """
        Изменение логина для банковских реквизитов
        """
        if not waffle.switch_is_active('enable_update_bank_login'):
            return
        data = {
            'from': old_login,
            'to': new_login,
        }
        result = cls._post('updatelogin', **data)
        if result.get('status') == 'ERROR':
            raise cls.error_class()
        elif result.get('status') == 'NOT_FOUND':
            # Note: если логин не найден, с высокой вероятностью мы просто
            # и не загружали банк.реквизиты по такому логину в OeBS.
            # Мы у себя ни факт загрузки, ни сами реквизиты не храним, поэтому пытаемся
            # отправить запрос на изменение логина для всех офферов.
            # На всякий случай логируем это событие, чтобы легче было расследовать проблемы.
            logger.warning('OEBS update_bank_login(%s, %s): not found', old_login, new_login)

    @classmethod
    def get_login_by_document(cls, document_type, document_id):
        """
        :return: Стока логина, если он найден. Иначе None
        :raise: BudgetPositionError
        """
        result = cls._post(
            method=f'find_{document_type}',
            **{document_type: document_id}
        )

        error = result['error']
        if error:
            if error == OEBS_PERSON_NOT_FOUND_ERROR:
                return
            raise cls.error_class(f'oebs_error: {error}')
        return result['login']


class OebsHireAPI(BaseOebsAPI):

    _url = f'{settings.OEBS_HIRE_API_URL}/rest/'
    error_class = OebsPersonError

    @classmethod
    def _check_result_for_errors(cls, result: dict):
        """
        Проверяет ответ Я.Найма на наличие ошибок и предупреждений.
        Note: Я.Найм в одних ручках в случае ошибок отдаёт список ошибок в errors,
        а в других ручках отдаёт status=ERROR и текст ошибки в message.
        """
        if result.get('errors'):
            raise cls.error_class(', '.join(result['errors']))
        if result.get('status') == 'ERROR':
            raise cls.error_class(result.get('message', ''))
        if result.get('warnings'):
            logger.warning(
                'Warnings were received from OEBS Hire: %s',
                ', '.join(result['warnings'])
            )

    @classmethod
    def _request(cls, *args, **kwargs):
        result = super()._request(*args, **kwargs)
        cls._check_result_for_errors(result)
        return result

    @classmethod
    def create_person(cls, data: dict = None) -> Optional[str]:
        data = data or {}
        try:
            result = cls._post('hirePerson', **data)
        except cls.error_class as exc:
            if OEBS_PERSON_EXISTS_ERROR in exc.message:
                raise OebsPersonExistsError(exc.message)
            raise
        return result['id']

    @classmethod
    def remove_person(cls, person_id: int):
        cls._post('cancelHire', personId=person_id)

    @classmethod
    def update_login(cls, person_id: int, login: str):
        cls._post(
            method='changeLoginHire',
            personId=person_id,
            login=login,
        )

    @classmethod
    def update_assignment(cls, person_id: int, department_id: int, join_at: datetime.date):
        cls._post(
            method='changeAssignHire',
            personId=person_id,
            departmentId=department_id,
            dateHire=join_at.isoformat(),
        )


class OebsFormulaAPI(BaseOebsAPI):

    _url = settings.OEBS_FORMULA_API_URL
    error_class = OebsFormulaError
    tvm_client_id = settings.TVM_OEBS_FORMULA_API_CLIENT_ID

    @classmethod
    def _raise_for_status(cls, response: requests.Response, method: str):
        try:
            super()._raise_for_status(response, method)
        except requests.exceptions.HTTPError:
            raise cls.error_class(response.content.decode('utf-8'))

    @classmethod
    def generate_offer(cls, filename, data):
        content = cls._post(method='offer/generate', is_json_response=False, **data)
        return ContentFile(content=content, name=filename)


get_budget_position = OebsAPI.get_budget_position
reset_budget_position_status = OebsAPI.reset_budget_position_status
