import logging

from constance import config
from dateutil.relativedelta import relativedelta
from django.conf import settings
from django.db.models import Prefetch

from intranet.femida.src.candidates.models import Verification
from intranet.femida.src.offers.choices import (
    CONTRACT_TYPES,
    CONTRACT_TYPES_TRANSLATIONS,
    CITIZENSHIP,
    WORK_PLACES,
    FORM_TYPES,
)
from intranet.femida.src.staff.models import DepartmentAdaptation
from intranet.femida.src.utils.pluralization import get_russian_plural_form
from intranet.femida.src.utils.strings import fetch_comma_separated_integers


logger = logging.getLogger(__name__)


CONTRACT_TYPE_DESCRIPTION_MAP = {
    None: None,
    '': None,
    CONTRACT_TYPES.indefinite: '{contract_type}',
    CONTRACT_TYPES.fixed_term: '{contract_type} ({contract_term} {contract_term_unit})',
    CONTRACT_TYPES.fixed_term_date: '{contract_type} ({contract_term_date})',
    CONTRACT_TYPES.project: '{contract_type}',
    CONTRACT_TYPES.temp_replacement: '{contract_type}',
}


BOOTCAMP_QUEUE_MAP = {
    settings.BOOTCAMP_BACK_DEPARTMENT_ID: settings.STARTREK_BOOTCAMP_BACK_QUEUE,
    settings.BOOTCAMP_FRONT_DEPARTMENT_ID: settings.STARTREK_BOOTCAMP_FRONT_QUEUE,
}


def get_offer_contract_type_description(offer):
    description_template = CONTRACT_TYPE_DESCRIPTION_MAP.get(offer.contract_type)
    if not description_template:
        return None
    return description_template.format(
        contract_type=CONTRACT_TYPES_TRANSLATIONS.get(offer.contract_type),
        contract_term=offer.contract_term,
        contract_term_unit=get_russian_plural_form(
            number=offer.contract_term or 0,
            singular='месяц',
            plural='месяца',
            plural_many='месяцев',
        ),
        contract_term_date=(
            offer.contract_term_date.strftime('%d.%m.%Y')
            if offer.contract_term_date
            else None
        ),
    )


def _get_offices_hiring_on_tuesday():
    return fetch_comma_separated_integers(config.OFFICES_HIRING_ON_TUESDAY)


def _get_offices_hiring_on_wednesday():
    return fetch_comma_separated_integers(config.OFFICES_HIRING_ON_WEDNESDAY)


def is_hiring_foreigner_inside(citizenship, organization, department):
    """
    Признак того, что в штат нанимают иностранца
    """
    return (
        citizenship is not None
        and citizenship not in ('', CITIZENSHIP.RU)
        and organization.is_russian
        and not department.is_in_tree(settings.EXTERNAL_DEPARTMENT_ID)
    )


def is_foreigner_offer(offer):
    # Note: по внутр.офферу мы не можем наверняка сказать, иностранец это или нет.
    # Поэтому всегда считаем, что нет
    if offer.is_internal:
        return False
    return is_hiring_foreigner_inside(offer.profile.citizenship, offer.org, offer.department)


def is_foreigner_preprofile(remote_preprofile):
    return is_hiring_foreigner_inside(
        citizenship=remote_preprofile.citizenship,
        organization=remote_preprofile.org,
        department=remote_preprofile.department,
    )


def is_hiring_intern_inside(offer):
    return (
        offer.is_internship
        and offer.org.is_russian
    )


def has_no_documents(citizenship, passport_pages, snils, form_type=FORM_TYPES.russian):
    """
    Проверка на недостающие документы по условиям:
    1. Российская анкета + гражданин РФ - если отсутствует паспорт либо СНИЛС.
    2. Российская анкета + иностранный гражданин - если отсутствует паспорт.
    3. Международная анкета - всегда.
    """
    return (
        form_type == FORM_TYPES.international
        or (
            citizenship not in ('', CITIZENSHIP.RU)
            and not passport_pages
        )
        or (
            citizenship == CITIZENSHIP.RU
            and not all([passport_pages, snils])
        )
    )


def has_no_documents_for_offer(offer):
    return has_no_documents(
        citizenship=offer.profile.citizenship,
        passport_pages=offer.passport_pages,
        snils=offer.snils,
        form_type=offer.form_type,
    )


def has_no_documents_for_preprofile(remote_preprofile):
    return has_no_documents(
        citizenship=remote_preprofile.citizenship,
        passport_pages=remote_preprofile.passport_pages,
        snils=remote_preprofile.snils,
    )


def get_join_isoweekday(offer):
    """
    День недели выхода сотрудника в зависимости от:
    - подразделения;
    - места работы;
    - офиса;
    - признака "стажер".

    Возвращает число 1-7, где
    1 - понедельник,
    ...
    7 - воскресенье,
    -1 - любой день.
    """
    if offer.department and offer.department.is_in_tree(settings.YANDEX_DEPARTMENT_ID):
        # Стажеры - среда
        if offer.is_internship:
            return 3
        # Надомники - среда
        if offer.work_place == WORK_PLACES.home:
            return 3

        # Вторник либо среда, в зависимости от офиса
        if offer.office_id in _get_offices_hiring_on_tuesday():
            return 2
        if offer.office_id in _get_offices_hiring_on_wednesday():
            return 3

    return -1


def is_adaptation_needed(offer):
    # Адаптация нужна только для людей, нанимающихся в ветку "Яндекс"
    # и некоторых дополнительных отделов
    if (
        not offer.department.is_in_tree(settings.YANDEX_DEPARTMENT_ID) and
        not is_outstaff_adaptation_needed(offer.department_id)
    ):
        return False
    return (
        offer.is_external
        or offer.is_rotation_within_yandex
        or offer.employee.department.is_in_tree(settings.OUTSTAFF_DEPARTMENT_ID)
    )


def has_signup_bonus(offer):
    if offer.signup_bonus or offer.signup_2year_bonus:
        return True
    if offer.signup_bonus_gross or offer.signup_2year_bonus_gross:
        return True
    return False


def is_bootcamp_offer(offer):
    return offer.department.is_in_trees(BOOTCAMP_QUEUE_MAP)


def is_eds_needed(instance):
    return instance.org.is_russian


def get_offer_contract_end(offer, join_at=None):
    """
    Вычисляет дату окончания срочного трудового договора, если это возможно
    """
    if offer.contract_type == CONTRACT_TYPES.fixed_term_date:
        return offer.contract_term_date
    if offer.contract_type == CONTRACT_TYPES.fixed_term and offer.contract_term:
        join_at = join_at or offer.join_at
        return join_at + relativedelta(months=offer.contract_term, days=-1)
    return None


def get_candidate_alive_verifications_prefetch():
    return Prefetch(
        lookup='candidate__verifications',
        queryset=Verification.objects.alive(),
        to_attr='alive_verifications',
    )


def is_outstaff_adaptation_needed(department_id):
    return _get_department_adaptation_qs(department_id).exists()


def get_department_adaptation_queue(department_id):
    return (
        _get_department_adaptation_qs(department_id)
        .values_list('queue', flat=True)
        .first()
    )


def get_department_adaptation_issue_type(department_id):
    return (
        _get_department_adaptation_qs(department_id)
        .values_list('issue_type', flat=True)
        .first()
    )


def _get_department_adaptation_qs(department_id):
    return DepartmentAdaptation.objects.filter(department_id=department_id)
