import logging

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError

from intranet.femida.src.offers.choices import (
    OFFER_STATUSES,
    EMPLOYEE_TYPES,
    INTERNAL_EMPLOYEE_TYPES,
)
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.utils.newhire import NewhireAPI


logger = logging.getLogger(__name__)
User = get_user_model()


class LoginErrorEnum:

    bad_length = 'login_error_bad_length'
    bad_symbols = 'login_error_bad_symbols'
    not_unique = 'login_error_not_unique'
    similar_exists = 'login_error_similar_exists'
    unsuitable = 'login_error_unsuitable'

    not_former_employee = 'login_error_not_former_employee'
    not_current_employee = 'login_error_not_current_employee'
    newhire_check_failed = 'login_error_newhire_check_failed'


LOGIN_VALIDATION_ERROR_PARAMS = {
    LoginErrorEnum.bad_length: {
        'min_length': settings.MIN_LOGIN_LENGTH,
        'max_length': settings.MAX_LOGIN_LENGTH,
    },
    LoginErrorEnum.bad_symbols: {
        'valid_symbols': ['a-z', '0-9', '"-"'],
    },
}


NEWHIRE_LOGIN_VALIDATION_ERRORS_MAP = {
    'login_bad_length': LoginErrorEnum.bad_length,
    'login_bad_symbols': LoginErrorEnum.bad_symbols,
    'login_not_unique': LoginErrorEnum.not_unique,
    'login_likeness': LoginErrorEnum.similar_exists,

    'login_in_blacklist': LoginErrorEnum.unsuitable,
    'login_ldap_conflict': LoginErrorEnum.unsuitable,
    'login_maillist_conflict': LoginErrorEnum.unsuitable,
    'login_dns_conflict': LoginErrorEnum.unsuitable,
}


class LoginValidationError(ValidationError):

    def __init__(self, code):
        super().__init__(
            code=code,
            message=code,
            params=LOGIN_VALIDATION_ERROR_PARAMS.get(code),
        )


class NewhireLoginValidationError(LoginValidationError):

    def __init__(self, code):
        code = NEWHIRE_LOGIN_VALIDATION_ERRORS_MAP.get(code, LoginErrorEnum.unsuitable)
        super().__init__(code)


class LoginValidator:
    """
    Набор валидаторов для проверки логина.
    """
    def __init__(self, employee_type=EMPLOYEE_TYPES.new, newhire_id=None, exclude_offer_ids=None):
        exclude_offer_ids = exclude_offer_ids or []
        if not isinstance(exclude_offer_ids, (list, tuple)):
            exclude_offer_ids = [exclude_offer_ids]
        self.employee_type = employee_type
        self.newhire_id = newhire_id
        self.exclude_offer_ids = exclude_offer_ids

    def _validate(self, validators, login):
        for validator in validators:
            validator(login)

    def validate(self, login):
        if not login:
            raise LoginValidationError(LoginErrorEnum.unsuitable)
        if self.employee_type == EMPLOYEE_TYPES.new:
            self._validate(self.validators, login)
        else:
            self.validate_old_login(login)

    def validate_old_login(self, login):
        is_dismissed = self.employee_type == EMPLOYEE_TYPES.former
        user = (
            User.objects
            .filter(
                username=login,
                is_dismissed=is_dismissed,
            )
            .select_related('department')
            .first()
        )

        if is_dismissed:
            if not user:
                raise LoginValidationError(LoginErrorEnum.not_former_employee)
            return

        if not user:
            raise LoginValidationError(LoginErrorEnum.not_current_employee)

        if self.employee_type in INTERNAL_EMPLOYEE_TYPES:
            # Внутренний оффер можно выставлять только действующим сотрудникам
            # веток Яндекс и Outstaff
            valid_department_ids = [
                settings.YANDEX_DEPARTMENT_ID,
                settings.OUTSTAFF_DEPARTMENT_ID,
            ]
        else:
            # Остальных действующих сотрудников можно нанимать
            # только для ветки "Внешние консультанты"
            valid_department_ids = [
                settings.EXTERNAL_DEPARTMENT_ID,
            ]
        if not user.department.is_in_trees(valid_department_ids):
            raise LoginValidationError(LoginErrorEnum.not_current_employee)

    def validate_length(self, login):
        if len(login) < settings.MIN_LOGIN_LENGTH or len(login) > settings.MAX_LOGIN_LENGTH:
            raise LoginValidationError(LoginErrorEnum.bad_length)

    def validate_uniqueness_in_offers(self, login):
        invalid_offer_statuses = [
            OFFER_STATUSES.rejected,
            OFFER_STATUSES.deleted,
        ]
        offer_exists = (
            Offer.unsafe
            .exclude(status__in=invalid_offer_statuses)
            .exclude(id__in=self.exclude_offer_ids)
            .filter(username=login)
            .exists()
        )
        if offer_exists:
            raise LoginValidationError(LoginErrorEnum.not_unique)

    def validate_in_newhire(self, login):
        data, status_code = NewhireAPI.check_login(login, self.newhire_id)
        if status_code == 400:
            try:
                error_code = data['errors']['login'][0]['code']
            except (KeyError, IndexError):
                error_code = LoginErrorEnum.unsuitable
            raise NewhireLoginValidationError(error_code)
        elif status_code != 200:
            raise LoginValidationError(LoginErrorEnum.newhire_check_failed)

    @property
    def validators(self):
        return [
            self.validate_length,
            self.validate_uniqueness_in_offers,
            self.validate_in_newhire,
        ]
