import uuid

import waffle
from django.utils.functional import cached_property
from rest_framework.throttling import AnonRateThrottle, ScopedRateThrottle

from intranet.femida.src.core.choices import DRF_THROTTLE_SCOPES
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.offers.choices import OFFER_STATUSES, FORM_TYPES
from intranet.femida.src.offers.controllers import PreprofileCtl
from intranet.femida.src.offers.models import Offer, Preprofile


class ExternalFormRatePerPathThrottle(AnonRateThrottle):
    """
    Ограничивает кол-во запросов по конкретному URL'у для всех анонимов.
    Т.е. задаётся допустимое кол-во запросов для одной ссылки в целом,
    независимо от того, сколько пользователей
    и с каких ip адресов попробуют обратиться к ручке.
    """
    rate = '100/day'

    def get_ident(self, request):
        if waffle.switch_is_active('disable_api_throttling'):
            return uuid.uuid4()
        return f'{request.path}|{self.rate}'


class ExternalFormRatePerIPAndViewThrottle(ScopedRateThrottle):
    """
    Ограничивает кол-во запросов в каждую конкретную вьюху для отдельного пользователя,
    независимо от того, приходит он в разные ручки или нет.
    """
    THROTTLE_RATES = {
        DRF_THROTTLE_SCOPES.ext_form_check_login: '100/day',
        DRF_THROTTLE_SCOPES.ext_form_attachment_upload: '50/day',
        # вьюха используется в ручке формы и в ручке сабмита,
        # поэтому допускаем x2 запросов
        DRF_THROTTLE_SCOPES.ext_form_accept: '100/day',
    }


def throttle_per_path(rate='100/day'):
    klass = ExternalFormRatePerPathThrottle
    return type(klass.__name__, (klass,), {'rate': rate})


class ThrottleViewMixin:
    throttle_classes = [
        ExternalFormRatePerPathThrottle,
        ExternalFormRatePerIPAndViewThrottle,
    ]


class OfferByUidViewMixin(ThrottleViewMixin):

    @cached_property
    def offer(self):
        queryset = Offer.unsafe.filter(
            status__in=[
                OFFER_STATUSES.sent,
                OFFER_STATUSES.accepted,
            ],
        )
        return get_object_or_40x(queryset, link__uid=self.kwargs['uid'])

    def get_language(self):
        lang = self.request.GET.get('lang', '').lower()
        if lang in ('ru', 'en'):
            return lang
        translations = {
            FORM_TYPES.russian: 'ru',
            FORM_TYPES.international: 'en',
        }
        return translations[self.offer.form_type]


class PreprofileByUidViewMixin(ThrottleViewMixin):

    @cached_property
    def preprofile(self):
        return get_object_or_40x(Preprofile, link__uid=self.kwargs['uid'])

    @cached_property
    def ctl(self):
        return PreprofileCtl(self.preprofile)

    def get_language(self):
        lang = self.request.GET.get('lang', 'ru').lower()
        return lang if lang in ('ru', 'en') else 'ru'
