import random
import uuid

from collections import defaultdict
from itertools import chain

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.postgres.fields import JSONField
from django.core.validators import MinValueValidator
from django.utils.functional import cached_property
from django.db import models
from model_utils.models import TimeStampedModel

from intranet.femida.src.core.db.fields import StartrekIssueKeyField
from intranet.femida.src.core.models import I18NNameModelMixin
from intranet.femida.src.offers.helpers import get_offer_contract_end
from intranet.femida.src.offers.managers import OfferManager, PositionManager
from intranet.femida.src.permissions.managers.offer import OfferPermManager
from intranet.femida.src.staff.helpers import get_department_chief, get_hr_analysts
from intranet.femida.src.vacancies.choices import VACANCY_TYPES
from intranet.femida.src.vacancies.startrek.memberships import IssueMembership

from . import choices
from .startrek.choices import REJECTION_REASONS


User = get_user_model()


class AttachmentShortcutMixin:
    """
    Миксин, который добавляет шорткаты к разным типам аттачей
    для офферов и препрофайлов.
    """
    attachments_related_manager_name = None

    @cached_property
    def attachments_by_type(self):
        attachments_manager = getattr(self, self.attachments_related_manager_name)
        result = defaultdict(list)
        if self.attachments_related_manager_name in getattr(self, '_prefetched_objects_cache', {}):
            active_attachments = [a for a in attachments_manager.all() if a.is_active]
        else:
            active_attachments = (
                attachments_manager
                .filter(is_active=True)
                .select_related('attachment')
            )

        for att in active_attachments:
            result[att.type].append(att.attachment)
        return result

    @property
    def photos(self):
        return self.attachments_by_type[choices.OFFER_ATTACHMENT_TYPES.photo]

    @property
    def photo(self):
        return self.photos[0] if self.photos else None

    @property
    def passport_pages(self):
        return self.attachments_by_type[choices.OFFER_ATTACHMENT_TYPES.passport_page]

    @property
    def snilses(self):
        return self.attachments_by_type[choices.OFFER_ATTACHMENT_TYPES.snils]

    @property
    def snils(self):
        return self.snilses[0] if self.snilses else None

    @property
    def documents(self):
        return self.attachments_by_type[choices.OFFER_ATTACHMENT_TYPES.document]

    @property
    def offer_pdf(self):
        offer_pdf_list = self.attachments_by_type[choices.OFFER_ATTACHMENT_TYPES.offer_pdf]
        return offer_pdf_list[0] if offer_pdf_list else None

    @property
    def all_documents_chain(self):
        return chain(
            self.passport_pages,
            self.snilses,
            self.documents,
        )


class BaseModel(models.Model):

    version = models.PositiveIntegerField(
        default=0,
        editable=False,
    )

    # TODO: переименовать в created_by
    creator = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        related_name='+',
        null=True,
        on_delete=models.SET_NULL,
    )

    # TODO: переименовать в modified_by
    modifier = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        editable=False,
        related_name='+',
        null=True,
        on_delete=models.SET_NULL,
    )

    class Meta:
        abstract = True


class Offer(AttachmentShortcutMixin, BaseModel, TimeStampedModel):
    """
    Основные данные по офферу
    """
    attachments_related_manager_name = 'offer_attachments'
    unsafe = OfferManager()
    objects = OfferPermManager()

    # Поля, доступные только для внешних офферов
    EXTERNAL_ONLY_FIELDS = {
        # Поля с основной формы
        'form_type',
        'probation_period',
        'probation_period_unit',
        'probation_period_type',
        'sick_leave_supplement',
        'housing_program',
        'cellular_compensation',
        'internet_compensation_amount',
        'signup_bonus',
        'signup_2year_bonus',
        'bonus_type',
        'bonus',
        'bonus_2year',
        # Поля с формы отправки на согласование
        'other_payments',
        'salary_expectations',
        'salary_expectations_currency',
        'current_company',
        'source',
        'source_description',
        # Ещё поля, которые не нужны на внутреннем оффере
        'startrek_hr_key',
    }

    newhire_id = models.PositiveIntegerField(
        null=True,
        blank=True,
        editable=False,
    )
    newhire_status = models.CharField(
        max_length=20,
        choices=choices.OFFER_NEWHIRE_STATUSES,
        null=True,
        blank=True,
    )

    status = models.CharField(
        max_length=20,
        choices=choices.OFFER_STATUSES,
        default=choices.OFFER_STATUSES.draft,
    )
    application = models.ForeignKey(
        to='interviews.Application',
        related_name='offers',
        on_delete=models.PROTECT,
    )
    vacancy = models.ForeignKey(
        to='vacancies.Vacancy',
        related_name='offers',
        on_delete=models.PROTECT,
    )
    candidate = models.ForeignKey(
        to='candidates.Candidate',
        related_name='offers',
        on_delete=models.PROTECT,
    )
    closed_at = models.DateTimeField(blank=True, null=True)

    full_name = models.CharField(max_length=255)

    position = models.ForeignKey(
        to='offers.Position',
        null=True,
        on_delete=models.SET_NULL,
    )
    staff_position_name = models.CharField(max_length=255)

    grade = models.PositiveSmallIntegerField(null=True)
    need_help_in_adaptation = models.BooleanField(default=False)

    # Данные по бюджетной позиции
    org = models.ForeignKey(
        to='staff.Organization',
        null=True,
        on_delete=models.SET_NULL,
    )
    office = models.ForeignKey(
        to='staff.Office',
        blank=True,
        null=True,
        on_delete=models.SET_NULL,
    )
    department = models.ForeignKey(
        to='staff.Department',
        on_delete=models.PROTECT,
        related_name='offers',
        null=True,
        blank=False,
    )
    geography = models.ForeignKey(
        to='staff.Geography',
        on_delete=models.SET_NULL,
        related_name='offers',
        null=True,
        blank=True,
    )

    # TODO: Удалить, когда решим, что точно оставляем
    # руководителя подразделения
    boss_old = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        related_name='employee_offers',
        null=True,
        db_column='boss_id',
    )

    join_at = models.DateField(
        null=True,
        blank=False,
    )
    payment_type = models.CharField(
        max_length=16,
        choices=choices.PAYMENT_TYPES,
        blank=True,
    )
    payment_currency = models.ForeignKey(
        to='core.Currency',
        on_delete=models.PROTECT,
        related_name='+',
        blank=True,
        null=True,
    )

    # TODO: есть ли смысл хранить поля для каждого типа оплаты?
    salary = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        validators=[MinValueValidator(0)],
        blank=True,
        null=True,
    )
    hourly_rate = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        validators=[MinValueValidator(0)],
        blank=True,
        null=True,
    )

    # Форма занятости
    employment_type = models.CharField(
        max_length=16,
        choices=choices.EMPLOYMENT_TYPES,
        blank=True,
    )
    work_hours_weekly = models.IntegerField(null=True, blank=True)
    is_main_work_place = models.BooleanField(default=True)

    # офис/надомник
    work_place = models.CharField(
        max_length=16,
        choices=choices.WORK_PLACES,
        blank=True,
    )
    homeworker_location = models.CharField(
        max_length=255,
        blank=True,
    )

    is_resident = models.BooleanField(
        default=True,
        verbose_name='Резидент (не используется FEMIDA-5420)',
    )
    contract_type = models.CharField(
        max_length=16,
        choices=choices.CONTRACT_TYPES,
        blank=True,
    )
    # Продолжительность срочного договора (месяцы)
    contract_term = models.IntegerField(null=True, blank=True)
    # Дата окончания срочного договора
    contract_term_date = models.DateField(null=True, blank=True)

    @property
    def contract_end(self):
        return get_offer_contract_end(self)

    probation_period = models.PositiveIntegerField(default=3)
    probation_period_unit = models.CharField(
        max_length=8,
        choices=choices.PROBATION_PERIOD_UNITS,
        default=choices.PROBATION_PERIOD_UNITS.month,
    )

    # Benefits
    vmi = models.NullBooleanField()
    # доплата больничного
    sick_leave_supplement = models.BooleanField(default=False)
    housing_program = models.BooleanField(default=False)
    cellular_compensation = models.BooleanField(default=False)
    internet_compensation_amount = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
    )
    rsu = models.IntegerField(default=0, verbose_name='RSU (не используется FEMIDA-3395)')
    rsu_cost = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
        null=True,
    )
    signup_bonus = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
    )
    signup_2year_bonus = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
        null=True,
        help_text='Сайнап второго года',
    )

    bonus_type = models.CharField(
        max_length=16,
        choices=choices.BONUS_TYPES,
        blank=True,
        null=True,
    )
    bonus = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
    )
    bonus_2year = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
        null=True,
        help_text='Бонус второго года',
    )

    allowance = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
    )
    need_relocation = models.BooleanField(default=False)

    relocation_package = models.CharField(
        max_length=16,
        choices=choices.RELOCATION_PACKAGE,
        null=True,
        blank=True,
    )

    employee_type = models.CharField(
        max_length=16,
        choices=choices.EMPLOYEE_TYPES,
        default=choices.EMPLOYEE_TYPES.new,
        null=True,
    )
    username = models.CharField(max_length=50, blank=True)
    is_rotation_within_yandex = models.NullBooleanField(help_text=(
        'Заполняется в момент, когда ротация полностью согласована (принятие внутр.оффера)'
    ))

    eds_phone = models.CharField(
        max_length=50,
        null=True,
        blank=True,
    )
    is_eds_phone_verified = models.NullBooleanField(default=False)

    @property
    def signup_bonus_gross(self):
        if self.bonus_type == choices.BONUS_TYPES.signup:
            return self.bonus
        return 0.00

    @property
    def signup_2year_bonus_gross(self):
        if self.bonus_type == choices.BONUS_TYPES.signup:
            return self.bonus_2year
        return 0.00

    @property
    def welcome_bonus_gross(self):
        if self.bonus_type == choices.BONUS_TYPES.welcome:
            return self.bonus
        return 0.00

    @property
    def welcome_2year_bonus_gross(self):
        if self.bonus_type == choices.BONUS_TYPES.welcome:
            return self.bonus_2year
        return 0.00

    @property
    def total_start_bonus(self):
        return self.signup_bonus + self.allowance

    @property
    def is_internship(self):
        return self.vacancy.type == VACANCY_TYPES.internship

    @property
    def is_autohire(self):
        return self.vacancy.type == VACANCY_TYPES.autohire

    @cached_property
    def boss(self):
        return get_department_chief(self.department)

    # Поля для тикета в Стартреке

    abc_services = models.ManyToManyField(
        to='staff.Service',
        related_name='offers',
        blank=True,
    )

    # не проверяем здесь на валидность,
    # потому что в стартреке это поле тоже string
    other_payments = models.CharField(
        max_length=255,
        blank=True,
    )

    profession = models.ForeignKey(
        to='professions.Profession',
        on_delete=models.PROTECT,
        null=True,
    )
    professional_level = models.CharField(
        max_length=16,
        choices=choices.PROFESSIONAL_LEVELS,
        null=True,
    )

    programming_language = models.CharField(
        max_length=20,
        choices=choices.ST_PROGRAMMING_LANGUAGES,
        blank=True,
    )

    salary_expectations = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        validators=[MinValueValidator(0)],
        null=True,
        blank=True,
    )
    salary_expectations_currency = models.ForeignKey(
        to='core.Currency',
        on_delete=models.PROTECT,
        related_name='+',
        null=True,
        blank=True,
    )

    # CandidateJob
    current_company = models.CharField(
        max_length=255,
        blank=True,
    )

    # источник, из которого кандидат узнал о вакансии
    # т.к. пока этого нигде нет, используем значения из Стартрека
    source = models.CharField(
        max_length=32,
        choices=choices.SOURCES,
        default=choices.SOURCES.other,
    )
    source_description = models.CharField(
        max_length=255,
        blank=True,
    )

    form_type = models.CharField(
        max_length=16,
        choices=choices.FORM_TYPES,
        default=choices.FORM_TYPES.russian,
    )

    # Тикеты в СТ
    startrek_salary_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_hr_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_relocation_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_signup_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_bonus_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_adaptation_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_hdrfs_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_bootcamp_key = StartrekIssueKeyField(null=True, blank=True)
    startrek_eds_key = StartrekIssueKeyField(null=True, blank=True)

    @property
    def startrek_job_key(self):
        return self.vacancy.startrek_key

    attachments = models.ManyToManyField(
        to='attachments.Attachment',
        through='offers.OfferAttachment',
        related_name='offers',
    )
    docs_processing_status = models.CharField(
        max_length=20,
        choices=choices.OFFER_DOCS_PROCESSING_STATUSES,
        null=True,
        blank=True,
    )
    docs_processing_resolution = models.CharField(
        max_length=32,
        choices=choices.OFFER_DOCS_PROCESSING_RESOLUTIONS,
        null=True,
        blank=True,
    )
    passport_data = JSONField(null=True, blank=True)
    snils_number = models.CharField(
        max_length=14,
        null=True,
        blank=True,
    )
    residence_address_data = JSONField(null=True, blank=True)
    registration_address_data = JSONField(null=True, blank=True)
    docs_request_count = models.IntegerField(
        default=0,
        null=True,
    )

    bp_transaction_id = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='ID транзакции в реестре БП',
    )
    oebs_person_id = models.IntegerField(null=True, blank=True)

    hardware_profile_type = models.CharField(
        max_length=32,
        choices=choices.HARDWARE_PROFILE_TYPES,
        null=True,
        blank=True,
        help_text='Тип профиля оборудования',
    )
    #  Поля, используемые только в автонайме
    # ---------------------------------------------
    is_confirmed_by_boss = models.NullBooleanField()
    is_internal_phone_needed = models.NullBooleanField()
    is_sip_redirect_needed = models.NullBooleanField()
    # ---------------------------------------------

    @property
    def is_bank_details_needed(self):
        return self.org.is_russian

    @property
    def is_internal(self):
        return self.employee_type in choices.INTERNAL_EMPLOYEE_TYPES

    @property
    def is_external(self):
        return self.employee_type in choices.EXTERNAL_EMPLOYEE_TYPES

    @cached_property
    def employee(self):
        return User.objects.filter(username=self.username).first()

    def __str__(self):
        return 'Offer %s' % self.pk

    class Meta:
        default_manager_name = 'unsafe'
        verbose_name = 'оффер'
        verbose_name_plural = 'офферы'


class InternalOffer(Offer):
    """
    Прокси-модель для внутренних офферов
    """
    @cached_property
    def _current_issue_membership(self):
        return IssueMembership(self.employee.department)

    @cached_property
    def _future_issue_membership(self):
        return IssueMembership.from_offer(self)

    @cached_property
    def current_boss(self):
        return self._current_issue_membership.head

    @cached_property
    def future_boss(self):
        return self._future_issue_membership.head

    @cached_property
    def current_bosses(self):
        department_bosses = self._current_issue_membership.chiefs
        return [u for u in department_bosses if u.username != self.username]

    @cached_property
    def future_bosses(self):
        return self._future_issue_membership.chiefs

    @cached_property
    def all_bosses(self):
        return list(set(chain(self.current_bosses, self.future_bosses)))

    @cached_property
    def current_hr_analyst(self):
        hr_analysts = get_hr_analysts(self.employee.department)
        return random.choice(hr_analysts) if hr_analysts else None

    @cached_property
    def future_hr_analyst(self):
        hr_analysts = get_hr_analysts(self.department)
        return random.choice(hr_analysts) if hr_analysts else None

    @cached_property
    def current_hr_partners(self):
        return self._current_issue_membership.hr_partners

    @cached_property
    def future_hr_partners(self):
        return self._future_issue_membership.hr_partners

    @cached_property
    def all_hr_partners(self):
        return list(set(chain(self.current_hr_partners, self.future_hr_partners)))

    class Meta:
        proxy = True


class OfferRejection(BaseModel, TimeStampedModel):
    """
    Отказ от оффера
    """
    offer = models.OneToOneField(
        to=Offer,
        on_delete=models.CASCADE,
        related_name='rejection',
    )
    rejection_side = models.CharField(
        choices=choices.REJECTION_SIDES,
        default=choices.REJECTION_SIDES.candidate,
        max_length=16,
        blank=True,
        null=True,
    )
    rejection_reason = models.CharField(choices=REJECTION_REASONS, max_length=16, blank=True)
    competing_offer_conditions = models.TextField(blank=True)
    competing_company = models.CharField(max_length=255, blank=True)
    comment = models.TextField(blank=True)
    bp_transaction_id = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='ID транзакции в реестре БП',
    )

    def __str__(self):
        return 'Offer %s rejection' % self.offer_id

    class Meta:
        verbose_name = 'отказ от оффера'
        verbose_name_plural = 'отказы от оффера'


class OfferProfileEducationMixin(models.Model):
    """
    FEMIDA-5528: Миксин с полями про образование,
    которые мы раньше спрашивали на внешней анкете, но теперь перестали.
    Пока здесь хранятся исторические данные. Возможно, когда-нибудь их выкинем
    """
    education = models.CharField(
        max_length=16,
        default='',
        choices=choices.EDU_STATUSES,
        blank=True,
    )
    education_direction = models.CharField(
        max_length=16,
        default='',
        choices=choices.EDU_DIRECTIONS,
        blank=True,
    )
    educational_institution = models.CharField(
        max_length=150,
        null=True,
        blank=True,
    )
    education_end_date = models.DateField(
        null=True,
        blank=True,
    )

    class Meta:
        abstract = True


class OfferProfile(OfferProfileEducationMixin, BaseModel, TimeStampedModel):
    """
    Анкета, заполняемая кандидатом
    """
    offer = models.OneToOneField(
        to=Offer,
        on_delete=models.CASCADE,
        related_name='profile',
    )

    # Фамилия
    last_name = models.CharField(
        max_length=100,
        null=True,
        blank=False,
    )

    # Имя
    first_name = models.CharField(
        max_length=50,
        null=True,
        blank=False,
    )

    # Отчество
    middle_name = models.CharField(
        max_length=50,
        null=True,
        blank=True,
    )

    # Фамилия (лат.буквы)
    last_name_en = models.CharField(
        max_length=100,
        null=True,
        blank=False,
    )

    # Имя (лат.буквы)
    first_name_en = models.CharField(
        max_length=50,
        null=True,
        blank=False,
    )

    @property
    def full_name(self):
        fio = [p for p in (self.last_name, self.first_name, self.middle_name) if p]
        return ' '.join(fio)

    # Пол
    gender = models.CharField(
        blank=False,
        max_length=1,
        choices=choices.GENDER,
    )

    # Дата рождения (год, месяц, день)
    birthday = models.DateField(
        null=True,
        blank=False,
    )

    # Гражданство
    citizenship = models.CharField(
        default='',
        choices=choices.CITIZENSHIP,
        max_length=2,
        null=True,
        blank=True,
    )

    # Трудовая книжка
    employment_book = models.CharField(
        default='',
        choices=choices.EMPLOYMENT_BOOK_OPTIONS,
        max_length=16,
        null=True,
        blank=True,
    )

    # Очная ли форма обучения
    is_full_time_education = models.NullBooleanField()

    # Наличие подтверждения очной формы обучения ко дню выхода
    has_education_confirmation = models.NullBooleanField()

    # Адрес (проживания)
    residence_address = models.CharField(
        max_length=1024,
        null=True,
        blank=False,
    )

    # Контактный телефон
    phone = models.CharField(
        max_length=50,
        null=True,
        blank=False,
    )

    # TODO: удалить, не актуально, см. FEMIDA-7027
    # ID card (для иностранцев)
    id_card = models.CharField(
        max_length=30,
        null=True,
        blank=True,
    )

    # ОС рабочей станции
    os = models.CharField(
        max_length=16,
        choices=choices.OPERATING_SYSTEMS,
        null=True,
        blank=False,
    )

    # E-mail (домашний)
    home_email = models.CharField(
        max_length=50,
        null=True,
        blank=True,
    )

    nda_accepted = models.BooleanField(default=False)

    main_language = models.CharField(
        max_length=2,
        null=True,
    )

    spoken_languages = JSONField(
        null=True,
    )

    preferred_first_and_last_name = models.CharField(
        max_length=255,
        null=True,
    )

    def __str__(self):
        return 'Анкета оффера #{offer_id}: {fio}'.format(
            offer_id=self.offer_id,
            fio=self.full_name
        )

    class Meta:
        verbose_name = 'анкета'
        verbose_name_plural = 'анкеты'


class OfferProfileComment(TimeStampedModel):

    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        editable=False,
        related_name='+',
        null=True,
        on_delete=models.SET_NULL,
    )
    offer_profile = models.ForeignKey(
        to='OfferProfile',
        editable=False,
        related_name='comments',
        on_delete=models.CASCADE,
    )

    text = models.TextField()
    is_hr = models.BooleanField(default=False)

    def __str__(self):
        return self.text

    class Meta:
        ordering = 'created',


class Link(TimeStampedModel):
    """
    Ссылка ведущая на внешнюю анкету.
    Может быть привязана как к офферу Фемиды,
    так и к препрофайлу Наниматора
    """
    offer = models.OneToOneField(
        to=Offer,
        on_delete=models.CASCADE,
        editable=False,
        null=True,
    )
    preprofile = models.OneToOneField(
        to='offers.Preprofile',
        on_delete=models.CASCADE,
        editable=False,
        null=True,
    )
    uid = models.UUIDField(
        default=uuid.uuid4,
        editable=False,
    )
    expiration_time = models.DateTimeField(null=True)
    version = models.PositiveIntegerField(
        default=0,
        editable=False,
    )

    def __str__(self):
        return self.uid.hex


class OfferAttachment(TimeStampedModel):

    unsafe = models.Manager()
    objects = OfferPermManager(perm_prefix='offer')

    offer = models.ForeignKey(
        to=Offer,
        on_delete=models.PROTECT,
        related_name='offer_attachments',
        null=True,
    )
    attachment = models.ForeignKey(
        to='attachments.Attachment',
        on_delete=models.PROTECT,
        related_name='offer_attachments',
    )
    type = models.CharField(
        max_length=16,
        choices=choices.OFFER_ATTACHMENT_TYPES,
        default=choices.OFFER_ATTACHMENT_TYPES.document,
    )
    is_active = models.BooleanField(default=True)

    # TODO: удалить после релиза FEMIDA-3368
    offer_profile = models.ForeignKey(
        to=OfferProfile,
        on_delete=models.PROTECT,
        related_name='offer_profile_attachments',
        null=True,
    )

    def __str__(self):
        return 'Offer attachment ({}, {})'.format(self.offer_id, self.attachment_id)

    class Meta:
        # TODO: переименовать в offers_offerattachment
        db_table = 'offers_offerprofileattachment'
        default_manager_name = 'unsafe'


class Position(I18NNameModelMixin, models.Model):
    """
    Справочник должностей, синхронизированный с OEBS
    """
    objects = PositionManager()

    staff_id = models.IntegerField(null=True, blank=True)
    name_ru = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255, blank=True)
    is_deleted = models.BooleanField(default=False)

    def __str__(self):
        return self.name_ru

    class Meta:
        verbose_name = 'должность'
        verbose_name_plural = 'должности'


class RawTemplate(TimeStampedModel):
    """
    Шаблон для формирования текста оффера,
    который отправляется кандидату.
    Хранится в виде django-шаблона.
    """
    name = models.CharField(max_length=255)
    text = models.TextField()

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = 'шаблон для оффера'
        verbose_name_plural = 'шаблоны для оффера'


class OfferTemplate(TimeStampedModel):
    """
    Шаблон оффера, уже отредактированный рекрутером.
    """
    offer = models.ForeignKey(
        to=Offer,
        on_delete=models.CASCADE,
        related_name='templates',
    )
    raw_template = models.ForeignKey(
        to=RawTemplate,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )
    text = models.TextField()
    is_active = models.BooleanField(default=True)

    def __str__(self):
        return 'Prepared Template: %s, %s' % (self.offer_id, self.raw_template_id)

    class Meta:
        verbose_name = 'текст оффера'
        verbose_name_plural = 'тексты офферов'


class Brochure(TimeStampedModel):
    """
    Брошюра для оффера
    """
    name = models.CharField(max_length=255)
    attachment = models.ForeignKey(
        to='attachments.Attachment',
        on_delete=models.PROTECT,
        related_name='brochure',
    )
    is_active = models.BooleanField(default=True)

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = 'брошюра для оффера'
        verbose_name_plural = 'брошюры для оффера'


class Preprofile(AttachmentShortcutMixin, TimeStampedModel):
    """
    Препрофайл в Наниматоре.
    Модель аналогичная Офферу, но с очень ограниченным составом полей.
    Препрофайл, в отличие от Оффера, не основан на каком-то прет-ве или даже кандидате.
    Он не согласовывается аналитиками привычным образом и, в целом,
    требует меньшего взаимодействия участников процесса найма.
    Препрофайл сам по себе живёт в Наниматоре, в Фемиде хранятся
    лишь связь с ним, данные полученные через внешнюю анкету
    и несколько системных полей.
    Используется в тех случаях, когда стандартный процесс найма
    (через собеседование, выставление оффера и т.д.) не применим.
    В частности, используется для вывода сотрудников в Outstaff.
    """
    attachments_related_manager_name = 'preprofile_attachments'

    id = models.PositiveIntegerField(primary_key=True)
    data = JSONField(null=True, blank=True)
    is_saved = models.BooleanField(default=False)
    startrek_hr_key = StartrekIssueKeyField(blank=True)
    startrek_eds_key = StartrekIssueKeyField(blank=True, null=True)
    attachments = models.ManyToManyField(
        to='attachments.Attachment',
        through='offers.PreprofileAttachment',
        related_name='preprofiles',
    )

    eds_phone = models.CharField(
        max_length=50,
        null=True,
        blank=True,
    )
    is_eds_phone_verified = models.NullBooleanField(default=False)

    @property
    def newhire_status(self):
        return self.data.get('newhire_status')

    def __str__(self):
        return 'Newhire Preprofile: %s' % self.id

    class Meta:
        verbose_name = 'препрофайл в Наниматоре'
        verbose_name_plural = 'препрофайлы в Наниматоре'


class PreprofileAttachment(TimeStampedModel):

    preprofile = models.ForeignKey(
        to=Preprofile,
        on_delete=models.PROTECT,
        related_name='preprofile_attachments',
        null=True,
    )
    attachment = models.ForeignKey(
        to='attachments.Attachment',
        on_delete=models.PROTECT,
        related_name='preprofile_attachments',
    )
    type = models.CharField(
        max_length=16,
        choices=choices.OFFER_ATTACHMENT_TYPES,
        default=choices.OFFER_ATTACHMENT_TYPES.document,
    )
    is_active = models.BooleanField(default=True)

    def __str__(self):
        return 'Preprofile attachment ({}, {})'.format(self.preprofile_id, self.attachment_id)


class OfferSchemesData(TimeStampedModel):
    """
    Схемы посчитанные стаффом и их данные
    """
    offer = models.OneToOneField(
        to=Offer,
        on_delete=models.CASCADE,
        editable=False,
        related_name='schemes_data',
    )
    review_scheme_id = models.IntegerField(
        null=True,
        blank=True,
        help_text='ID схемы ревью',
    )
    bonus_scheme_id = models.IntegerField(
        null=True,
        blank=True,
        help_text='ID схемы бонусов',
    )
    reward_scheme_id = models.IntegerField(
        null=True,
        blank=True,
        help_text='ID схемы компенсаций',
    )
    review_scheme_name = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='Схема ревью',
    )
    bonus_scheme_name = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='Схема премирования',
    )
    reward_scheme_name = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='Схема компенсаций',
    )
    has_health_insurance = models.BooleanField(
        default=False,
        help_text='ДМС',
    )
    has_life_insurance = models.BooleanField(
        default=False,
        help_text='Страхование несчастных случаев (НС)',
    )
    has_food_compensation = models.BooleanField(
        default=False,
        help_text='Компенсация питания',
    )
    food_scheme = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='Схема компенсации питания',
    )
    has_review = models.BooleanField(
        default=False,
        help_text='Ревью',
    )
    review_bonus = models.FloatField(
        null=True,
        blank=True,
        help_text='Премия по результам ревью',
    )
    non_review_bonus = models.FloatField(
        null=True,
        blank=True,
        help_text='Премия вне ревью',
    )
    is_reward_category_changed = models.BooleanField(
        default=False,
        help_text='Меняется категория',
    )
    reward_category = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='Новая категория',
    )
    current_reward_category = models.CharField(
        max_length=64,
        null=True,
        blank=True,
        help_text='Текущая категория',
    )

    class Meta:
        verbose_name = 'схема ревью, премирования и компенсации оффера'
        verbose_name_plural = 'схемы ревью, премирования и компенсаций'


class RelocationPackage(models.Model):
    """
    Пакеты релокации: silver, gold, platinum
    """
    type = models.CharField(
        unique=True,
        max_length=16,
        choices=choices.RELOCATION_PACKAGE,
    )
    base_allowance = models.DecimalField(
        max_digits=10,
        decimal_places=2,
        default=0,
        validators=[MinValueValidator(0)],
    )
    min_grade = models.IntegerField()
    max_grade = models.IntegerField()
