from rest_framework import serializers

from intranet.femida.src.core.serializers import MapperField
from intranet.femida.src.offers import choices as offer_choices
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.staff.helpers import get_department_chief
from intranet.femida.src.vacancies.choices import VACANCY_TYPES


CITIZENSHIP_MAP = {
    offer_choices.CITIZENSHIP.RU: 'RUSSIAN',
    offer_choices.CITIZENSHIP.BY: 'BELORUSSIAN',
    offer_choices.CITIZENSHIP.UA: 'UKRAINIAN',
    offer_choices.CITIZENSHIP.KZ: 'KAZAKHSTAN',
    offer_choices.CITIZENSHIP.KG: 'KYRGYZSTAN',
    offer_choices.CITIZENSHIP.AM: 'ARMENIA',
    offer_choices.CITIZENSHIP.ZZ: 'OTHER',
}

NEWHIRE_STATUSES_REVERSED_MAP = {
    'NEW': offer_choices.OFFER_NEWHIRE_STATUSES.new,
    'PREPARED': offer_choices.OFFER_NEWHIRE_STATUSES.prepared,
    'APPROVED': offer_choices.OFFER_NEWHIRE_STATUSES.approved,
    'READY': offer_choices.OFFER_NEWHIRE_STATUSES.ready,
    'CLOSED': offer_choices.OFFER_NEWHIRE_STATUSES.closed,
    'CANCELLED': offer_choices.OFFER_NEWHIRE_STATUSES.cancelled,
}

EMPLOYEE_TYPES_MAP = {
    offer_choices.EMPLOYEE_TYPES.new: 'NEW_EMPLOYEE',
    offer_choices.EMPLOYEE_TYPES.former: 'FORMER_EMPLOYEE',
    offer_choices.EMPLOYEE_TYPES.current: 'EXTERNAL_EMPLOYEE',
    offer_choices.EMPLOYEE_TYPES.rotation: 'CURRENT_EMPLOYEE',
    offer_choices.EMPLOYEE_TYPES.intern: 'CURRENT_EMPLOYEE',
}
EMPLOYEE_TYPES_REVERSED_MAP = {v: k for k, v in EMPLOYEE_TYPES_MAP.items()}
EMPLOYEE_TYPES_REVERSED_MAP['CURRENT_EMPLOYEE'] = offer_choices.EMPLOYEE_TYPES.rotation

OPERATING_SYSTEMS_MAP = {
    offer_choices.OPERATING_SYSTEMS.ubuntu: 'UBUNTU',
    offer_choices.OPERATING_SYSTEMS.kubuntu: 'KUBUNTU',
    offer_choices.OPERATING_SYSTEMS.windows: 'WINDOWS',
    offer_choices.OPERATING_SYSTEMS.mac: 'MAC',
    offer_choices.OPERATING_SYSTEMS.none: 'NO_OS',
}

LANGUAGES_MAP = {
    offer_choices.FORM_TYPES.russian: 'ru',
    offer_choices.FORM_TYPES.international: 'en',
}


class ProfileReadonlyField(serializers.ReadOnlyField):

    def bind(self, field_name, parent):
        if self.source is None:
            self.source = 'profile.%s' % field_name
        super().bind(field_name, parent)


class ProfileMapperField(MapperField, ProfileReadonlyField):

    pass


class PreprofileDataReadonlyField(serializers.ReadOnlyField):

    def bind(self, field_name, parent):
        if self.source is None:
            self.source = 'data.%s' % field_name
        super().bind(field_name, parent)


class NewhireOfferBaseSerializer(serializers.Serializer):
    """
    Базовый сериализатор оффера для отправки в Наниматор
    """
    femida_offer_id = serializers.ReadOnlyField(source='id')
    recruiter = serializers.SerializerMethodField()

    login = serializers.ReadOnlyField(source='username')
    join_at = serializers.DateField()
    candidate_type = MapperField(source='employee_type', mapping=EMPLOYEE_TYPES_MAP)
    employment_type = serializers.SerializerMethodField()
    department = serializers.ReadOnlyField(source='department.url')
    office = serializers.ReadOnlyField(source='office_id')
    organization = serializers.ReadOnlyField(source='org_id')
    position = serializers.ReadOnlyField(source='position.name_ru')
    position_staff_text = serializers.ReadOnlyField(source='staff_position_name')
    abc_services = serializers.SerializerMethodField()
    date_completion_internship = serializers.SerializerMethodField()
    profession = serializers.ReadOnlyField(source='vacancy.profession_id')
    professional_sphere = serializers.ReadOnlyField(source='vacancy.professional_sphere_id')
    is_autohire = serializers.SerializerMethodField()
    hardware_profile_type = serializers.ReadOnlyField()
    need_internal_phone = serializers.ReadOnlyField(source='is_internal_phone_needed')
    need_sip_redirect = serializers.ReadOnlyField(source='is_sip_redirect_needed')

    def get_recruiter(self, obj):
        """
        Создатель оффера либо его руководитель, если создатель уволился
        """
        recruiter = obj.creator
        if recruiter.is_dismissed:
            recruiter = get_department_chief(recruiter.department)
        return recruiter.username if recruiter else None

    def get_employment_type(self, obj):
        if not obj.is_main_work_place:
            return 'D'
        elif obj.employment_type == offer_choices.EMPLOYMENT_TYPES.full:
            return 'F'
        elif obj.employment_type == offer_choices.EMPLOYMENT_TYPES.partial:
            return 'P'

    def get_abc_services(self, obj):
        return [s.group_id for s in obj.abc_services.filter(is_deleted=False)]

    def get_date_completion_internship(self, obj):
        if obj.vacancy.type == VACANCY_TYPES.internship:
            return obj.contract_end.strftime('%Y-%m-%d')

    def get_is_autohire(self, obj):
        return obj.is_autohire and obj.is_confirmed_by_boss

    def to_representation(self, data):
        data = super().to_representation(data)
        for key in list(data):
            if data[key] is None:
                del data[key]
        return data

    @classmethod
    def serialize(cls, instance):
        if instance.is_internal:
            return NewhireInternalOfferSerializer(instance).data
        return NewhireExternalOfferSerializer(instance).data


class NewhireInternalOfferSerializer(NewhireOfferBaseSerializer):
    """
    Сериализатор внутреннего оффера для отправки в Наниматор
    """


class NewhireExternalOfferSerializer(NewhireOfferBaseSerializer):
    """
    Сериализатор внешнего оффера для отправки в Наниматор
    """
    last_name = ProfileReadonlyField()
    first_name = ProfileReadonlyField()
    middle_name = ProfileReadonlyField()
    last_name_en = ProfileReadonlyField()
    first_name_en = ProfileReadonlyField()
    gender = ProfileReadonlyField()
    birthday = serializers.DateField(source='profile.birthday')
    photo = serializers.ReadOnlyField(source='photo.attached_file.url')

    phone = ProfileReadonlyField()
    email = serializers.ReadOnlyField(source='profile.home_email')
    address = serializers.ReadOnlyField(source='profile.residence_address')

    os = ProfileMapperField(mapping=OPERATING_SYSTEMS_MAP)
    lang_ui = MapperField(source='form_type', mapping=LANGUAGES_MAP)

    is_eds_phone_verified = serializers.ReadOnlyField()


class NewhireOfferUpdateSerializer(serializers.Serializer):

    join_at = serializers.DateField(read_only=True)
    candidate_type = MapperField(mapping=EMPLOYEE_TYPES_MAP, source='employee_type')
    login = serializers.ReadOnlyField(source='username')
    department = serializers.ReadOnlyField(source='department.url')
    date_completion_internship = serializers.DateField(source='contract_end', read_only=True)


class NewhireOfferRemoteSerializer(serializers.Serializer):

    join_at = serializers.DateField(read_only=True)
    employee_type = MapperField(
        mapping=EMPLOYEE_TYPES_REVERSED_MAP,
        source='candidate_type',
    )
    username = serializers.ReadOnlyField(source='login')
    startrek_hdrfs_key = serializers.ReadOnlyField(source='hdrfs_ticket')
    newhire_status = MapperField(
        mapping=NEWHIRE_STATUSES_REVERSED_MAP,
        source='status',
    )


class NewhireOfferStoredSerializer(serializers.ModelSerializer):

    class Meta:
        model = Offer
        fields = (
            'join_at',
            'employee_type',
            'username',
            'startrek_hdrfs_key',
            'newhire_status',
        )


class PreprofileSerializer(serializers.Serializer):

    birthday = serializers.DateField()
    join_at = serializers.DateField()

    def to_representation(self, data):
        return dict(data, **super().to_representation(data))


class NewhirePreprofileSerializer(serializers.Serializer):

    last_name = PreprofileDataReadonlyField()
    first_name = PreprofileDataReadonlyField()
    middle_name = PreprofileDataReadonlyField()
    last_name_en = PreprofileDataReadonlyField()
    first_name_en = PreprofileDataReadonlyField()
    gender = PreprofileDataReadonlyField()
    birthday = serializers.DateField(source='data.birthday')
    phone = PreprofileDataReadonlyField()
    email = serializers.ReadOnlyField(source='data.home_email')
    address = serializers.ReadOnlyField(source='data.residence_address')
    login = serializers.ReadOnlyField(source='data.username')
    join_at = serializers.DateField(source='data.join_at')
    photo = serializers.ReadOnlyField(source='photo.attached_file.url')
    citizenship = MapperField(mapping=CITIZENSHIP_MAP, source='data.citizenship')
    is_eds_phone_verified = serializers.ReadOnlyField()

    def to_representation(self, data):
        data = super().to_representation(data)
        for key in data:
            if data[key] is None:
                del data[key]
        return data


class NewhirePreprofileRemoteSerializer(NewhireOfferRemoteSerializer):

    org = serializers.ReadOnlyField(source='organization_id')
    department = serializers.ReadOnlyField(source='department_id')
    recruiter = serializers.ReadOnlyField(source='recruiter_login')
    boss = serializers.ReadOnlyField(source='chief_login')
    position = serializers.ReadOnlyField()
