import logging

from itertools import chain

from django.conf import settings
from django.core.files.base import ContentFile
from django.template import loader

from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.notifications.utils import get_base_context
from intranet.femida.src.offers.choices import FORM_TYPES, EMPLOYEE_TYPES
from intranet.femida.src.offers.helpers import (
    get_department_adaptation_issue_type,
    is_adaptation_needed,
    is_bootcamp_offer,
    is_eds_needed,
    has_signup_bonus,
)
from intranet.femida.src.offers.models import InternalOffer
from intranet.femida.src.offers.startrek.serializers import (
    AdaptationIssueContextSerializer,
    BootcampIssueCreateFieldsSerializer,
    EdsIssueFieldsSerializer,
    HRIssueContextSerializer,
    HRIssueFieldsSerializer,
    HRIssueUpdateFieldsSerializer,
    InternRelocationIssueFieldsSerializer,
    PreprofileHRIssueFieldsSerializer,
    RelocationIssueContextSerializer,
    RelocationIssueFieldsSerializer,
    SalaryIssueFieldsSerializer,
    serialize_offer_for_adaptation_issue_create,
    SignupIssueFieldsSerializer,
    WelcomeBonusIssueFieldsSerializer,
)
from intranet.femida.src.staff.helpers import get_hr_partners, get_hr_analyst_groups
from intranet.femida.src.startrek.operations import (
    IssueCommentOperation,
    IssueTransitionOperation,
    IssueUpdateOperation,
)
from intranet.femida.src.startrek.tasks import add_issue_comment_task
from intranet.femida.src.startrek.utils import (
    add_comment,
    create_issue,
    IssueTypeEnum,
    update_issue,
)
from intranet.femida.src.utils.ok import create_tracker_approvement_comment
from intranet.femida.src.utils.strings import slugify


logger = logging.getLogger(__name__)


def create_salary_issue(offer, initiator):
    assert isinstance(offer, InternalOffer)
    template_name = 'startrek/offers/create-salary-issue.wiki'
    fields = SalaryIssueFieldsSerializer(offer).data
    context = get_base_context()
    context['instance'] = offer
    context['user'] = initiator

    issue = create_issue(
        queue=settings.STARTREK_SALARY_QUEUE,
        summary='Ротация: {o.full_name}, {o.username}@'.format(o=offer),
        description=loader.render_to_string(template_name, context),
        followers=[initiator.username],
        **fields
    )
    update_instance(offer, {
        'startrek_salary_key': issue.key,
    })

    # TODO: непонятно, что делать, если HR-аналитика по какой-то причине нет
    hr_analyst = offer.current_hr_analyst
    hr_analyst_login = hr_analyst.username if hr_analyst else None

    # Запускаем согласование в созданном SALARY-тикете
    current_hr_partners = get_hr_partners(offer.employee.department)
    approvers_chain = chain(current_hr_partners, offer.current_bosses)
    context['ok_approvement'] = create_tracker_approvement_comment(
        issue_key=issue.key,
        author=hr_analyst_login,
        approvers=[i.username for i in approvers_chain],
        groups=get_hr_analyst_groups(offer.department),
    )
    comment = loader.render_to_string('startrek/offers/salary-approvement.wiki', context)
    add_issue_comment_task.delay(issue.key, comment)

    # Пишем в JOB, что начали согласовывать ротацию
    comment = loader.render_to_string(
        template_name='startrek/offers/offer-on-rotation-approval.wiki',
        context=context,
    )
    operation = IssueCommentOperation(offer.vacancy.startrek_key)
    operation.delay(comment)

    return issue


def create_hr_issue(offer, has_bank_details=True, startrek_timeout=4):
    template_name = 'startrek/offers/create-hr-issue.wiki'
    hr_partners = get_hr_partners(offer.department)

    fields = HRIssueFieldsSerializer(
        instance=offer,
        context={
            # Таскаем has_bank_details через весь код,
            # потому что не храним у себя банковские реквизиты
            'has_bank_details': has_bank_details,
        },
    ).data
    context_extra_data = HRIssueContextSerializer(offer).data

    context = get_base_context()
    context.update({
        'offer': offer,
        'hr_partners': hr_partners,
        'hr_comments': offer.profile.comments.filter(is_hr=True),
        'extra_data': context_extra_data,
    })

    followers = {offer.creator.username}
    followers |= {u.username for u in hr_partners}
    if offer.boss:
        followers.add(offer.boss.username)
    followers = list(followers)

    attachments = []
    for att in offer.all_documents_chain:
        attachments.append(ContentFile(
            content=att.attached_file.read(),
            name=slugify(att.name),
        ))

    issue = create_issue(
        queue=settings.STARTREK_HR_QUEUE,
        summary='Новый сотрудник: {full_name} (Фемида)'.format(
            full_name=offer.full_name,
        ),
        description=loader.render_to_string(template_name, context),
        author=offer.boss.username if offer.boss else None,
        timeout=startrek_timeout,
        type=IssueTypeEnum.new_employee,
        followers=followers,
        attachments=attachments,
        **fields
    )
    update_instance(offer, {
        'startrek_hr_key': issue.key,
    })
    return issue


def create_preprofile_hr_issue(preprofile, resubmit=False, has_bank_details=True):
    from intranet.femida.src.offers.controllers import RemotePreprofile
    remote_preprofile = RemotePreprofile(preprofile)
    if remote_preprofile.org.id == settings.EXTERNAL_ORGANIZATION_ID:
        return

    template_name = 'startrek/preprofiles/create-hr-issue.wiki'
    fields = PreprofileHRIssueFieldsSerializer(
        instance=remote_preprofile,
        context={
            # Таскаем has_bank_details через весь код,
            # потому что не храним у себя банковские реквизиты
            'has_bank_details': has_bank_details,
        },
    ).data

    context = get_base_context()
    context['remote_preprofile'] = remote_preprofile
    context['is_eds_needed'] = is_eds_needed(remote_preprofile)
    description = loader.render_to_string(template_name, context)

    followers = [remote_preprofile.recruiter.username]
    if remote_preprofile.boss:
        followers.append(remote_preprofile.boss.username)
    attachments = []
    for att in remote_preprofile.all_documents_chain:
        attachments.append(ContentFile(
            content=att.attached_file.read(),
            name=slugify(att.name),
        ))

    if not resubmit:
        issue = create_issue(
            queue=settings.STARTREK_HR_QUEUE,
            summary='Новый сотрудник: {full_name}'.format(
                full_name=remote_preprofile.full_name,
            ),
            description=description,
            type=IssueTypeEnum.new_employee,
            followers=followers,
            attachments=attachments,
            **fields
        )
        update_instance(preprofile, {
            'startrek_hr_key': issue.key,
        })
    else:
        update_issue(
            keys=preprofile.startrek_hr_key,
            **fields
        )
        # Если все делать через update_issue, аттачи к комменту не цепляются
        add_comment(
            keys=preprofile.startrek_hr_key,
            text=description,
            attachments=attachments,
        )


def update_preprofile_issues(preprofile):
    from intranet.femida.src.offers.controllers import RemotePreprofile
    remote_preprofile = RemotePreprofile(preprofile)
    issue_data = HRIssueUpdateFieldsSerializer(remote_preprofile).data
    fields = (
        'startrek_hr_key',
        'startrek_eds_key',
    )
    for field in fields:
        key = getattr(preprofile, field)
        if not key:
            continue
        IssueUpdateOperation(key).delay(start=issue_data.get('start'))


def create_relocation_issue(offer):
    assert offer.need_relocation
    if offer.is_internship:
        serializer_class = InternRelocationIssueFieldsSerializer
    else:
        serializer_class = RelocationIssueFieldsSerializer

    template_name = 'startrek/offers/create-relocation-issue.wiki'
    fields = serializer_class(offer).data
    context_extra_data = RelocationIssueContextSerializer(offer).data
    hr_partners = get_hr_partners(offer.department)

    context = get_base_context()
    context.update({
        'offer': offer,
        'extra_data': context_extra_data,
    })
    issue = create_issue(
        summary='{full_name} ({username}@)'.format(
            full_name=offer.full_name,
            username=offer.username,
        ),
        description=loader.render_to_string(template_name, context),
        followers=[u.username for u in hr_partners],
        access=[offer.creator.username],
        **fields
    )
    update_instance(offer, {
        'startrek_relocation_key': issue.key,
    })
    return issue


def create_signup_issue(offer):
    assert has_signup_bonus(offer)
    template_name = 'startrek/offers/create-signup-issue.wiki'
    fields = SignupIssueFieldsSerializer(offer).data
    hr_partners = get_hr_partners(offer.department)

    context = get_base_context()
    context['offer'] = offer
    issue = create_issue(
        queue=settings.STARTREK_SIGNUP_QUEUE,
        summary='{full_name} ({username}@)'.format(
            full_name=offer.full_name,
            username=offer.username,
        ),
        description=loader.render_to_string(template_name, context),
        access=[offer.creator.username] + [u.username for u in hr_partners],
        **fields
    )
    update_instance(offer, {
        'startrek_signup_key': issue.key,
    })
    return issue


def create_welcome_bonus_issue(offer):
    assert offer.welcome_bonus_gross or offer.welcome_2year_bonus_gross
    template_name = 'startrek/offers/create-bonus-issue.wiki'
    fields = WelcomeBonusIssueFieldsSerializer(offer).data
    hr_partners = get_hr_partners(offer.department)

    context = get_base_context()
    context['offer'] = offer
    issue = create_issue(
        summary='{full_name} ({username}@)'.format(
            full_name=offer.full_name,
            username=offer.username,
        ),
        description=loader.render_to_string(template_name, context),
        access=[offer.creator.username] + [u.username for u in hr_partners],
        **fields
    )
    update_instance(offer, {
        'startrek_bonus_key': issue.key,
    })
    return issue


def create_adaptation_issue(offer):
    assert is_adaptation_needed(offer)
    template_name = 'startrek/offers/create-adaptation-issue.wiki'
    fields = serialize_offer_for_adaptation_issue_create(offer)
    context_extra_data = AdaptationIssueContextSerializer(offer).data

    context = get_base_context()
    context.update({
        'offer': offer,
        'candidate': offer.candidate,
        'extra_data': context_extra_data,
        'show_main_language': offer.form_type == FORM_TYPES.international,
    })

    issue_type = (
        get_department_adaptation_issue_type(offer.department_id) or
        IssueTypeEnum.new_employee
    )

    if offer.is_external:
        prefixes = {
            EMPLOYEE_TYPES.current: '[Перевод ВнешКонс] ',
        }
        prefix = prefixes.get(offer.employee_type, '')
        summary = f'{prefix}{offer.full_name} ({offer.username}@)'
    else:
        candidate = offer.candidate
        summary = f'[Ротация] {candidate.last_name} {candidate.first_name} ({offer.username}@)'

    issue = create_issue(
        # Note: queue для тикетов адаптации является динамическим значением,
        # т.к. для стажеров очередь одна, для остальных - другая.
        # Корректный queue прилетает в fields из сериализатора.
        summary=summary,
        description=loader.render_to_string(template_name, context),
        type=issue_type,
        **fields
    )
    update_instance(offer, {
        'startrek_adaptation_key': issue.key,
    })
    return issue


def create_bootcamp_issue(offer):
    assert is_bootcamp_offer(offer)
    template_name = 'startrek/offers/create-bootcamp-issue.wiki'
    fields = BootcampIssueCreateFieldsSerializer(offer).data
    context = get_base_context()
    context['candidate'] = offer.candidate

    issue = create_issue(
        # queue прилетает в fields из сериализатора
        summary='{full_name} ({username}@)'.format(
            full_name=offer.candidate.get_full_name(),
            username=offer.username,
        ),
        description=loader.render_to_string(template_name, context),
        **fields
    )
    update_instance(offer, {
        'startrek_bootcamp_key': issue.key,
    })
    return issue


def create_eds_issue(instance, template_name, context):
    fields = EdsIssueFieldsSerializer(instance).data
    issue = create_issue(
        queue=settings.STARTREK_EDS_QUEUE,
        summary=f'Получение ЭП {instance.full_name} ({instance.username}@)',
        type=IssueTypeEnum.service_request,
        description=loader.render_to_string(template_name, context),
        **fields
    )
    return issue


def create_offer_eds_issue(offer):
    assert offer.is_eds_phone_verified

    context = get_base_context()
    context['offer'] = offer
    template_name = 'startrek/offers/create-eds-issue.wiki'

    issue = create_eds_issue(offer, template_name, context)
    update_instance(offer, {
        'startrek_eds_key': issue.key,
    })
    return issue


def create_preprofile_eds_issue(preprofile):
    from intranet.femida.src.offers.controllers import RemotePreprofile
    assert preprofile.is_eds_phone_verified

    remote_preprofile = RemotePreprofile(preprofile)
    context = get_base_context()
    context['remote_preprofile'] = remote_preprofile
    template_name = 'startrek/preprofiles/create-eds-issue.wiki'

    issue = create_eds_issue(remote_preprofile, template_name, context)
    update_instance(preprofile, {
        'startrek_eds_key': issue.key,
    })
    return issue


def execute_adaptation_issue_transition(offer, transition, **fields):
    """
    Выполняет заданный переход в тикетах адаптации.
    При ротации мы решили идти другим путём – в тикетах нет явного перехода,
    вместо него проставляется аналогичный тег
    """
    assert 'tags' not in fields, 'tags cannot be used explicitly'
    # Это просто задел на будущее, если захотят по-своему назвать теги
    transition_to_tag = {}
    key = offer.startrek_adaptation_key
    if not key:
        return
    if offer.is_rotation_within_yandex:
        tag = transition_to_tag.get(transition, transition)
        IssueUpdateOperation(key).delay(tags={'add': [tag]}, **fields)
    else:
        IssueTransitionOperation(key).delay(transition=transition, **fields)
