import random
from datetime import timedelta

import waffle
import yenv
from django.conf import settings
from rest_framework import serializers

from intranet.femida.src.api.candidates.utils import get_skype_accounts
from intranet.femida.src.core.serializers import (
    FakeField,
    MapperField,
    StartrekLocalField,
    StartrekUniqueField,
)
from intranet.femida.src.oebs.api import get_budget_position, BudgetPositionError
from intranet.femida.src.offers.choices import (
    CONTRACT_TYPES,
    SOURCES,
    WORK_PLACES,
    PAYMENT_TYPES,
    PROBATION_PERIOD_UNITS,
    EMPLOYEE_TYPES,
    EMPLOYMENT_BOOK_TRANSLATIONS,
    FORM_TYPES,
)
from intranet.femida.src.offers.helpers import (
    get_department_adaptation_queue,
    get_offer_contract_type_description,
    is_eds_needed,
    is_foreigner_offer,
    is_foreigner_preprofile,
    is_hiring_intern_inside,
    has_no_documents_for_preprofile,
    has_no_documents_for_offer,
    BOOTCAMP_QUEUE_MAP,
)
from intranet.femida.src.staff.helpers import get_hr_partners
from intranet.femida.src.startrek.utils import ContractTypesEnum, ProLevelEnum, SalarySystemEnum
from intranet.femida.src.vacancies.choices import VACANCY_TYPES, VACANCY_REPLACEMENT_REASONS
from intranet.femida.src.vacancies.startrek.serializers import VACANCY_TYPES_MAP


ST_TRUE, ST_FALSE = 'yes', 'no'


CONTRACT_TYPES_MAP = {
    None: None,
    '': None,
    CONTRACT_TYPES.fixed_term: ContractTypesEnum.fixed,
    CONTRACT_TYPES.fixed_term_date: ContractTypesEnum.fixed,
    CONTRACT_TYPES.project: ContractTypesEnum.fixed,
    CONTRACT_TYPES.temp_replacement: ContractTypesEnum.fixed,
    CONTRACT_TYPES.indefinite: ContractTypesEnum.open,
}

SOURCES_MAP = {
    None: None,
    '': None,
    SOURCES.internal_reference: 'internalReference',
    SOURCES.external_reference: 'extternalReference',
    SOURCES.rotation: 'rotation',
    SOURCES.found_by_hiring_manager: 'foundByHiringManager',
    SOURCES.academic_project: 'academicProject',
    SOURCES.expert_evaluation: 'afterTheInternship',
    SOURCES.candidates_list: 'fromTheListOfCandidates',
    SOURCES.agency: '\u0430gency',
    SOURCES.implant: 'implant',
    SOURCES.yandex_job_website: 'fromYandexJobWebsite ',
    SOURCES.candidates_base: 'fromTheCandidatesBase ',
    SOURCES.external_website: 'externalWebsite ',
    SOURCES.internal_event: 'internalEvent',
    SOURCES.external_event: 'externalEvent',
    SOURCES.networking: 'networking',
    SOURCES.other: 'other',
}

# Note: в тестинге Трекера у поля "source" неправильный тип,
# поэтому мы всегда прокидываем None, пока не решили тикет STARTREK-10460
if yenv.type != 'production':
    SOURCES_MAP = {}


PAYMENT_TYPES_MAP = {
    PAYMENT_TYPES.monthly: SalarySystemEnum.fixed,
    PAYMENT_TYPES.piecework: SalarySystemEnum.piecework,
    PAYMENT_TYPES.hourly: SalarySystemEnum.hourly,
}


# Note: в понятиях HR сотрудник является бывшим,
# если он реально бывший, а также если он является
# действующим сотрудником Яндекс.Денег или внешним консультантом.
# Это основано на том, что такие сотрудники всё равно проходят через
# увольнение, и в какой-то момент перед выходом становятся бывшими.
# Новые сотрудники считаются НЕ бывшими.
# Все остальные вообще не имеют определенного признака.
FORMER_EMPLOYEE_TYPES_MAP = {
    EMPLOYEE_TYPES.former: ST_TRUE,
    EMPLOYEE_TYPES.current: ST_TRUE,
    EMPLOYEE_TYPES.new: ST_FALSE,
}


FIELDS_TO_CLEAN_ON_REJECT = (
    'candidate',
    'hrEmployeeLevel',
    'legalEntity',
    'newPosition',
    'newSalary',
    'office',
    'options',
    'optionValue',
    'otherPayments',
    'relocationGrant',
    'signOnBonus',
    'signUpBonusSecondYear',
    'totalOfferLevel',
    'typeOfEmploymentContract',
    'currentCompany',
    'source',
    'sourceDescription',
    'rejectionReasonSelect',
    'conditionsOfCompetingOffer',
    'competingCompany',
    'salaryExpectations',
    'offerSentBy',
)

WORK_FORMS_MAP = {
    True: 'Основная',
    False: 'По совместительству',
}


def _get_office(offer):
    if offer.work_place == WORK_PLACES.office:
        return offer.office.name_ru
    elif offer.work_place == WORK_PLACES.home:
        return 'Надомник (%s)' % offer.homeworker_location


def _get_money_field(offer, field_name):
    amount = getattr(offer, field_name, None)
    if not amount:
        return None

    return '{amount} {currency}'.format(
        amount=amount,
        currency=offer.payment_currency.code,
    )


def _get_probation_period(obj):
    if obj.is_internal:
        return None

    unit_translation_map = {
        PROBATION_PERIOD_UNITS.week: 'нед.',
        PROBATION_PERIOD_UNITS.month: 'мес.',
    }
    return '{amount} {unit}'.format(
        amount=obj.probation_period,
        unit=unit_translation_map.get(obj.probation_period_unit),
    )


def _get_department_chain(department):
    return ' / '.join(i.name for i in department.chain)


class STBooleanField(serializers.ReadOnlyField):

    def to_representation(self, value):
        return ST_TRUE if bool(value) else ST_FALSE


class STAdaptationMeetingDateField(serializers.ReadOnlyField):
    """
    Дата беседы адаптации с новичком.
    Формируется в зависимости от даты выхода сотрудника.
    """
    def __init__(self, days_offset, **kwargs):
        self.days_offset = days_offset
        kwargs['source'] = 'join_at'
        super().__init__(**kwargs)

    def to_representation(self, value):
        dt = value + timedelta(self.days_offset)
        return dt.strftime('%Y-%m-%d')


class BPSerializerMixin:

    def get_bp_data(self, obj):
        if not hasattr(self, '_bp_data'):
            try:
                from_date = obj.join_at.strftime('%Y-%m-%d') if obj.join_at else None
                self._bp_data = get_budget_position(obj.vacancy.budget_position_id, from_date)
            except BudgetPositionError:
                self._bp_data = {}
        return self._bp_data


class BPField(serializers.SerializerMethodField):
    """
    Поле, которое тянет данные из БП.
    Может использоваться только в подклассах BPSerializerMixin.
    """
    def __init__(self, key, **kwargs):
        self.key = key
        super().__init__(**kwargs)

    def bind(self, field_name, parent):
        assert isinstance(parent, BPSerializerMixin), (
            'BPField can only be used in BPSerializerMixin subclasses'
        )
        return super().bind(field_name, parent)

    def to_representation(self, value):
        return self.parent.get_bp_data(value).get(self.key)


class IssueBaseSerializer(serializers.Serializer):

    def to_representation(self, data):
        data = super().to_representation(data)
        for key in data:
            # Затираем пустые значения
            if data[key] == '':
                data[key] = None
        return data


class SalaryIssueFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация внутреннего оффера для полей в SALARY-тикете
    """
    unique = StartrekUniqueField(queue=settings.STARTREK_SALARY_QUEUE)
    assignee = serializers.ReadOnlyField(source='current_hr_analyst.username')
    employees = serializers.SerializerMethodField()
    currentDepartment = serializers.ReadOnlyField(source='employee.department.name')
    newDepartment = serializers.ReadOnlyField(source='department.name')
    newActivity = serializers.SerializerMethodField(method_name='get_future_abc_services')
    analytics = serializers.SerializerMethodField(method_name='get_hr_analysts')
    analyst = serializers.ReadOnlyField(source='future_hr_analyst.username')
    newHead = serializers.ReadOnlyField(source='future_boss.username')
    allHead = serializers.SerializerMethodField(method_name='get_all_bosses')
    access = serializers.SerializerMethodField(method_name='get_all_bosses')
    raiseDate = serializers.DateField(source='join_at')
    staffDate = serializers.DateField(source='join_at')
    tags = serializers.SerializerMethodField()
    newSalary = serializers.SerializerMethodField()
    newPosition = serializers.ReadOnlyField(source='position.name_ru')
    office = serializers.SerializerMethodField()
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    bpNumber = serializers.ReadOnlyField(source='vacancy.budget_position_id')
    salarySystem = MapperField(PAYMENT_TYPES_MAP, source='payment_type')
    typeOfEmploymentContract = serializers.SerializerMethodField()

    def get_employees(self, obj):
        return [obj.username]

    def get_future_abc_services(self, obj):
        return [s.id for s in obj.abc_services.all()]

    def get_hr_analysts(self, obj):
        analysts = [obj.current_hr_analyst, obj.future_hr_analyst]
        return [u.username for u in analysts if u]

    def get_all_bosses(self, obj):
        return [u.username for u in obj.all_bosses]

    def get_tags(self, obj):
        return [settings.STARTREK_JOB_ROTATION_TAG]

    def get_newSalary(self, obj):
        return _get_money_field(obj, 'salary')

    def get_office(self, obj):
        return _get_office(obj)

    def get_typeOfEmploymentContract(self, obj):
        return get_offer_contract_type_description(obj)


class JobIssueFieldsBaseSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей в JOB-тикете
    """
    start = serializers.DateField(source='join_at')
    offerSentBy = serializers.ReadOnlyField(source='creator.username')
    followers = serializers.SerializerMethodField()
    activity = serializers.SerializerMethodField()
    newPosition = serializers.ReadOnlyField(source='position.name_ru')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    newDepartment = serializers.ReadOnlyField(source='department.name')
    newHead = serializers.ReadOnlyField(source='boss.username')
    geography2 = serializers.ReadOnlyField(source='geography.startrek_id', required=False)
    countryTo = serializers.SerializerMethodField()
    office = serializers.SerializerMethodField()
    newSalary = serializers.SerializerMethodField()
    hourlyRateHR = serializers.SerializerMethodField()
    professionalSphere = serializers.ReadOnlyField(source='profession.startrek_id')
    hrPl = serializers.ReadOnlyField(source='programming_language')
    candidate = serializers.SerializerMethodField()
    options = serializers.ReadOnlyField(source='rsu')
    optionValue = serializers.SerializerMethodField()
    relocationGrant = serializers.SerializerMethodField()
    relocationLevel = serializers.ReadOnlyField(source='relocation_package')
    contractType = MapperField(CONTRACT_TYPES_MAP, source='contract_type')
    totalOfferLevel = serializers.ReadOnlyField(source='grade')
    hrEmployeeLevel = serializers.SerializerMethodField()
    typeOfEmploymentContract = serializers.SerializerMethodField()
    salarySystem = MapperField(PAYMENT_TYPES_MAP, source='payment_type')

    def __init__(self, *args, **kwargs):
        # TODO: удалить после релиза FEMIDA-5447
        if not waffle.switch_is_active('enable_hourly_rate'):
            self.fields.pop('hourlyRateHR')

        super().__init__(*args, **kwargs)

    def get_countryTo(self, obj):
        geography = obj.geography
        return geography and geography.get_kind_display()

    def get_activity(self, obj):
        return [s.id for s in obj.abc_services.all()]

    def get_office(self, obj):
        return _get_office(obj)

    def get_candidate(self, obj):
        return '%s %s' % (obj.candidate.last_name, obj.candidate.first_name)

    def get_newSalary(self, obj):
        return _get_money_field(obj, 'salary')

    def get_hourlyRateHR(self, obj):
        return _get_money_field(obj, 'hourly_rate')

    def get_relocationGrant(self, obj):
        return _get_money_field(obj, 'allowance')

    def get_optionValue(self, obj):
        if obj.rsu_cost:
            return {
                'amount': float(obj.rsu_cost),
                'unit': 'USD',
            }

    def get_typeOfEmploymentContract(self, obj):
        return get_offer_contract_type_description(obj)

    def get_hrEmployeeLevel(self, obj):
        if obj.professional_level:
            return getattr(ProLevelEnum, obj.professional_level)

    def get_followers(self, obj):
        return {
            'add': [obj.creator.username],
        }

    @staticmethod
    def serialize(instance):
        """
        Сериализует оффер для согласования в JOB-тикете,
        генерит разный набор данных для внутреннего и внешнего офферов.
        :param instance: Offer
        """
        if instance.is_internal:
            return JobIssueFieldsInternalSerializer(instance).data
        return JobIssueFieldsExternalSerializer(instance).data


class JobIssueFieldsInternalSerializer(JobIssueFieldsBaseSerializer):
    """
    Сериализация полей внутреннего оффера для согласования в JOB-тикете
    """
    tags = serializers.SerializerMethodField()
    afterInternship = MapperField(
        source='employee_type',
        mapping={EMPLOYEE_TYPES.intern: ST_TRUE},
    )

    def get_tags(self, obj):
        return {
            'add': [settings.STARTREK_JOB_ROTATION_TAG],
        }


class JobIssueFieldsExternalSerializer(JobIssueFieldsBaseSerializer):
    """
    Сериализация полей внешнего оффера для согласования в JOB-тикете
    """
    currentCompany = serializers.ReadOnlyField(source='current_company')
    otherPayments = serializers.ReadOnlyField(source='other_payments')
    salaryExpectations = serializers.SerializerMethodField()
    signOnBonus = serializers.SerializerMethodField()
    signUpBonusSecondYear = serializers.SerializerMethodField()
    signUpGross = StartrekLocalField.new(
        settings.STARTREK_JOB_SIGNUP_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='signup_bonus_gross',
    )
    signUpSecondYearGross = StartrekLocalField.new(
        settings.STARTREK_JOB_SIGNUP_2YEAR_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='signup_2year_bonus_gross',
    )
    welcomeBonusGross = StartrekLocalField.new(
        settings.STARTREK_JOB_WELCOME_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='welcome_bonus_gross',
    )
    welcomeBonusSecondYearGross = StartrekLocalField.new(
        settings.STARTREK_JOB_WELCOME_2YEAR_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='welcome_2year_bonus_gross',
    )
    currency = serializers.ReadOnlyField(source='payment_currency.code')
    source = MapperField(SOURCES_MAP)
    sourceDescription = serializers.ReadOnlyField(source='source_description')
    formerWorker = MapperField(FORMER_EMPLOYEE_TYPES_MAP, source='employee_type')

    def __init__(self, *args, **kwargs):
        # TODO: удалить после релиза FEMIDA-7240
        if not waffle.switch_is_active('enable_new_bonus'):
            self.fields.pop('signUpGross')
            self.fields.pop('signUpSecondYearGross')
            self.fields.pop('welcomeBonusGross')
            self.fields.pop('welcomeBonusSecondYearGross')

        super().__init__(*args, **kwargs)

    def get_salaryExpectations(self, obj):
        if obj.salary_expectations and obj.salary_expectations_currency:
            return {
                'amount': float(obj.salary_expectations),
                'unit': obj.salary_expectations_currency.code,
            }

    def get_signOnBonus(self, obj):
        return _get_money_field(obj, 'signup_bonus')

    def get_signUpBonusSecondYear(self, obj):
        return _get_money_field(obj, 'signup_2year_bonus')


class HRIssueFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей в HR-тикете при создании.
    """
    unique = StartrekUniqueField(queue=settings.STARTREK_HR_QUEUE)
    start = serializers.DateField(source='join_at')
    internship = STBooleanField(source='is_internship')
    hrPartner = serializers.SerializerMethodField()
    office = serializers.SerializerMethodField()
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    components = serializers.SerializerMethodField()
    tags = serializers.SerializerMethodField()
    salarySystem = MapperField(PAYMENT_TYPES_MAP, source='payment_type')
    newPosition = serializers.ReadOnlyField(source='position.name_ru')
    newDepartment = serializers.ReadOnlyField(source='department.name')
    head = serializers.ReadOnlyField(source='boss.username')
    formOfWork = MapperField(WORK_FORMS_MAP, source='is_main_work_place')
    employmentForm = serializers.ReadOnlyField(source='employment_type')
    fullPassportName = serializers.ReadOnlyField(source='full_name')
    completeName = serializers.SerializerMethodField()
    userLogin = serializers.ReadOnlyField(source='username')
    phoneNumber = serializers.ReadOnlyField(source='profile.phone')
    probation = serializers.SerializerMethodField()
    newSalary = serializers.SerializerMethodField()
    typeOfEmploymentContract = serializers.SerializerMethodField()
    bpNumber = serializers.ReadOnlyField(source='vacancy.budget_position_id')
    endDateOfEmploymentContract = serializers.DateField(source='contract_end')
    vacancyType = MapperField(VACANCY_TYPES_MAP, source='vacancy.type')
    address = serializers.ReadOnlyField(source='profile.residence_address')
    recruiter = serializers.ReadOnlyField(source='creator.username')
    currency = serializers.ReadOnlyField(source='payment_currency.code')
    candidateCitizenship = serializers.ReadOnlyField(source='profile.citizenship')
    candidateEmail = serializers.ReadOnlyField(source='profile.home_email')

    def get_hrPartner(self, obj):
        """
        В Трекере поле принимает один логин,
        в то время как HR-партнеров может быть несколько.
        Поэтому берем любого случайного FEMIDA-2694.
        """
        usernames = list({u.username for u in get_hr_partners(obj.department)})
        if usernames:
            return random.choice(usernames)

    def get_office(self, obj):
        return _get_office(obj)

    def get_components(self, obj):
        components = [settings.STARTREK_HR_NEW_EMPLOYEE_COMPONENT]
        if is_foreigner_offer(obj):
            components.append(settings.STARTREK_HR_FOREIGNERS_COMPONENT)
        if obj.need_relocation:
            components.append(settings.STARTREK_HR_RELOCATION_COMPONENT)
        if has_no_documents_for_offer(obj):
            components.append(settings.STARTREK_HR_NO_DOCUMENTS_COMPONENT)
        if obj.contract_type != CONTRACT_TYPES.indefinite:
            components.append(settings.STARTREK_HR_FIXED_TERM_CONTRACT_COMPONENT)
        if obj.is_internship:
            components.append(settings.STARTREK_HR_INTERN_COMPONENT)
        return components

    def get_tags(self, obj):
        tags = []
        if not self.root.context['has_bank_details']:
            tags.append(settings.STARTREK_HR_NO_BANK_DETAILS_TAG)
        return tags

    def get_completeName(self, obj):
        return f'{obj.profile.first_name_en} {obj.profile.last_name_en}'

    def get_probation(self, obj):
        return _get_probation_period(obj)

    def get_newSalary(self, obj):
        return _get_money_field(obj, 'salary')

    def get_typeOfEmploymentContract(self, obj):
        return get_offer_contract_type_description(obj)


class HRIssueUpdateFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей HR-тикета,
    которые могут периодически меняться.
    """
    start = serializers.DateField(source='join_at')


class HRIssueContextSerializer(BPSerializerMixin, serializers.Serializer):
    """
    Сериализация оффера для описания в HR-тикете
    """
    business_group = BPField('businessGroup')
    country = BPField('recCountry')
    headcount = BPField('is_headcount')
    instead_of = serializers.SerializerMethodField()
    employment_book = MapperField(EMPLOYMENT_BOOK_TRANSLATIONS, source='profile.employment_book')
    is_hiring_intern_inside = serializers.SerializerMethodField()
    is_eds_needed = serializers.SerializerMethodField()

    def get_instead_of(self, obj):
        # В кадрах понимание "замещения"
        # - это только в случае декрета (FEMIDA-1845).
        is_fixed_term = obj.contract_type in (
            CONTRACT_TYPES.fixed_term,
            CONTRACT_TYPES.fixed_term_date,
            CONTRACT_TYPES.temp_replacement,
        )

        if (obj.vacancy.type == VACANCY_TYPES.replacement
                and obj.vacancy.replacement_reason == VACANCY_REPLACEMENT_REASONS.maternity_leave
                and is_fixed_term):
            return {
                'full_name': obj.vacancy.instead_of.get_full_name(),
                'username': obj.vacancy.instead_of.username,
            }

    def get_is_hiring_intern_inside(self, obj):
        return is_hiring_intern_inside(obj)

    def get_is_eds_needed(self, obj):
        return is_eds_needed(obj)


class RelocationIssueFieldsBaseSerializer(IssueBaseSerializer):
    """
    Базовый класс сериализатора для различных условий релокации
    """
    start = serializers.DateField(source='join_at')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    office = serializers.SerializerMethodField()
    fullPassportName = serializers.ReadOnlyField(source='full_name')
    phoneNumber = serializers.ReadOnlyField(source='profile.phone')
    newDepartment = serializers.ReadOnlyField(source='department.name')
    recruiter = serializers.ReadOnlyField(source='creator.username')
    address = serializers.ReadOnlyField(source='profile.residence_address')
    cityTo = serializers.ReadOnlyField(source='office.city.name_ru')
    totalOfferLevel = serializers.ReadOnlyField(source='grade')
    candidateCitizenship = serializers.ReadOnlyField(source='profile.citizenship')
    candidateEmail = serializers.ReadOnlyField(source='profile.home_email')
    currency = serializers.ReadOnlyField(source='payment_currency.code')
    signOnBonus = serializers.SerializerMethodField()
    signUpBonusSecondYear = serializers.SerializerMethodField()
    relocationGrant = serializers.SerializerMethodField()
    relocationLevel = serializers.ReadOnlyField(source='relocation_package')

    def get_office(self, obj):
        return _get_office(obj)

    def get_signOnBonus(self, obj):
        return _get_money_field(obj, 'signup_bonus')

    def get_signUpBonusSecondYear(self, obj):
        return _get_money_field(obj, 'signup_2year_bonus')

    def get_relocationGrant(self, obj):
        return _get_money_field(obj, 'allowance')


class RelocationIssueFieldsSerializer(RelocationIssueFieldsBaseSerializer):
    """
    Сериализация оффера для полей в RELOCATION-тикете при создании.
    """
    queue = serializers.SerializerMethodField()
    unique = StartrekUniqueField(queue=settings.STARTREK_RELOCATION_QUEUE)
    signUpGross = StartrekLocalField.new(
        settings.STARTREK_RELOCATION_SIGNUP_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='signup_bonus_gross',
    )
    signUpSecondYearGross = StartrekLocalField.new(
        settings.STARTREK_RELOCATION_SIGNUP_2YEAR_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='signup_2year_bonus_gross',
    )

    def __init__(self, *args, **kwargs):
        # TODO: удалить после релиза FEMIDA-7240
        if not waffle.switch_is_active('enable_new_bonus'):
            self.fields.pop('signUpGross')
            self.fields.pop('signUpSecondYearGross')

        super().__init__(*args, **kwargs)

    def get_queue(self, obj):
        return settings.STARTREK_RELOCATION_QUEUE


class InternRelocationIssueFieldsSerializer(RelocationIssueFieldsBaseSerializer):
    """
    Сериализация оффера для полей в RELOCATEINTERN-тикете при создании.
    """
    queue = serializers.SerializerMethodField()
    unique = StartrekUniqueField(queue=settings.STARTREK_INTERN_RELOCATION_QUEUE)
    endDateOfEmploymentContract = serializers.DateField(source='contract_end')

    def get_queue(self, obj):
        return settings.STARTREK_INTERN_RELOCATION_QUEUE


class SignupIssueFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей в SIGNUP-тикете при создании.
    """
    unique = StartrekUniqueField(queue=settings.STARTREK_SIGNUP_QUEUE)
    start = serializers.DateField(source='join_at')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    office = serializers.SerializerMethodField()
    fullPassportName = serializers.ReadOnlyField(source='full_name')
    newDepartment = serializers.ReadOnlyField(source='department.name')
    recruiter = serializers.ReadOnlyField(source='creator.username')
    relocationGrant = serializers.SerializerMethodField()
    signOnBonus = serializers.SerializerMethodField()
    signUpBonusSecondYear = serializers.SerializerMethodField()
    signUpGross = StartrekLocalField.new(
        settings.STARTREK_SIGNUP_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='signup_bonus_gross',
    )
    signUpSecondYearGross = StartrekLocalField.new(
        settings.STARTREK_SIGNUP_2YEAR_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='signup_2year_bonus_gross',
    )
    currency = serializers.ReadOnlyField(source='payment_currency.code')
    candidateCitizenship = serializers.ReadOnlyField(source='profile.citizenship')

    def __init__(self, *args, **kwargs):
        # TODO: удалить после релиза FEMIDA-7240
        if not waffle.switch_is_active('enable_new_bonus'):
            self.fields.pop('signUpGross')
            self.fields.pop('signUpSecondYearGross')

        super().__init__(*args, **kwargs)

    def get_relocationGrant(self, obj):
        return _get_money_field(obj, 'allowance')

    def get_signOnBonus(self, obj):
        return _get_money_field(obj, 'signup_bonus')

    def get_signUpBonusSecondYear(self, obj):
        return _get_money_field(obj, 'signup_2year_bonus')

    def get_office(self, obj):
        return _get_office(obj)


class WelcomeBonusIssueFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера c бонусами для полей в BONUS-тикете при создании.
    """
    queue = serializers.SerializerMethodField()
    unique = StartrekUniqueField(queue=settings.STARTREK_BONUS_QUEUE)
    start = serializers.DateField(source='join_at')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    office = serializers.SerializerMethodField()
    fullPassportName = serializers.ReadOnlyField(source='full_name')
    newDepartment = serializers.ReadOnlyField(source='department.name')
    recruiter = serializers.ReadOnlyField(source='creator.username')
    welcomeBonusGross = StartrekLocalField.new(
        settings.STARTREK_BONUS_WELCOME_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='welcome_bonus_gross',
    )
    welcomeBonusSecondYearGross = StartrekLocalField.new(
        settings.STARTREK_BONUS_WELCOME_2YEAR_GROSS_FIELD,
        field_class=serializers.DecimalField,
        max_digits=10,
        decimal_places=2,
        source='welcome_2year_bonus_gross',
    )
    currency = serializers.ReadOnlyField(source='payment_currency.code')
    candidateCitizenship = serializers.ReadOnlyField(source='profile.citizenship')

    def __init__(self, *args, **kwargs):
        # TODO: удалить после релиза FEMIDA-7240
        if not waffle.switch_is_active('enable_new_bonus'):
            self.fields.pop('welcomeBonusGross')
            self.fields.pop('welcomeBonusSecondYearGross')

        super().__init__(*args, **kwargs)

    def get_queue(self, _):
        return settings.STARTREK_BONUS_QUEUE

    def get_office(self, obj):
        return _get_office(obj)


class RelocationIssueContextSerializer(serializers.Serializer):
    """
    Сериализация оффера для описания в RELOCATION-тикете
    """
    skype = serializers.SerializerMethodField()
    office = serializers.SerializerMethodField()

    def get_skype(self, obj):
        return get_skype_accounts(obj.candidate)

    def get_office(self, obj):
        return _get_office(obj)


class ReferenceOfferIssueFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей тикетов REFERENCE,
    при выводе сотрудника на Стафф
    """
    employee = serializers.ReadOnlyField(source='username')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    newPosition = serializers.ReadOnlyField(source='position.name_ru')
    professionalSphere = serializers.ReadOnlyField(source='profession.startrek_id')


class AdaptationIssueFieldsBaseSerializer(IssueBaseSerializer):

    start = serializers.DateField(source='join_at')
    probationPeriod = serializers.SerializerMethodField()
    personalEmail = serializers.ReadOnlyField(source='profile.home_email')
    personalPhone = serializers.ReadOnlyField(source='profile.phone')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    office = serializers.SerializerMethodField()
    typeOfEmploymentContract = serializers.SerializerMethodField()
    formerWorker = MapperField(FORMER_EMPLOYEE_TYPES_MAP, source='employee_type')
    hrbp = serializers.SerializerMethodField()
    recruiter = serializers.ReadOnlyField(source='creator.username')
    head = serializers.ReadOnlyField(source='boss.username')
    newPosition = serializers.ReadOnlyField(source='position.name_ru')
    newDepartment = serializers.SerializerMethodField()
    tags = serializers.SerializerMethodField()
    address = serializers.ReadOnlyField(source='profile.residence_address')
    candidateCitizenship = serializers.ReadOnlyField(source='profile.citizenship')
    hrPl = serializers.ReadOnlyField(source='programming_language')

    def get_probationPeriod(self, obj):
        return _get_probation_period(obj)

    def get_office(self, obj):
        return _get_office(obj)

    def get_typeOfEmploymentContract(self, obj):
        return get_offer_contract_type_description(obj)

    def get_hrbp(self, obj):
        # HR-партнеры
        return list({u.username for u in get_hr_partners(obj.department)})

    def get_tags(self, obj):
        tags = []
        if obj.is_internal:
            tags.append(settings.STARTREK_ADAPTATION_ROTATION_TAG)
        if obj.form_type == FORM_TYPES.international:
            tags.append(FORM_TYPES.international)
            main_language = obj.candidate.main_language
            if main_language:
                if main_language.tag == 'en':
                    tags.append(settings.STARTREK_ADAPTATION_LANG_ENG_TAG)
                elif main_language.tag == 'ru':
                    tags.append(settings.STARTREK_ADAPTATION_LANG_RUS_TAG)
        return tags

    def get_newDepartment(self, obj):
        return _get_department_chain(obj.department)


class RegularAdaptationIssueFieldsSerializer(AdaptationIssueFieldsBaseSerializer):
    """
    Сериализация оффера для полей в тикете Адаптации при создании.
    """
    queue = serializers.SerializerMethodField()
    unique = StartrekUniqueField(queue=settings.STARTREK_REGULAR_ADAPTATION_QUEUE)

    # qazaq@: я устал бороться за более-менее нормальные
    # названия полей в Трекере, поэтому оставляем эту ерунду.
    # Даты бесед адаптации с новичком:
    # +30, +60, +90 дней от даты выхода.
    iS = STAdaptationMeetingDateField(30)
    iS2 = STAdaptationMeetingDateField(60)
    iS3 = STAdaptationMeetingDateField(90)
    components = serializers.SerializerMethodField()
    formOfWork = MapperField(WORK_FORMS_MAP, source='is_main_work_place')
    employmentForm = serializers.ReadOnlyField(source='employment_type')

    def get_queue(self, obj):
        return (
            get_department_adaptation_queue(obj.department_id) or
            settings.STARTREK_REGULAR_ADAPTATION_QUEUE
        )

    def get_components(self, obj):
        components = []
        if is_foreigner_offer(obj):
            components.append(settings.STARTREK_ADAPTATION_FOREIGNERS_COMPONENT)
        return components


class RegularAdaptationIssueUpdateFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей тикета Адаптации,
    которые могут периодически меняться.
    """
    start = serializers.DateField(source='join_at')
    iS = STAdaptationMeetingDateField(30)
    iS2 = STAdaptationMeetingDateField(60)
    iS3 = STAdaptationMeetingDateField(90)


class InternAdaptationIssueFieldsSerializer(AdaptationIssueFieldsBaseSerializer):
    """
    Сериализация оффера для полей в тикете Адаптации стажеров при создании.
    """
    queue = serializers.SerializerMethodField()
    unique = StartrekUniqueField(queue=settings.STARTREK_INTERN_ADAPTATION_QUEUE)
    end = serializers.DateField(source='contract_end')
    cityTo = serializers.ReadOnlyField(source='office.city.name_ru')
    professionalSphere = serializers.ReadOnlyField(source='profession.startrek_id')

    def get_queue(self, obj):
        return settings.STARTREK_INTERN_ADAPTATION_QUEUE


class InternAdaptationIssueUpdateFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей тикета Адаптации стажеров,
    которые могут периодически меняться.
    """
    start = serializers.DateField(source='join_at')
    end = serializers.DateField(source='contract_end')


class RotationAdaptationIssueFieldsSerializer(AdaptationIssueFieldsBaseSerializer):
    """
    Сериализация оффера для создания тикета адаптации при ротации.
    """
    queue = FakeField(settings.STARTREK_ROTATION_ADAPTATION_QUEUE)
    unique = StartrekUniqueField(queue=settings.STARTREK_ROTATION_ADAPTATION_QUEUE)
    employee = serializers.ReadOnlyField(source='username')
    currentDepartment = serializers.SerializerMethodField()
    current_business_unit = StartrekLocalField.new(
        settings.STARTREK_ROTATION_ADAPTATION_CURRENT_BU_FIELD,
        source='employee.department.business_unit_or_experiment.url',
    )
    new_business_unit = StartrekLocalField.new(
        settings.STARTREK_ROTATION_ADAPTATION_NEW_BU_FIELD,
        source='department.business_unit_or_experiment.url',
    )

    def get_currentDepartment(self, obj):
        return _get_department_chain(obj.employee.department)


class RotationAdaptationIssueUpdateFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для редактирования тикета адаптации при ротации.
    """
    start = serializers.DateField(source='join_at')


class BootcampIssueCreateFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация оффера для полей тикета Буткэмпа.
    """
    queue = serializers.SerializerMethodField()

    start = serializers.DateField(source='join_at')
    candidate = serializers.CharField(source='candidate.get_full_name')
    candidateCity = serializers.ReadOnlyField(source='office.city.name_ru')
    boards = serializers.SerializerMethodField()

    def get_queue(self, obj):
        for department_id in BOOTCAMP_QUEUE_MAP:
            if obj.department.is_in_tree(department_id):
                return BOOTCAMP_QUEUE_MAP[department_id]

    def get_boards(self, obj):
        return [settings.STARTREK_BOOTCAMP_BOARD_ID]


class AdaptationIssueContextSerializer(serializers.Serializer):
    """
    Сериализация оффера для описания в тикете Адаптации
    """
    office_verbose = serializers.SerializerMethodField()

    def get_office_verbose(self, obj):
        return _get_office(obj)


def serialize_offer_for_adaptation_issue_create(offer):
    if offer.is_internship:
        serializer_class = InternAdaptationIssueFieldsSerializer
    elif offer.is_rotation_within_yandex:
        serializer_class = RotationAdaptationIssueFieldsSerializer
    else:
        serializer_class = RegularAdaptationIssueFieldsSerializer
    return serializer_class(offer).data


def serialize_offer_for_adaptation_issue_update(offer):
    if offer.is_internship:
        serializer_class = InternAdaptationIssueUpdateFieldsSerializer
    elif offer.is_rotation_within_yandex:
        serializer_class = RotationAdaptationIssueUpdateFieldsSerializer
    else:
        serializer_class = RegularAdaptationIssueUpdateFieldsSerializer
    return serializer_class(offer).data


class EdsIssueFieldsSerializer(IssueBaseSerializer):

    newDepartment = serializers.ReadOnlyField(source='department.name')
    start = serializers.DateField(source='join_at')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    userLogin = serializers.ReadOnlyField(source='username')
    tags = serializers.SerializerMethodField()
    links = serializers.SerializerMethodField()
    office = serializers.SerializerMethodField()

    def get_tags(self, obj):
        return [settings.STARTREK_EDS_TAG]

    def get_links(self, obj):
        if obj.startrek_hr_key:
            return [{
                'relationship': 'relates',
                'issue': obj.startrek_hr_key,
            }]

    def get_office(self, obj):
        if hasattr(obj, 'office'):
            return _get_office(obj)


# Препрофайлы


class PreprofileHRIssueFieldsSerializer(IssueBaseSerializer):
    """
    Сериализация препрофайла для полей в HR-тикете при создании.
    """

    # Note: есть риск, что unique совпадет с оффером,
    # если существует оффер с таким же id, созданный ровно
    # в то же время. Но это какая-то дичь
    unique = StartrekUniqueField(queue=settings.STARTREK_HR_QUEUE)

    start = serializers.DateField(source='join_at')
    legalEntity = serializers.ReadOnlyField(source='org.startrek_id')
    formerWorker = MapperField(FORMER_EMPLOYEE_TYPES_MAP, source='employee_type')
    recruiter = serializers.ReadOnlyField(source='recruiter.username')
    components = serializers.SerializerMethodField()
    tags = serializers.SerializerMethodField()
    newPosition = serializers.ReadOnlyField(source='position.name_ru')
    newDepartment = serializers.ReadOnlyField(source='department.name')
    head = serializers.ReadOnlyField(source='boss.username')
    userLogin = serializers.ReadOnlyField(source='username')

    def get_components(self, obj):
        components = [
            settings.STARTREK_HR_NEW_EMPLOYEE_COMPONENT,
            settings.STARTREK_HR_ASSESSORS_COMPONENT,
        ]
        if is_foreigner_preprofile(obj):
            components.append(settings.STARTREK_HR_FOREIGNERS_COMPONENT)
        if has_no_documents_for_preprofile(obj):
            components.append(settings.STARTREK_HR_NO_DOCUMENTS_COMPONENT)
        return components

    def get_tags(self, obj):
        tags = []
        if not self.root.context['has_bank_details']:
            tags.append(settings.STARTREK_HR_NO_BANK_DETAILS_TAG)
        return tags
