import logging
import pytz
import waffle

from datetime import timedelta, datetime

from constance import config
from django.contrib.auth import get_user_model
from django.db import transaction
from django.db.models import Exists, OuterRef
from django.utils import timezone
from ids.exceptions import BackendError

from intranet.femida.src.candidates.tasks import finish_reference_issue_task
from intranet.femida.src.celery_app import app, get_retry_countdown
from intranet.femida.src.notifications.offers import (
    OfferInternalAcceptedNotification,
    OfferAcceptedNotification,
    notify_hr_about_docs_processing_problem,
)
from intranet.femida.src.oebs.api import OebsAPI
from intranet.femida.src.offers.choices import (
    EMPLOYEE_TYPES,
    OFFER_STATUSES,
    INTERNAL_EMPLOYEE_TYPES,
    EXTERNAL_EMPLOYEE_TYPES,
    OFFER_DOCS_PROCESSING_STATUSES,
    OFFER_DOCS_PROCESSING_RESOLUTIONS,
    OFFER_ATTACHMENT_TYPES,
    PAYMENT_TYPES,
)
from intranet.femida.src.offers.helpers import (
    is_adaptation_needed,
    is_bootcamp_offer,
    get_candidate_alive_verifications_prefetch,
    has_signup_bonus,
)
from intranet.femida.src.offers.models import Link, Offer, Preprofile, InternalOffer
from intranet.femida.src.offers.newhire.serializers import NewhirePreprofileSerializer
from intranet.femida.src.offers.oebs.controllers import check_oebs_login
from intranet.femida.src.offers.signals import offer_unapproved
from intranet.femida.src.offers.startrek.issues import (
    create_salary_issue,
    create_hr_issue,
    create_relocation_issue,
    create_signup_issue,
    create_adaptation_issue,
    create_bootcamp_issue,
    create_offer_eds_issue,
    create_preprofile_eds_issue,
    create_preprofile_hr_issue,
    create_welcome_bonus_issue,
    update_preprofile_issues,
)
from intranet.femida.src.offers.startrek.serializers import ReferenceOfferIssueFieldsSerializer
from intranet.femida.src.staff.models import StaffSync
from intranet.femida.src.startrek.operations import (
    IssueTransitionOperation,
    IssueUpdateOperation,
)
from intranet.femida.src.startrek.tasks import add_issue_comment_task
from intranet.femida.src.startrek.utils import (
    get_issue,
    StartrekError,
    StatusEnum,
    ResolutionEnum,
    TransitionEnum,
)
from intranet.femida.src.utils.datetime import shifted_now
from intranet.femida.src.utils.lock import locked_task
from intranet.femida.src.utils.newhire import NewhireAPI, NewhireError
from intranet.femida.src.vacancies.choices import VACANCY_TYPES


User = get_user_model()

logger = logging.getLogger(__name__)


def _get_offers_closed_last_day():
    """
    Отдаёт офферы, закрытые за последние сутки.
    Час при этом сохраняется, а минуты, секунды и т.д. обнуляются.
    Например, если вызвать 01.12.2019 в 23:16:13.300010,
    отдаст офферы, закрытые в промежутке [30.11 23:00 - 01.12 23:00)
    """
    tz = pytz.timezone('Europe/Moscow')
    to_date = timezone.now().astimezone(tz)
    to_date = to_date.replace(minute=0, second=0, microsecond=0)
    from_date = to_date - timedelta(days=1)
    return Offer.unsafe.filter(
        status=OFFER_STATUSES.closed,
        closed_at__gte=from_date,
        closed_at__lt=to_date,
    )


@app.task
@locked_task
def clean_expired_offer_links():
    links = Link.objects.filter(expiration_time__lte=timezone.now())
    links.delete()


@app.autoretry_task(max_retries=5)
def create_salary_issue_task(offer_id, initiator_id):
    offer = InternalOffer.unsafe.get(id=offer_id)
    initiator = User.objects.get(id=initiator_id)
    if offer.startrek_salary_key:
        logger.warning(
            'SALARY issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_salary_key,
        )
        return
    create_salary_issue(offer, initiator)


@app.autoretry_task(max_retries=5)
def create_hr_issue_task(offer_id, has_bank_details=True):
    offer = (
        Offer.unsafe
        .select_related(
            'candidate',
            'org',
            'office',
            'position',
            'department',
            'profile',
            'creator',
        )
        .get(id=offer_id)
    )
    if offer.startrek_hr_key:
        logger.warning(
            'HR issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_hr_key,
        )
        return

    startrek_timeout = 4 + 2 * create_hr_issue_task.request.retries
    create_hr_issue(offer, has_bank_details, startrek_timeout)


@app.autoretry_task(max_retries=5)
def create_relocation_issue_task(offer_id):
    offer = Offer.unsafe.get(id=offer_id)
    if not offer.need_relocation:
        return
    if offer.startrek_relocation_key:
        logger.warning(
            'RELOCATION issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_relocation_key,
        )
        return
    create_relocation_issue(offer)


@app.autoretry_task(max_retries=5)
def create_signup_issue_task(offer_id):
    offer = Offer.unsafe.get(id=offer_id)
    if not has_signup_bonus(offer) or offer.need_relocation:
        return
    if offer.startrek_signup_key:
        logger.warning(
            'SIGNUP issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_signup_key,
        )
        return
    create_signup_issue(offer)


@app.autoretry_task(max_retries=5)
def create_bonus_issue_task(offer_id):
    offer = Offer.unsafe.get(id=offer_id)
    if not (offer.welcome_bonus_gross or offer.welcome_2year_bonus_gross):
        return
    if offer.startrek_bonus_key:
        logger.warning(
            'BONUS issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_bonus_key,
        )
        return
    create_welcome_bonus_issue(offer)


@app.autoretry_task(max_retries=5)
def create_adaptation_issue_task(offer_id):
    offer = Offer.unsafe.get(id=offer_id)
    if not is_adaptation_needed(offer):
        return
    if offer.startrek_adaptation_key:
        logger.warning(
            'ADAPTATION issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_adaptation_key,
        )
        return
    create_adaptation_issue(offer)


@app.autoretry_task(max_retries=5)
def create_bootcamp_issue_task(offer_id):
    offer = Offer.unsafe.get(id=offer_id)
    if not is_bootcamp_offer(offer):
        return
    if offer.startrek_bootcamp_key:
        logger.warning(
            'BOOTCAMP issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_bootcamp_key,
        )
        return
    create_bootcamp_issue(offer)


@app.autoretry_task(max_retries=5)
def create_eds_issue_task(offer_id):
    offer = (
        Offer.unsafe
        .select_related(
            'org',
            'office',
            'position',
            'department',
        )
        .get(id=offer_id)
    )
    if not offer.is_eds_phone_verified:
        return
    if offer.startrek_eds_key:
        logger.warning(
            'EDS issue was already created for offer %s: %s',
            offer_id,
            offer.startrek_eds_key,
        )
        return
    create_offer_eds_issue(offer)


@app.task
@locked_task
def close_job_issues_task(offer_ids=None):
    """
    Таск массового закрытия JOB-тикетов.
    :param offer_ids: список id офферов. Нужен для запуска таски из админки
        для конкретных офферов (в основном, для тестирования)
    """
    if offer_ids:
        offers = Offer.unsafe.filter(
            status=OFFER_STATUSES.closed,
            id__in=offer_ids,
        )
    else:
        offers = _get_offers_closed_last_day()

    offers_data = offers.values_list(
        'vacancy__startrek_key',
        'username',
        'employee_type',
    )

    for startrek_key, login, employee_type in offers_data:
        if employee_type in INTERNAL_EMPLOYEE_TYPES:
            transition = TransitionEnum.close_internal
        else:
            transition = TransitionEnum.employment
        operation = IssueTransitionOperation(startrek_key)
        try:
            operation(transition, employees=[login])
        except StartrekError:
            operation.delay(transition, employees=[login])


@app.task(max_retries=3)
@locked_task
def set_issues_employee_task(offer_closed_time=None):
    """
    Таск добавления логина сотрудника в тикеты очередей
    HR, RELOCATION, SIGNUP, ADAPTATION, BOOTCAMPER(BACK/FRONT)
    """
    dt_format = '%Y-%m-%d %H'
    offer_closed_time = (
        datetime.strptime(offer_closed_time, dt_format)
        if offer_closed_time
        else shifted_now(
            hours=-config.TRACKER_STAFF_SYNC_TIME - 1,
        ).replace(minute=0, second=0, microsecond=0)
    ).astimezone(pytz.utc)

    try:
        offers = Offer.unsafe.filter(
            status=OFFER_STATUSES.closed,
            closed_at__gte=offer_closed_time,
            closed_at__lt=offer_closed_time + timedelta(hours=1),
        )
    except Exception as exc:
        set_issues_employee_task.retry(
            countdown=get_retry_countdown(set_issues_employee_task.request.retries),
            args=[],
            kwargs={'offer_closed_time': offer_closed_time.strftime(dt_format)},
            exc=exc,
        )
        return

    issues_to_update = (
        'startrek_hr_key',
        'startrek_relocation_key',
        'startrek_signup_key',
        'startrek_bonus_key',
        'startrek_adaptation_key',
        'startrek_bootcamp_key',
        'startrek_eds_key',
    )
    for offer in offers:
        for field in issues_to_update:
            key = getattr(offer, field)
            if key:
                operation = IssueUpdateOperation(key)
                try:
                    operation(
                        employee=offer.username,
                        employees=[offer.username],
                    )
                except StartrekError:
                    operation.delay(
                        employee=offer.username,
                        employees=[offer.username],
                    )


@app.task
@locked_task
def finish_references_task(offer_ids=None):
    """
    Таск перевода связанных тикетов REFERENCE в статус "Добавлен на Стафф".

    Это финальная точка работы Фемиды с тикетом. Закрывают его руками после выплаты вознаграждения.
    :param offer_ids: список id офферов. Нужен для запуска таски из админки
        для конкретных офферов (в основном, для тестирования)
    """
    if offer_ids:
        offers = Offer.unsafe.filter(
            status=OFFER_STATUSES.closed,
            id__in=offer_ids,
        )
    else:
        offers = _get_offers_closed_last_day()

    for offer in offers:
        fields = ReferenceOfferIssueFieldsSerializer(offer).data
        finish_reference_issue_task.delay(offer.candidate_id, **fields)


@app.autoretry_task(max_retries=4)
def save_in_newhire_task(offer_id):
    from intranet.femida.src.offers.controllers import OfferCtl
    offer = Offer.unsafe.get(id=offer_id)
    if offer.newhire_id:
        return
    ctl = OfferCtl(offer)
    ctl.save_in_newhire()


@app.autoretry_task(max_retries=3)
def sync_newhire_offer_task(offer_id):
    # Синкаем только офферы с newhire_id и до закрытия
    offer = (
        Offer.unsafe
        .filter(
            id=offer_id,
            newhire_id__isnull=False,
            status=OFFER_STATUSES.accepted,
        )
        .values_list('id', 'newhire_id', named=True)
        .first()
    )
    if offer is None:
        return

    data, status_code = NewhireAPI.get_preprofile(offer.newhire_id)
    if status_code != 200:
        raise NewhireError(f'Newhire: get offer {offer_id} failed: {status_code}, {data}')

    remote_offer_id = data.get('femida_offer_id')
    if offer_id != remote_offer_id:
        raise NewhireError(
            f'Newhire: inconsistent data, preprofile {offer.newhire_id} '
            f'refers to offer {remote_offer_id}'
        )

    from intranet.femida.src.offers.controllers import OfferCtl
    OfferCtl.update_from_newhire_with_lock(offer.id, data)


@app.autoretry_task(max_retries=3)
def sync_newhire_preprofile_task(preprofile_id):
    from intranet.femida.src.offers.controllers import PreprofileCtl
    data, status_code = NewhireAPI.get_preprofile(preprofile_id)
    if status_code != 200:
        raise NewhireError(f'Newhire: get preprofile {preprofile_id} failed: {status_code}, {data}')

    try:
        preprofile = Preprofile.objects.get(id=preprofile_id)
    except Preprofile.DoesNotExist:
        logger.warning('Preprofile not found %s', preprofile_id)
        return
    else:
        ctl = PreprofileCtl(preprofile)
        ctl.update_from_newhire(data)


@app.autoretry_task(max_retries=3)
@locked_task
def sync_newhire_offers_task():
    from intranet.femida.src.offers.controllers import OfferCtl, PreprofileCtl

    with transaction.atomic():
        sync_entry, created = StaffSync.objects.get_or_create(
            name='newhire_offers',
            defaults={
                'synced_at': timezone.datetime.min.replace(tzinfo=pytz.utc),
            },
        )
        synced_at = sync_entry.synced_at
        response_data, status_code = NewhireAPI.get_preprofiles(date_from=synced_at)
        if status_code != 200:
            raise NewhireError(
                f'Newhire: get preprofiles modified after {synced_at} failed: '
                f'{status_code}, {response_data}'
            )

        newhire_preprofiles = response_data
        if not newhire_preprofiles:
            return

        naive_dt = max(
            timezone.datetime.strptime(preprofile['modified_at'], "%Y-%m-%dT%H:%M:%S")
            for preprofile in newhire_preprofiles
        )
        tz = pytz.timezone('Europe/Moscow')
        aware_dt = tz.localize(naive_dt)
        sync_entry.synced_at = aware_dt
        sync_entry.save()

    newhire_preprofiles_map = {o['id']: o for o in newhire_preprofiles}

    # Синкаем офферы
    offers = (
        Offer.unsafe
        .filter(
            newhire_id__in=newhire_preprofiles_map.keys(),
            status=OFFER_STATUSES.accepted,
        )
        .values_list('id', 'newhire_id', named=True)
    )
    for offer in offers:
        data = newhire_preprofiles_map[offer.newhire_id]
        try:
            if offer.id != data.get('femida_offer_id'):
                raise Exception
            OfferCtl.update_from_newhire_with_lock(offer.id, data)
        except Exception:
            sync_newhire_offer_task.delay(offer.id)

    # Синкаем препрофайлы
    preprofiles = (
        Preprofile.objects
        .filter(
            id__in=newhire_preprofiles_map.keys(),
            is_saved=True,
        )
    )
    for preprofile in preprofiles:
        data = newhire_preprofiles_map[preprofile.id]
        try:
            ctl = PreprofileCtl(preprofile)
            ctl.update_from_newhire(data)
        except Exception:
            sync_newhire_preprofile_task.delay(preprofile.id)


@app.autoretry_task(max_retries=5)
def push_offer_bank_details(offer_id, bank_details):
    offer = Offer.unsafe.get(id=offer_id)
    return OebsAPI.push_bank_details(offer.username, offer.full_name, **bank_details)


@app.autoretry_task(max_retries=3)
def offer_confirm_by_current_team_task(offer_id):
    from intranet.femida.src.offers.controllers import offer_confirm_by_current_team

    offer = Offer.unsafe.get(id=offer_id)
    issue = get_issue(offer.startrek_salary_key)
    offer_confirm_by_current_team(offer, issue)


@app.autoretry_task(max_retries=3)
def offer_approve_by_issue_task(offer_id):
    from intranet.femida.src.offers.controllers import offer_approve_by_issue
    offer = Offer.unsafe.get(id=offer_id)
    offer_approve_by_issue(offer)


@app.autoretry_task(max_retries=3)
def offer_confirm_by_issue_task(offer_id):
    from intranet.femida.src.offers.controllers import offer_confirm_by_issue

    offer = Offer.unsafe.get(id=offer_id)
    issue = get_issue(offer.vacancy.startrek_key)
    offer_confirm_by_issue(offer, issue)


@app.autoretry_task(max_retries=3)
def offer_decline_by_issue_task(offer_id):
    """
    Отклонение оффера через JOB-тикет (оффер не согласован).
    На данный момент таск актуален только для офферов по автонайму.
    Сам по себе таск не делает почти ничего – только проверяет,
    что оффер действительно не согласован, и отправляет об этом сигнал.
    """
    offer = (
        Offer.unsafe
        .select_related('vacancy')
        .get(id=offer_id)
    )
    assert offer.is_autohire

    issue = get_issue(offer.startrek_job_key)
    assert issue.status.key == StatusEnum.closed and issue.resolution.key == ResolutionEnum.wont_fix

    offer_unapproved.send(Offer, offer=offer)


@app.autoretry_task(max_retries=3)
def send_internal_offer_accept_notification(offer_id):
    offer = InternalOffer.unsafe.get(id=offer_id)
    OfferInternalAcceptedNotification(offer).send()
    OfferAcceptedNotification(offer).send()


@app.task(max_retries=3)
def check_oebs_login_task(offer_id):
    offer = (
        Offer.unsafe
        .select_related('org')
        .get(id=offer_id)
    )
    try:
        check_oebs_login(offer)
    except Exception as exc:
        retries = check_oebs_login_task.request.retries
        if retries == check_oebs_login_task.max_retries:
            comment = 'Не удалось проверить логин кандидата в OEBS.'
            add_issue_comment_task.delay(offer.startrek_hr_key, comment)
            return
        check_oebs_login_task.retry(
            countdown=get_retry_countdown(retries),
            exc=exc,
        )


@app.autoretry_task(max_retries=3)
@locked_task
def handle_expired_doc_requests_task():
    from intranet.femida.src.offers.controllers import create_oebs_person_and_assignment

    offers = (
        Offer.unsafe
        .filter(
            status=OFFER_STATUSES.accepted,
            # Ограничение по времени применяется только к перезапросам
            docs_processing_status=OFFER_DOCS_PROCESSING_STATUSES.need_information,
            # Просроченные ссылки предварительно удаляются в clean_expired_offer_links
            link__isnull=True,
        )
        .select_related(
            'payment_currency',
            'profile',
            'vacancy__instead_of',
            'position',
            'candidate',
        )
        .prefetch_related(
            get_candidate_alive_verifications_prefetch(),
        )
    )
    for offer in offers:
        with transaction.atomic():
            offer.docs_processing_status = OFFER_DOCS_PROCESSING_STATUSES.finished
            offer.docs_processing_resolution = OFFER_DOCS_PROCESSING_RESOLUTIONS.expired
            offer.save(update_fields=('docs_processing_status', 'docs_processing_resolution'))

            create_oebs_person_and_assignment(offer=offer, change_hr_issue_status=False)
            notify_hr_about_docs_processing_problem(offer)


@app.autoretry_task(max_retries=5)
def link_offer_issues_task(offer_id, attr1, attr2, relationship='relates'):
    """
    Связывает тикеты offer.<attr1> и offer.<attr2>.
    Ретраится, если номера тикетов ещё не сохранены в оффере.
    """
    offer = Offer.unsafe.get(id=offer_id)
    key1 = getattr(offer, attr1)
    key2 = getattr(offer, attr2)
    assert key1, f'{attr1} is not defined in offer {offer_id}'
    assert key2, f'{attr2} is not defined in offer {offer_id}'

    issue = get_issue(key1)
    try:
        issue.links.create(relationship, key2)
    except BackendError as exc:
        # Считаем, что тикеты уже связаны, раз Трекер отдаёт 422,
        # в противном случае рейзим дальше
        if exc.status_code != 422:
            raise


@app.autoretry_task(max_retries=3)
@transaction.atomic
def generate_offer_pdf_task(offer_id):
    from intranet.femida.src.offers.controllers import generate_offer_pdf

    if not waffle.switch_is_active('enable_offer_pdf_generation'):
        return
    offer = (
        Offer.unsafe
        .filter(
            id=offer_id,
            status__in=(
                OFFER_STATUSES.ready_for_approval,
                OFFER_STATUSES.on_rotation_approval,
                OFFER_STATUSES.on_approval,
                OFFER_STATUSES.sent,
            ),
        )
        .first()
    )
    if not offer:
        return
    attachments = offer.offer_attachments.filter(type=OFFER_ATTACHMENT_TYPES.offer_pdf)
    attachments.delete()
    is_offer_pdf_needed = (
        offer.employee_type in EXTERNAL_EMPLOYEE_TYPES
        and offer.vacancy.type in (VACANCY_TYPES.new, VACANCY_TYPES.replacement)
        and offer.office and offer.office.city.country.code == 'RU'
        and offer.grade and offer.grade <= 18
        and offer.payment_type == PAYMENT_TYPES.monthly
        and offer.payment_currency.code == 'RUB'
    )
    if is_offer_pdf_needed:
        generate_offer_pdf(offer)


@app.autoretry_task(max_retries=3)
def generate_candidate_offer_pdf_task(candidate_id):
    offers = (
        Offer.unsafe.alive()
        .filter(candidate_id=candidate_id)
        .values_list('id', flat=True)
    )
    for offer in offers:
        generate_offer_pdf_task.delay(offer)


# Препрофайлы


@app.autoretry_task(max_retries=3)
def accept_preprofile_task(preprofile_id):
    preprofile = Preprofile.objects.get(id=preprofile_id)
    data = NewhirePreprofileSerializer(preprofile).data
    result, status_code = NewhireAPI.accept_preprofile(
        preprofile_id=preprofile_id,
        data=data,
    )
    if status_code != 200:
        raise NewhireError(
            f'Newhire: accept prepofile {preprofile_id} failed: {status_code}, {result}'
        )
    preprofile.is_saved = True
    preprofile.save()


@app.autoretry_task(max_retries=5)
def create_preprofile_hr_issue_task(preprofile_id, resubmit=False, has_bank_details=True):
    preprofile = Preprofile.objects.get(id=preprofile_id)
    if not resubmit and preprofile.startrek_hr_key:
        logger.warning(
            'HR issue was already created for preprofile %s: %s',
            preprofile_id,
            preprofile.startrek_hr_key,
        )
        return
    create_preprofile_hr_issue(preprofile, resubmit, has_bank_details)
    if not resubmit:
        # Сразу же после создания двигаем HR-тикет в статус "новый"
        if preprofile.startrek_hr_key:
            operation = IssueTransitionOperation(preprofile.startrek_hr_key)
            operation.delay(TransitionEnum.new)


@app.autoretry_task(max_retries=3)
def update_newhire_hr_issue(offer_id: id):
    from intranet.femida.src.offers.controllers import OfferCtl
    ctl = OfferCtl(Offer.unsafe.get(id=offer_id))
    assert ctl.instance.startrek_hr_key is not None

    newhire_data = ctl._newhire_raw_data | {
        'hr_ticket': ctl.instance.startrek_hr_key,
        'department': ctl.instance.department.url,
    }

    return NewhireAPI.update_preprofile(ctl.instance.newhire_id, newhire_data)


@app.autoretry_task(max_retries=5)
def create_preprofile_eds_issue_task(preprofile_id):
    preprofile = Preprofile.objects.get(id=preprofile_id)
    if not preprofile.is_eds_phone_verified:
        return
    if preprofile.startrek_eds_key:
        logger.warning(
            'EDS issue was already created for preprofile %s: %s',
            preprofile_id,
            preprofile.startrek_eds_key,
        )
        return
    create_preprofile_eds_issue(preprofile)


@app.autoretry_task(max_retries=3)
def update_preprofile_issues_task(preprofile_id):
    preprofile = Preprofile.objects.get(id=preprofile_id)
    update_preprofile_issues(preprofile)


@app.autoretry_task(max_retries=5)
def push_preprofile_bank_details(preprofile_id, bank_details):
    from intranet.femida.src.offers.controllers import RemotePreprofile
    preprofile = Preprofile.objects.get(id=preprofile_id)
    remote_preprofile = RemotePreprofile(preprofile)
    return OebsAPI.push_bank_details(
        username=remote_preprofile.username,
        full_name=remote_preprofile.full_name,
        **bank_details
    )


@app.autoretry_task(max_retries=5)
def update_newhire_employee_type_task(offer_id, employee_type):
    from intranet.femida.src.offers.controllers import OfferCtl
    offer = Offer.unsafe.get(id=offer_id)
    if offer.newhire_id is None:
        return
    ctl = OfferCtl(offer)
    ctl.update_in_newhire({'employee_type': employee_type})


@app.task
@locked_task
def update_offer_current_dismissed_to_former_task():
    """
    FEMIDA-6317 проверка наличия офферов с типом сотрудника
    "Действующий Внешний Консультант", который уже уволен,
    замена типа сотрудника в оффере и анкете в Наниматоре на "бывший"
    """
    is_user_dismissed_subquery = Exists(
        User.objects
        .filter(
            username=OuterRef('username'),
            is_dismissed=True,
        )
    )
    offer_qs = (
        Offer.unsafe
        .annotate(is_user_dismissed=is_user_dismissed_subquery)
        .filter(
            status__in=(OFFER_STATUSES.sent, OFFER_STATUSES.accepted),
            employee_type=EMPLOYEE_TYPES.current,
            is_user_dismissed=True,
        )
    )

    # Оффер отправлен, но не принят, или принят, но не отправлен в наниматор
    offer_qs.filter(newhire_id__isnull=True).update(employee_type=EMPLOYEE_TYPES.former)

    # Оффер принят, заведена анкета в наниматоре с типом "текущий"
    for offer in offer_qs.filter(newhire_id__isnull=False):
        update_newhire_employee_type_task.delay(offer.id, EMPLOYEE_TYPES.former)
