import logging
import waffle

from datetime import datetime
from typing import Dict

from constance import config
from django.conf import settings
from django.template import loader
from django.utils.functional import cached_property

from intranet.femida.src.candidates.choices import (
    REFERENCE_EVENTS,
    ROTATION_EVENTS,
    ROTATION_EVENTS_TRANSLATIONS,
    VERIFICATION_RESOLUTIONS,
    VERIFICATION_STATUSES,
)
from intranet.femida.src.candidates.considerations.controllers import (
    update_consideration_extended_status,
)
from intranet.femida.src.candidates.deduplication.exceptions import HasDuplicatesError
from intranet.femida.src.candidates.helpers import (
    CandidatePolyglot,
    get_active_duplication_case_ids,
)
from intranet.femida.src.candidates.tasks import (
    reopen_reference_issue_task,
    resolve_candidate_reference_issue_task,
    send_candidate_reference_event_task,
)
from intranet.femida.src.candidates.verifications.helpers import is_verification_required
from intranet.femida.src.core.switches import TemporarySwitch
from intranet.femida.src.core.workflow import Action, Workflow, WorkflowError
from intranet.femida.src.interviews.choices import APPLICATION_RESOLUTIONS, APPLICATION_STATUSES
from intranet.femida.src.interviews.helpers import is_unfinished_interview_exists
from intranet.femida.src.monitoring.checkpoints import irreversible_checkpoints
from intranet.femida.src.notifications.offers import (
    OfferAcceptedNotification,
    OfferDeletedNotification,
    OfferRejectedNotification,
    OfferSentForApprovalNotification,
    OfferSentNotification,
    send_offer_to_candidate,
)
from intranet.femida.src.oebs.api import (
    BudgetPositionError,
    get_budget_position,
    reset_budget_position_status,
)
from intranet.femida.src.oebs.choices import (
    BUDGET_POSITION_PAYMENT_SYSTEMS,
    BUDGET_POSITION_STATUSES,
)
from intranet.femida.src.oebs.models import OEBSLog
from intranet.femida.src.oebs.tasks import remove_oebs_person_task, update_oebs_assignment_task
from intranet.femida.src.offers.bp_registry.serializers import BPRegistryOfferRejectionSerializer
from intranet.femida.src.offers.choices import (
    OFFER_DOCS_PROCESSING_STATUSES,
    OFFER_NEWHIRE_STATUSES,
    OFFER_STATUSES,
    REJECTION_SIDES,
    OPEN_OFFER_STATUSES,
    EMPLOYEE_TYPES,
)
from intranet.femida.src.offers.controllers import OfferCtl, OfferSchemesController
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.offers.oebs.serializers import OfferBPLogSerializer
from intranet.femida.src.offers.signals import offer_accepted
from intranet.femida.src.offers.startrek.choices import REJECTION_REASONS_TRANSLATIONS
from intranet.femida.src.offers.startrek.issues import execute_adaptation_issue_transition
from intranet.femida.src.offers.startrek.serializers import (
    FIELDS_TO_CLEAN_ON_REJECT,
    JobIssueFieldsBaseSerializer,
)
from intranet.femida.src.offers.tasks import (
    create_adaptation_issue_task,
    create_bootcamp_issue_task,
    create_bonus_issue_task,
    create_eds_issue_task,
    create_hr_issue_task,
    create_relocation_issue_task,
    create_salary_issue_task,
    create_signup_issue_task,
    push_offer_bank_details,
    save_in_newhire_task,
    send_internal_offer_accept_notification,
    sync_newhire_offer_task,
    update_newhire_hr_issue,
)
from intranet.femida.src.publications.controllers import (
    archive_vacancy_publications,
    unarchive_vacancy_publications,
)
from intranet.femida.src.staff.bp_registry import (
    BPRegistryAPI,
    BPRegistryError,
    BPRegistryUnrecoverableError,
)
from intranet.femida.src.staff.helpers import get_hr_analyst_groups
from intranet.femida.src.startrek.operations import (
    IssueCommentOperation,
    IssueTransitionOperation,
    IssueUpdateOperation,
)
from intranet.femida.src.startrek.tasks import add_issue_comment_task
from intranet.femida.src.startrek.utils import (
    get_issue,
    IssueInvalidStatus,
    KnownStartrekError,
    ResolutionEnum,
    StartrekError,
    StatusEnum,
    TransitionDoesNotExist,
    TransitionEnum,
    TransitionFailed,
)
from intranet.femida.src.utils.newhire import NewhireError
from intranet.femida.src.utils.ok import create_tracker_approvement_comment
from intranet.femida.src.utils.strings import fetch_comma_separated_integers
from intranet.femida.src.utils.switches import is_newhire_rotation_enabled
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES, VACANCY_TYPES
from intranet.femida.src.vacancies.startrek.memberships import IssueMembership
from intranet.femida.src.yt.tasks import save_offer_data_in_yt


logger = logging.getLogger(__name__)


class OfferAction(Action):

    validators = []

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.ctl = OfferCtl(self.instance)
        self.validator_errors = []

    def add_irreversible_checkpoint(self, name='', **kwargs):
        irreversible_checkpoints.add(
            name=name,
            action_class=self.__module__ + '.' + self.__class__.__name__,
            instance_id=self.instance.id,
            username=self.user.username if self.user else None,
            **kwargs
        )

    def check_validators(self):
        for validator in self.validators:
            try:
                validator()
            except WorkflowError as exc:
                self.validator_errors.append(exc)
        return True if not self.validator_errors else False

    def has_permission(self):
        return (
            self.user in self.instance.vacancy.recruiters
            or self.user.is_recruiting_manager
        )

    def is_available(self):
        return self.is_visible() and self.check_validators()

    def _get_bp_date(self):
        return self.instance.join_at.strftime('%Y-%m-%d') if self.instance.join_at else None

    def _log_bp_data(self):
        OEBSLog.log(
            related_object=self.instance,
            related_data=OfferBPLogSerializer(self.instance).data,
            oebs_data=self.bp_data,
            action=self.__class__.__name__,
        )

    def _get_rotation_keys(self):
        rotation = self.instance.application.consideration.rotation
        if not rotation:
            return tuple()
        return (
            rotation.startrek_rotation_key,
            rotation.startrek_myrotation_key,
        )

    def _validate_interviews(self):
        if is_unfinished_interview_exists(candidate=self.instance.candidate):
            raise WorkflowError('candidate_has_unfinished_interviews')

    @cached_property
    def bp_id(self):
        return self.instance.vacancy.budget_position_id

    @cached_property
    def bp_data(self):
        if not self.bp_id:
            return {}
        try:
            return get_budget_position(self.bp_id, self._get_bp_date())
        except BudgetPositionError as ex:
            raise WorkflowError(ex.message)

    @property
    def bp_status(self):
        return self.bp_data.get('status')

    @cached_property
    def newhire_data(self):
        return self.ctl.newhire_data

    @property
    def newhire_status(self):
        return self.newhire_data['newhire_status']

    def _get_issue(self, key):
        try:
            return get_issue(key)
        except StartrekError as exc:
            self.raise_error(exc.message)

    @cached_property
    def job_issue(self):
        return self._get_issue(self.instance.vacancy.startrek_key)

    def _get_startrek_status(self, key):
        if not key:
            return None
        try:
            issue = get_issue(key)
            return issue.status.key
        except StartrekError:
            return None

    @property
    def startrek_job_status(self):
        return self._get_startrek_status(self.instance.vacancy.startrek_key)

    @property
    def startrek_hr_status(self):
        return self._get_startrek_status(self.instance.startrek_hr_key)

    def _change_issue_status(self, key, transition, **fields):
        error_message = None
        operation = IssueTransitionOperation(key)
        try:
            operation(transition, **fields)
        except KnownStartrekError as exc:
            error_message = exc.message
        except StartrekError:
            error_message = TransitionFailed.message
        else:
            self.add_irreversible_checkpoint(
                name='change_issue_status',
                key=key,
                transition=transition,
            )

        if error_message:
            operation.delay(transition, **fields)
            self.raise_error(error_message, fail_silently=True)

    def change_job_issue_status(self, transition, delay=False, **fields):
        if waffle.switch_is_active('ignore_job_issue_workflow'):
            return
        operation = IssueTransitionOperation(self.instance.vacancy.startrek_key)
        fields['transition'] = transition

        if delay:
            operation.delay(**fields)
            return

        error_message = None
        retry = True
        try:
            operation(**fields)
        except TransitionDoesNotExist as exc:
            error_message = exc.message
            # Если перехода не существует, ретраить смысла нет
            retry = False
        except KnownStartrekError as exc:
            error_message = exc.message
        except StartrekError:
            error_message = TransitionFailed.message

        # Если мы перевели статус тикета, либо была попытка, и мы собираемся ретраить,
        # помечаем данное событие точкой невозврата
        if error_message is None or retry:
            self.add_irreversible_checkpoint(
                name='change_issue_status',
                key=self.instance.vacancy.startrek_key,
                transition=transition,
            )

        if error_message:
            if retry:
                operation.delay(**fields)
            self.raise_error(error_message, fail_silently=retry)

    def change_hr_issue_status(self, transition, **fields):
        return self._change_issue_status(
            key=self.instance.startrek_hr_key,
            transition=transition,
            **fields
        )

    def reset_bp_status(self):
        # FIXME: Cвитч нужен на случай, если окажется,
        #  что мы не любую БП технически можем сбросить.
        #  Если тут проблем не будет, is_reset_allowed можно целиком удалить
        is_reset_allowed = (
            waffle.switch_is_active('enable_bp_reset_without_status_check')
            or self.bp_status == BUDGET_POSITION_STATUSES.offer
        )
        return (
            is_reset_allowed
            and reset_budget_position_status(self.instance.vacancy.budget_position_id)
        )

    def send_reference_event(self, event):
        send_candidate_reference_event_task.delay(self.instance.candidate_id, event)


class UpdateAction(OfferAction):
    """
    Редактирование оффера
    """
    valid_statuses = (
        OFFER_STATUSES.draft,
        OFFER_STATUSES.ready_for_approval,
        OFFER_STATUSES.on_rotation_approval,
        OFFER_STATUSES.on_approval,
        OFFER_STATUSES.sent,
    )

    def perform(self, **params):
        if self.instance.status == OFFER_STATUSES.draft:
            params['status'] = OFFER_STATUSES.ready_for_approval

        if waffle.switch_is_active('enable_scheme_requests'):
            offer_schemes_ctl = OfferSchemesController(self.instance, params)
            if offer_schemes_ctl.new_schemes_should_be_requested():
                params['schemes_data'] = offer_schemes_ctl.request_schemes_from_staff()

        self.ctl.update(params)
        return self.instance


class ApproveAction(OfferAction):

    valid_statuses = (
        OFFER_STATUSES.ready_for_approval,
    )

    def _get_bp_date(self):
        # Отправляем запрос в OEBS без даты
        # https://st.yandex-team.ru/FEMIDA-2754
        return None

    def _get_ok_approvement(self):
        """
        Генерит коммент с предзаполненной формой согласования в ОК
        """
        author = self.job_issue.recruitmentPartner
        issue_membership = IssueMembership.from_offer(self.instance)
        groups = get_hr_analyst_groups(self.instance.department)
        groups.append(settings.OK_REC_OFFER_APPROVERS_GROUP_URL)
        return create_tracker_approvement_comment(
            issue_key=self.job_issue.key,
            author=author.login if author else None,
            approvers=[i.username for i in issue_membership.chiefs],
            text='Привет!\nПодтверждаешь оффер?\nСпасибо!',
            groups=groups,
        )

    def _validate_duplicates(self):
        if not waffle.switch_is_active('enable_has_duplicates_error'):
            return
        duplication_case_ids = get_active_duplication_case_ids(self.instance.candidate_id)
        if duplication_case_ids:
            raise HasDuplicatesError(duplication_case_ids)

    @property
    def validators(self):
        return [
            self._validate_interviews,
            self._validate_duplicates,
        ]

    def perform(self, **params):
        if self.bp_status not in (BUDGET_POSITION_STATUSES.offer, BUDGET_POSITION_STATUSES.vacancy):
            raise WorkflowError('budget_position_invalid_status')

        self.ctl.update(params)

        if self.instance.is_external:
            self.ctl.create_link()

        self._log_bp_data()
        self.instance.status = OFFER_STATUSES.on_approval
        self.instance.save()

        update_consideration_extended_status(self.instance.application.consideration)

        bp_status_changed = self.reset_bp_status()

        context = self.get_context()
        context['bp_status_changed'] = bp_status_changed
        if self.instance.vacancy.type != VACANCY_TYPES.autohire:
            context['ok_approvement'] = self._get_ok_approvement()

        comment = loader.render_to_string('startrek/offers/approve.wiki', context)

        # Отправляем на согласование
        fields = JobIssueFieldsBaseSerializer.serialize(self.instance)
        self.change_job_issue_status(
            transition=TransitionEnum.agree_offer,
            delay=self.instance.is_autohire,
            comment=comment,
            **fields
        )

        notification = OfferSentForApprovalNotification(self.instance, self.user)
        notification.send()

        if self.ctl.schemes_data and self.ctl.schemes_data.is_reward_category_changed:
            category_changed_comment = loader.render_to_string(
                'startrek/offers/category-changed.wiki',
                context,
            )
            add_issue_comment_task.delay(
                self.instance.vacancy.startrek_key,
                category_changed_comment,
            )

        return self.instance


class ApproveByCurrentTeamAction(ApproveAction):
    """
    Согласование внутреннего оффера нынешней командой (отдающей стороной).
    Экшн не виден для фронта, но при попытке согласовать
    внутренний оффер, всегда запускается он, а не ApproveAction
    """
    def is_visible(self):
        return False

    def is_available(self):
        return (
            # Проверяем видимость по суперклассу, потому что
            # конкретно этот экшн всегда невидим для фронта
            super().is_visible()
            and self.instance.is_internal
            and self.check_validators()
        )

    def perform(self, **params):
        if self.bp_status not in (BUDGET_POSITION_STATUSES.offer, BUDGET_POSITION_STATUSES.vacancy):
            raise WorkflowError('budget_position_invalid_status')

        self.ctl.update(params)
        self._log_bp_data()
        self.instance.status = OFFER_STATUSES.on_rotation_approval
        self.instance.save()

        update_consideration_extended_status(self.instance.application.consideration)

        create_salary_issue_task.delay(self.instance.id, self.user.id)

        # Добавляем комментарий в тикеты ROTATION и MYROTATION
        context = {'creator': self.instance.creator}
        rotation = self.instance.application.consideration.rotation
        if rotation:
            add_issue_comment_task.delay(
                keys=rotation.startrek_myrotation_key,
                text=loader.render_to_string(
                    template_name='startrek/rotations/myrotation-offer-on-approval.wiki',
                    context=context,
                ),
            )
            add_issue_comment_task.delay(
                keys=rotation.startrek_rotation_key,
                text=ROTATION_EVENTS_TRANSLATIONS[ROTATION_EVENTS.rotation_on_approval],
            )
        return self.instance


class ApproveByFutureTeamAction(ApproveAction):
    """
    Согласование внутреннего оффера новой командой (принимающей стороной).
    Экшн не виден для фронта и для него нет ручки.
    Но он доступен без проверки прав пользователя,
    потому что выполняется без авторизации, при срабатывании триггера в Трекере.
    """
    valid_statuses = (
        OFFER_STATUSES.on_rotation_approval,
    )

    def _validate_duplicates(self):
        # Не проверяем дубли на этом шаге согласований,
        # потому что это происходит в фоне через триггеры и реального
        # отв-ного пользователя на этом шаге нет
        pass

    def is_visible(self):
        return False

    def is_available(self):
        return (
            # Не проверяем has_permissions, потому что экшн
            # выполняется без авторизации
            self.instance.is_internal
            and self.is_status_correct()
            and self.check_validators()
        )

    def add_irreversible_checkpoint(self, name='', **kwargs):
        # Для этого экшна это неактуально + невозможно,
        # т.к. он запускается не из ручки, а из таска
        pass


class ReapproveAction(OfferAction):
    """
    Повторное согласование.
    Возвращение оффера в работу.
    """
    valid_statuses = (
        OFFER_STATUSES.on_approval,
        OFFER_STATUSES.sent,
    )

    def is_status_correct(self):
        # Пересогласовавывать можно только уже отправленные внешние офферы,
        # либо внешние офферы на согласовании, если JOB-тикет ещё в работе
        return (
            super().is_status_correct()
            and self.instance.is_external
            and (
                self.instance.status == OFFER_STATUSES.sent
                or self.startrek_job_status == StatusEnum.in_progress
            )
        )

    def perform(self, **params):
        old_status = self.instance.status
        self.instance.status = OFFER_STATUSES.ready_for_approval
        self.instance.save()
        self.ctl.reset_offer_text()

        update_consideration_extended_status(self.instance.application.consideration)

        bp_status_changed = self.reset_bp_status()

        context = self.get_context()
        context['bp_status_changed'] = bp_status_changed
        comment_template_name = 'startrek/offers/reapprove.wiki'

        if old_status == OFFER_STATUSES.sent:
            comment = loader.render_to_string(comment_template_name, context)
            self.change_job_issue_status(
                transition=TransitionEnum.offer_declined,
                comment=comment,
            )

        return self.instance


class SendAction(OfferAction):
    """
    Отправка оффера кандидату.
    В данный момент это просто пометка, что оффер отправлен.
    """
    valid_statuses = (
        OFFER_STATUSES.on_approval,
    )

    def has_permission(self):
        return (
            self.instance.is_external
            and super().has_permission()
        )

    @property
    def validators(self):
        return [
            self._validate_bp_status,
            self._validate_bp_salary,
            self._validate_bp_available_funds,
            self._validate_bp_join_at,
            self._validate_startrek_status,
            self._validate_interviews,
            self._validate_verification_status,
        ]

    def _bp_has_fixed_pay_system(self):
        return self.bp_data.get('paySystem') in (
            BUDGET_POSITION_PAYMENT_SYSTEMS.fixed_salary,
            BUDGET_POSITION_PAYMENT_SYSTEMS.jobprice,
        )

    def _validate_bp_status(self):
        if self.bp_status != BUDGET_POSITION_STATUSES.offer:
            raise WorkflowError('budget_position_invalid_status')

    def _validate_bp_salary(self):
        max_salary = (
            self._bp_has_fixed_pay_system()
            and self.bp_data.get('salary')
        )
        salary = self.instance.salary
        if max_salary and salary is not None and salary > max_salary:
            raise WorkflowError('salary_max_value')

    def _validate_bp_available_funds(self):
        available_funds = (
            self._bp_has_fixed_pay_system()
            and self.bp_data.get('availableFunds')
        )
        salary = self.instance.salary
        if available_funds and salary is not None and salary > available_funds:
            raise WorkflowError('available_funds_max_value')

    def _validate_bp_join_at(self):
        bp_join_at = self.bp_data.get('hiringDate')
        if bp_join_at:
            min_date = datetime.strptime(bp_join_at, '%Y-%m-%d').date()
            if self.instance.join_at < min_date:
                raise WorkflowError('join_at_min_value')

    def _validate_startrek_status(self):
        if waffle.switch_is_active('ignore_job_issue_workflow'):
            return
        if self.startrek_job_status != StatusEnum.resolved:
            raise WorkflowError(IssueInvalidStatus.message)

    def _validate_verification_status(self):
        if not waffle.switch_is_active('enable_verification_check_on_offer_sending'):
            return

        # Ротируемые и стажеры проходили проверку КИ при найме, но ее может не быть в Фемиде
        if self.instance.is_internal:
            return

        if not is_verification_required(self.instance):
            return

        last_verification = self.instance.candidate.verifications.alive().first()
        if not last_verification:
            raise WorkflowError('no_verifications')
        if last_verification.status != VERIFICATION_STATUSES.closed:
            raise WorkflowError('verification_in_progress')
        if last_verification.resolution != VERIFICATION_RESOLUTIONS.hire:
            raise WorkflowError('verification_resolution_negative')

    def perform(self, sender=None, **params):
        self._log_bp_data()

        self.instance.status = OFFER_STATUSES.sent
        self.instance.save()

        update_consideration_extended_status(self.instance.application.consideration)

        self.ctl.activate_link()
        self.ctl.offer_text = params['offer_text']

        if params['receiver']:
            send_offer_to_candidate(
                offer_id=self.instance.id,
                subject=params['subject'],
                message=params['message'],
                offer_text=params['offer_text'],
                receiver=params['receiver'],
                sender=sender or self.user.email,
                attachments=params.get('attachments', []),
                bcc=params['bcc'],
            )

        context = self.get_context()
        context['offer_text'] = params['offer_text']
        context['is_custom_offer_text'] = self.ctl.has_custom_offer_text()

        operation = IssueCommentOperation(self.instance.vacancy.startrek_key)
        comment = loader.render_to_string('startrek/offers/sent.wiki', context)
        operation.delay(comment)

        notification = OfferSentNotification(self.instance, self.user)
        notification.send()

        return self.instance


class AcceptAction(OfferAction):
    """
    Экшн принятия внешнего оффера
    """
    def is_visible(self):
        return False

    def is_available(self):
        return True

    # TODO: функция очень длинная, может быть ее порефакторить?
    def perform(self, **params):
        comment = params.pop('comment', None)
        params.pop('is_agree', None)
        is_eds_needed = params.pop('is_eds_needed', None)

        # Побочный эффект: поля для языков в params будут изменены.
        self._mark_candidate_and_profile_with_languages(params)

        if self.instance.docs_processing_status == OFFER_DOCS_PROCESSING_STATUSES.need_information:
            return self._process_additional_documents(params)

        bank_details = params.pop('bank_details')
        is_bank_details_needed = self.instance.is_bank_details_needed

        # Банковские реквизиты не нужны, либо указаны все нужные
        has_bank_details = not is_bank_details_needed or all(bank_details.values())

        self.ctl.get_or_create_profile()
        offer_data = {
            'full_name': params.pop('full_name'),
            'join_at': params.pop('join_at'),
            'profile': params.pop('profile', {}),
            'status': OFFER_STATUSES.accepted,
        }
        if 'username' in params:
            offer_data['username'] = params.pop('username')

        # TODO: https://st.yandex-team.ru/FEMIDA-7312
        # Тут вообще немного странная логика, возможно ее стоит переосмыслить:
        # Ферифицирован один телефон, в форме может приехать другой, вместо ошибки
        # верификации просто молча ставим признак, что телефон не верифицирован
        offer_data['is_eds_phone_verified'] = (
            # None не должно быть, но при локальном тестировании тут он
            # может быть. На всякий случай проверим.
            # Логика простая, если в этом поле None, то считаем, что
            # это эквивалентно False, т.е. номер телефона не был заранее верифицирован
            self.instance.is_eds_phone_verified is not None
            and self.instance.is_eds_phone_verified
            and is_eds_needed is not None
            and is_eds_needed
            and self.instance.eds_phone == offer_data['profile'].get('phone')
        )

        self.ctl.update(offer_data)

        if comment:
            self.ctl.add_comment(comment)

        self.change_job_issue_status(
            transition=TransitionEnum.offer_accepted,
            delay=True,
            start=self.instance.join_at.strftime('%Y-%m-%d'),
        )

        update_consideration_extended_status(self.instance.application.consideration)

        application = self.instance.application
        vacancy = self.instance.vacancy

        if waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_PUBLICATIONS_ARCHIVE):
            vacancy.is_published = False
        vacancy.status = VACANCY_STATUSES.offer_accepted
        vacancy.save(update_fields=['modified', 'status', 'is_published'])

        application.status = APPLICATION_STATUSES.closed
        application.resolution = APPLICATION_RESOLUTIONS.offer_accepted
        application.save(update_fields=['modified', 'status', 'resolution'])

        self.ctl.remove_link()

        notification = OfferAcceptedNotification(self.instance)
        notification.send()

        tasks_chain = (
            save_in_newhire_task.si(self.instance.id)
            | create_hr_issue_task.si(self.instance.id, has_bank_details)
            | create_relocation_issue_task.si(self.instance.id)
            | create_signup_issue_task.si(self.instance.id)
            | create_bonus_issue_task.si(self.instance.id)
            | create_adaptation_issue_task.si(self.instance.id)
            | create_bootcamp_issue_task.si(self.instance.id)
            | create_eds_issue_task.si(self.instance.id)
            | update_newhire_hr_issue.si(self.instance.id)
        )

        if self.instance.is_autohire:
            tasks_chain |= sync_newhire_offer_task.si(self.instance.id)

        tasks_chain.delay()

        if is_bank_details_needed:
            push_offer_bank_details.delay(self.instance.id, bank_details)

        resolve_candidate_reference_issue_task.delay(
            candidate_id=self.instance.candidate_id,
            grade=self.instance.grade,
            legalEntity=self.instance.org.startrek_id,
            start=self.instance.join_at.strftime('%Y-%m-%d'),
        )

        offer_accepted.send(Offer, offer=self.instance)

        if waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_PUBLICATIONS_ARCHIVE):
            # Архивируем публикации вакансии
            archive_vacancy_publications(vacancy)

        return self.instance

    def _mark_candidate_and_profile_with_languages(self, params):
        """
        Размечаем кандидата языками.
        Побочный эффект: функция модифицирует params
        """
        if 'profile' in params:
            main_language = params['profile'].pop('main_language', None)
            spoken_languages = params['profile'].pop('spoken_languages', None)
            polyglot = CandidatePolyglot(self.instance.candidate)
            polyglot.update_known_languages(main_language=main_language, spoken_languages=spoken_languages)

            # Получаем актуальные данные по языкам кандидата
            # TODO: Здесь можно в будущем пооптимизировать, например,
            #       научить polyglot возвращать актуальный список языков,
            #       тогда ходить в базу не надо будет.
            params['profile']['main_language'] = self.instance.candidate.main_language
            params['profile']['spoken_languages'] = self.instance.candidate.spoken_languages

    def _process_additional_documents(self, params: Dict) -> Offer:
        """
        Отдельный набор действий при загрузке
        кандидатом недостающих документов
        """
        offer_data = {
            'docs_processing_status': OFFER_DOCS_PROCESSING_STATUSES.in_progress,
            'profile': params,
        }
        self.ctl.update(offer_data)
        self.ctl.remove_link()
        save_offer_data_in_yt.delay(self.instance.id)
        return self.instance


class AcceptInternalAction(OfferAction):
    """
    Экшн принятия внутреннего оффера
    """
    valid_statuses = (
        OFFER_STATUSES.on_approval,
    )

    def is_visible(self):
        return False

    def is_available(self):
        return self.instance.is_internal and self.is_status_correct()

    def perform(self, **params):
        self.instance.status = OFFER_STATUSES.accepted
        # Жёстко фиксируем, что это ротация внутри Яндекса
        self.instance.is_rotation_within_yandex = (
            self.instance.employee_type == EMPLOYEE_TYPES.rotation
            and waffle.switch_is_active('enable_rotation_adaptation')
            and self.instance.department.is_in_tree(settings.YANDEX_DEPARTMENT_ID)
            and self.instance.employee.department.is_in_tree(settings.YANDEX_DEPARTMENT_ID)
        )
        self.instance.save(update_fields=['status', 'is_rotation_within_yandex', 'modified'])

        update_consideration_extended_status(self.instance.application.consideration)

        application = self.instance.application
        vacancy = self.instance.vacancy

        if waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_PUBLICATIONS_ARCHIVE):
            vacancy.is_published = False
        vacancy.status = VACANCY_STATUSES.offer_accepted
        vacancy.save(update_fields=['modified', 'status', 'is_published'])

        application.status = APPLICATION_STATUSES.closed
        application.resolution = APPLICATION_RESOLUTIONS.rotated
        application.save()

        # Проставляем в JOB-тикете резолюцию "Перевод"
        operation = IssueUpdateOperation(self.instance.vacancy.startrek_key)
        operation.delay(resolution=ResolutionEnum.transfer)

        # Отправляем коммент в SALARY-тикет, что всё согласовано
        add_issue_comment_task.delay(
            keys=self.instance.startrek_salary_key,
            text=loader.render_to_string(
                template_name='startrek/offers/salary-accept-internal.wiki',
                context=self.get_context(),
            ),
        )

        # Переводим тикеты ROTATION и MYROTATION в состояние Решен
        for key in self._get_rotation_keys():
            IssueTransitionOperation(key).delay(
                transition=TransitionEnum.resolve,
                comment=ROTATION_EVENTS_TRANSLATIONS[ROTATION_EVENTS.rotation_approved],
                start=self.instance.join_at.strftime('%Y-%m-%d'),
            )

        tasks_chain = (
            save_in_newhire_task.si(self.instance.id)
            | create_relocation_issue_task.si(self.instance.id)
            | create_adaptation_issue_task.si(self.instance.id)
            | send_internal_offer_accept_notification.si(self.instance.id)
            | create_bootcamp_issue_task.si(self.instance.id)
        )
        if is_newhire_rotation_enabled():
            tasks_chain.delay()

        if waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_PUBLICATIONS_ARCHIVE):
            # Архивируем публикации вакансии
            archive_vacancy_publications(vacancy)

        return self.instance


class AcceptedOfferAction(OfferAction):
    """
    Базовый класс для всех экшнов
    принятого оффера, который уже
    сохранен в Наниматоре.
    """
    valid_statuses = (
        OFFER_STATUSES.accepted,
    )
    valid_newhire_statuses = None

    def is_newhire_status_correct(self):
        if self.valid_newhire_statuses is None:
            return True
        return self.newhire_status in self.valid_newhire_statuses

    def has_permission(self):
        return (
            super().has_permission()
            and self.instance.newhire_id is not None
            and self.is_newhire_status_correct()
        )

    def _update_in_newhire(self, data, sync=True):
        try:
            self.ctl.update_in_newhire(data, sync=sync)
        except NewhireError as exc:
            self.raise_error(exc.message)


class UpdateJoinAtAction(AcceptedOfferAction):
    """
    Изменение даты выхода
    """
    valid_newhire_statuses = (
        OFFER_NEWHIRE_STATUSES.new,
        OFFER_NEWHIRE_STATUSES.approved,
        OFFER_NEWHIRE_STATUSES.ready,
    )

    def perform(self, **params):
        self._update_in_newhire(params)
        return self.instance


class UpdateUsernameAction(AcceptedOfferAction):
    """
    Изменение логина
    """
    valid_newhire_statuses = (
        OFFER_NEWHIRE_STATUSES.new,
        OFFER_NEWHIRE_STATUSES.approved,
    )

    def has_permission(self):
        return (
            self.instance.is_external
            and super().has_permission()
        )

    def perform(self, **params):
        self._update_in_newhire(params)
        return self.instance


class UpdateDepartmentAction(AcceptedOfferAction):
    """
    Изменение подразделения
    """
    valid_newhire_statuses = (
        OFFER_NEWHIRE_STATUSES.new,
        OFFER_NEWHIRE_STATUSES.approved,
        OFFER_NEWHIRE_STATUSES.ready,
    )

    def perform(self, **params):
        # Не флудим, если ничего не поменялось
        if not params.get('department') or self.instance.department == params['department']:
            return self.instance

        context = self.get_context()
        context['old_department'] = self.instance.department
        context['department'] = params['department']

        self.ctl.update(params)
        self._update_in_newhire(params, sync=False)

        comment = loader.render_to_string('startrek/offers/department-changed.wiki', context)
        fields = (
            'startrek_hr_key',
            'startrek_adaptation_key',
            'startrek_bootcamp_key',
            'startrek_relocation_key',
            'startrek_signup_key',
            'startrek_bonus_key',
            'startrek_eds_key',
        )
        for field in fields:
            key = getattr(self.instance, field)
            if key:
                IssueCommentOperation(key).delay(comment)

        job_comment = loader.render_to_string(
            template_name='startrek/offers/department-changed-job.wiki',
            context=context,
        )
        operation = IssueUpdateOperation(self.instance.vacancy.startrek_key)
        operation.delay(
            comment=job_comment,
            tags={
                'add': [settings.STARTREK_JOB_DEPARTMENT_CHANGE_TAG],
            },
        )

        # Обновляем подразделение в Я.Найме
        if self.instance.oebs_person_id:
            update_oebs_assignment_task.delay(self.instance.id)

        return self.instance


class DeleteActionBase(OfferAction):

    def perform(self, **params):
        old_status = self.instance.status
        self.instance.status = OFFER_STATUSES.deleted
        self.instance.save()

        self.ctl.remove_link()

        update_consideration_extended_status(self.instance.application.consideration)

        application = self.instance.application
        vacancy = self.instance.vacancy

        vacancy.status = VACANCY_STATUSES.in_progress
        vacancy.save()

        application.status = APPLICATION_STATUSES.closed
        application.resolution = APPLICATION_RESOLUTIONS.no_offer
        application.save()

        notification = OfferDeletedNotification(self.instance, self.user)
        notification.send()

        if old_status == OFFER_STATUSES.on_approval:
            self.send_reference_event(REFERENCE_EVENTS.offer_deleted)

        return self.instance


class DeleteAction(DeleteActionBase):
    """
    Удаление не выставленного оффера
    """
    valid_statuses = (
        OFFER_STATUSES.draft,
        OFFER_STATUSES.ready_for_approval,
        OFFER_STATUSES.on_rotation_approval,
        OFFER_STATUSES.on_approval,
    )

    @property
    def is_external_on_approval(self):
        return self.instance.is_external and self.instance.status == OFFER_STATUSES.on_approval

    @property
    def validators(self):
        return [self._validate_job_issue_status]

    def _validate_job_issue_status(self):
        # Проверка статуса JOB-тикета актуальна только для внеш.офферов,
        # которые находятся на согласовании
        if not self.is_external_on_approval:
            return
        if self.startrek_job_status not in (StatusEnum.in_progress, StatusEnum.resolved):
            raise WorkflowError('invalid_job_issue_status')

    def perform(self, **params):
        is_external_on_approval = self.is_external_on_approval
        super().perform(**params)

        if self.instance.status != OFFER_STATUSES.draft:
            rejection_data = {
                'comment': params['comment'],
                'rejection_side': REJECTION_SIDES.recruiter,
            }
            offer_rejection = self.ctl.reject(data=rejection_data, initiator=self.user)

        # Откатываем БП и тикет JOB
        if is_external_on_approval:
            bp_status_changed = self.reset_bp_status()
            context = self.get_context()
            context['offer_rejection'] = offer_rejection
            context['bp_status_changed'] = bp_status_changed
            self._rollback_job_issue(context)

        # Отправляем коммент в тикет SALARY
        if self.instance.startrek_salary_key:
            context = self.get_context()
            context['offer_rejection'] = offer_rejection
            comment = loader.render_to_string('startrek/offers/rejected.wiki', context)
            add_issue_comment_task.delay(self.instance.startrek_salary_key, comment)

        return self.instance

    def _rollback_job_issue(self, context):
        comment = loader.render_to_string('startrek/offers/deleted-job.wiki', context)
        fields = {k: None for k in FIELDS_TO_CLEAN_ON_REJECT}
        if self.startrek_job_status == StatusEnum.resolved:
            self.change_job_issue_status(
                transition=TransitionEnum.offer_declined,
                comment=comment,
                **fields
            )
        else:
            operation = IssueUpdateOperation(self.instance.vacancy.startrek_key)
            operation.delay(
                comment=comment,
                **fields
            )


# TODO: FEMIDA-5651 Рефакторинг Delete и Reject экшенов
class DeleteForHireOrderAction(DeleteActionBase):

    valid_statuses = OPEN_OFFER_STATUSES

    def perform(self, **params):
        super().perform(**params)

        self.reset_bp_status()

        if self.instance.newhire_id:
            try:
                self.ctl.cancel_in_newhire()
            except NewhireError as exc:
                self.raise_error(exc.message)

        if self.instance.oebs_person_id:
            remove_oebs_person_task.delay(self.instance.oebs_person_id)

        comment = loader.render_to_string('startrek/offers/deleted-for-hire-order.wiki')
        if self.instance.startrek_hr_key:
            if self.startrek_hr_status == StatusEnum.draft:
                operation = IssueTransitionOperation(self.instance.startrek_hr_key)
                operation.delay(
                    transition=TransitionEnum.refused,
                    comment=comment,
                )
            else:
                add_issue_comment_task.delay(self.instance.startrek_hr_key, comment)

        if self.instance.startrek_eds_key:
            add_issue_comment_task.delay(self.instance.startrek_eds_key, comment)

        return self.instance


class RejectAction(OfferAction):
    """
    Кандидат отказался от выставленного оффера
    """
    valid_statuses = (
        OFFER_STATUSES.sent,
        OFFER_STATUSES.accepted,
    )

    def has_permission(self):
        return (
            super().has_permission()
            and (
                # Можно отменять офферы, которые уже сохранены в Наниматоре,
                self.instance.newhire_id is not None
                # либо те, которые еще только отправлены,
                or self.instance.status == OFFER_STATUSES.sent
                # либо те, которые дежурный разрешил отменить, без сохранения в Наниматор
                or self._is_safe_to_reject()
            )
        )

    def _is_safe_to_reject(self):
        safe_to_reject = fetch_comma_separated_integers(config.OFFER_IDS_SAFE_TO_REJECT)
        return self.instance.id in safe_to_reject

    def perform(self, **params):
        old_status = self.instance.status
        self.instance.status = OFFER_STATUSES.rejected
        self.instance.save()

        update_consideration_extended_status(self.instance.application.consideration)

        rejection_fields = (
            'rejection_reason',
            'competing_offer_conditions',
            'competing_company',
            'comment',
        )

        rejection_data = {f: params.get(f) for f in rejection_fields}
        rejection_data['rejection_side'] = REJECTION_SIDES.candidate
        bp_error = None
        bp_errors_ru = None
        bp_errors_en = None
        try:
            rejection_data['bp_transaction_id'] = self._reject_offer_in_bp_registry(self.instance)
        except BPRegistryUnrecoverableError as e:
            bp_error = e.message
            bp_errors_ru = e.error_messages_ru
            bp_errors_en = e.error_messages_en
        except BPRegistryError as e:
            bp_error = e.message

        self.ctl.remove_link()
        offer_rejection = self.ctl.reject(data=rejection_data, initiator=self.user)
        self.ctl.update({
            'salary_expectations': params.get('salary_expectations'),
            'salary_expectations_currency': params.get('salary_expectations_currency'),
        })

        application = self.instance.application
        vacancy = self.instance.vacancy

        if waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_PUBLICATIONS_ARCHIVE):
            vacancy.is_published = True
        vacancy.status = VACANCY_STATUSES.in_progress
        vacancy.save(update_fields=['modified', 'status', 'is_published'])

        application.status = APPLICATION_STATUSES.closed
        application.resolution = APPLICATION_RESOLUTIONS.offer_rejected
        application.save()

        if self.instance.newhire_id is not None:
            try:
                self.ctl.cancel_in_newhire()
            except NewhireError as exc:
                self.raise_error(exc.message)

        context = self.get_context()
        context['rejection_reason'] = REJECTION_REASONS_TRANSLATIONS.get(params['rejection_reason'])
        context['comment'] = params['comment']
        context['enable_bp_registry'] = waffle.switch_is_active('enable_bp_registry')
        context['bp_error'] = bp_error
        context['bp_errors_ru'] = bp_errors_ru
        context['bp_errors_en'] = bp_errors_en
        context['offer_rejection'] = offer_rejection

        fields = {k: None for k in FIELDS_TO_CLEAN_ON_REJECT}
        issue_fields = vacancy.issue_data.get('fields') or {}
        self.change_job_issue_status(
            transition=TransitionEnum.offer_declined,
            comment=loader.render_to_string('startrek/offers/rejected-job.wiki', context),
            recruiter=params['recruiter'].username,
            salarySystem=issue_fields.get('salarySystem'),
            **fields
        )

        # Отправляем коммент в тикет HR
        if self.instance.startrek_hr_key:
            comment = loader.render_to_string('startrek/offers/rejected-hr.wiki', context)
            if self.startrek_hr_status == StatusEnum.draft:
                self.change_hr_issue_status(
                    transition=TransitionEnum.refused,
                    comment=comment,
                )
            else:
                add_issue_comment_task.delay(self.instance.startrek_hr_key, comment)

        # Отправляем коммент в тикет RELOCATION, SIGNUP и SALARY
        common_comment = loader.render_to_string('startrek/offers/rejected.wiki', context)
        issues_to_comment = (
            'startrek_relocation_key',
            'startrek_signup_key',
            'startrek_bonus_key',
            'startrek_salary_key',
            'startrek_bootcamp_key',
            'startrek_eds_key',
        )
        for field in issues_to_comment:
            key = getattr(self.instance, field)
            if key:
                add_issue_comment_task.delay(key, common_comment)

        # Отправляем коммент в тикет SIGNUP
        if self.instance.startrek_signup_key:
            add_issue_comment_task.delay(self.instance.startrek_signup_key, common_comment)

        # Закрываем тикет Адаптации
        execute_adaptation_issue_transition(
            offer=self.instance,
            transition=TransitionEnum.refused,
            comment=common_comment,
        )

        if old_status == OFFER_STATUSES.accepted:
            reopen_reference_issue_task.delay(self.instance.candidate_id)

            # Возвращаем тикеты ROTATION и MYROTATION обратно в работу
            for key in self._get_rotation_keys():
                IssueTransitionOperation(key).delay(
                    transition=TransitionEnum.in_progress,
                    comment=ROTATION_EVENTS_TRANSLATIONS[ROTATION_EVENTS.rotation_rejected],
                    start=None,
                )

        # Отмена найма в Я.Найме (удаление ФЛ)
        if self.instance.oebs_person_id:
            remove_oebs_person_task.delay(self.instance.oebs_person_id)

        notification = OfferRejectedNotification(self.instance, self.user)
        notification.send()

        self.send_reference_event(REFERENCE_EVENTS.offer_rejected)

        if waffle.switch_is_active(TemporarySwitch.ENABLE_NEW_PUBLICATIONS_ARCHIVE):
            # Разархивируем публикации вакансии
            unarchive_vacancy_publications(vacancy)

        return self.instance

    def _reject_offer_in_bp_registry(self, offer):
        bp_transaction_data = BPRegistryOfferRejectionSerializer(offer).data
        bp_transaction_id = BPRegistryAPI.create_transaction(bp_transaction_data)
        return bp_transaction_id


class OfferWorkflow(Workflow):

    ACTION_MAP = {
        'update': UpdateAction,
        'approve': ApproveAction,
        'approve_by_current_team': ApproveByCurrentTeamAction,
        'approve_by_future_team': ApproveByFutureTeamAction,
        'reapprove': ReapproveAction,
        'send': SendAction,
        'delete': DeleteAction,
        'delete_for_hire_order': DeleteForHireOrderAction,
        'reject': RejectAction,
        'accept': AcceptAction,
        'accept_internal': AcceptInternalAction,
        'update_join_at': UpdateJoinAtAction,
        'update_username': UpdateUsernameAction,
        'update_department': UpdateDepartmentAction,
    }
