from typing import Dict, List, Set

from model_utils import Choices

from intranet.femida.src.offers.choices import (
    OFFER_DOC_DATA_FIELDS as DOC_DATA_FIELDS,
    OFFER_YANG_TABLE_COLUMNS as DOC_TABLE_COLUMNS,
)
from intranet.femida.src.offers.models import Offer


RESULTS = Choices(
    'OK',
    'BAD',
    # Кандидат прислал правильный документ, но его содержание не укладывается в автоматизацию.
    # Например, в паспорте нет прописки. Или по адресу не определить КЛАДР, т.к. заграница
    'BLOCK',
)


def get_yang_processed_doc_data(row: Dict) -> Dict:
    """
    Возвращает словарь с атрибутами оффера и успешно распознанными данными.
    """
    mapping = {
        DOC_DATA_FIELDS.passport_data: DOC_TABLE_COLUMNS.passport,
        DOC_DATA_FIELDS.snils_number: DOC_TABLE_COLUMNS.snils,
        DOC_DATA_FIELDS.residence_address_data: DOC_TABLE_COLUMNS.residence_address,
        DOC_DATA_FIELDS.registration_address_data: DOC_TABLE_COLUMNS.registration_address,
    }
    return {
        k: row[v]['data'] for k, v in mapping.items()
        if row[v] and row[v]['result'] == RESULTS.OK
    }


def get_yang_unprocessed_doc_fields(offer: Offer) -> Set[str]:
    """
    Возвращает множество имён документов оффера,
    для которых в оффере отсутствуют распознанные данные.
    """
    mapping = {
        DOC_DATA_FIELDS.passport_data: 'passport_pages',
        DOC_DATA_FIELDS.snils_number: 'snils',
        DOC_DATA_FIELDS.registration_address_data: 'passport_pages',
        DOC_DATA_FIELDS.residence_address_data: 'residence_address',
    }
    return {mapping[field] for field in mapping if not getattr(offer, field)}


def get_yang_unprocessed_doc_data_fields(offer: Offer) -> Set[str]:
    """
    Возвращает множество атрибутов оффера,
    для которых в оффере отсутствуют распознанные данные.
    """
    return {field for field, _ in DOC_DATA_FIELDS if not getattr(offer, field)}


def get_yang_error_list(row: Dict) -> List[str]:
    """
    Возвращает список ошибок из распознанных данных.
    """
    errors = []
    for field, _ in DOC_TABLE_COLUMNS:
        data = row[field]
        if isinstance(data, dict) and data['result'] == RESULTS.BAD:
            errors.extend(data['errors'])
    return [e.replace('other:', '') for e in errors]


def is_yang_unprocessable(row: Dict) -> bool:
    """
    Проверяет, есть ли среди данных от Янга данные с резолюцией BLOCK.
    В таком случае распарсить документ уже не получится.
    """
    return any(
        isinstance(row[field], dict) and row[field]['result'] == RESULTS.BLOCK
        for field, _ in DOC_TABLE_COLUMNS
    )
