from django.db import models
from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.contenttypes.fields import GenericRelation
from django.contrib.postgres.search import SearchVectorField

from model_utils.models import TimeStampedModel

from intranet.femida.src.wf.models import WFModelMixin
from intranet.femida.src.problems.managers import PresetProblemManager

from .choices import COMPLAINT_KINDS_TRANSLATIONS, COMPLAINT_RESOLUTIONS


User = get_user_model()


class Category(TimeStampedModel):
    name = models.CharField(max_length=100)
    subscribers = models.ManyToManyField(
        to=User,
        through='notifications.CategorySubscription',
    )

    def __str__(self):
        return 'Category {}: {}'.format(self.id, self.name)

    class Meta:
        ordering = ['name']


class AliveManager(models.Manager):

    def get_queryset(self):
        return super().get_queryset().filter(deleted=False)


class Problem(WFModelMixin, TimeStampedModel):

    objects = models.Manager()
    alive_objects = AliveManager()

    WIKI_FIELDS_MAP = {
        'description': 'formatted_description',
        'solution': 'formatted_solution',
    }

    summary = models.CharField(
        max_length=255
    )

    description = models.TextField(
        default='',
    )
    formatted_description = models.TextField(
        default='',
    )

    # Ожидаемый ответ на задачу
    solution = models.TextField(
        default='',
        blank=True,
    )
    formatted_solution = models.TextField(
        default='',
        blank=True,
    )

    categories = models.ManyToManyField(
        to=Category,
        related_name='problems',
    )

    fans = models.ManyToManyField(
        to=settings.AUTH_USER_MODEL,
        related_name='favorite_problems',
    )

    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='problems',
        null=True,
        blank=True,
    )
    original = models.ForeignKey(
        to='Problem',
        on_delete=models.CASCADE,
        related_name='duplicates',
        null=True,
        blank=True,
    )
    is_deprecated = models.BooleanField(default=False)
    deleted = models.BooleanField(default=False)

    search_vector = SearchVectorField()

    def __str__(self):
        return 'Problem {}: {}'.format(self.id, self.summary)

    class Meta:
        ordering = ['-id']


class Preset(TimeStampedModel):
    name = models.CharField(max_length=100)
    problems = models.ManyToManyField(
        to=Problem,
        related_name='presets',
        through='PresetProblem',
    )
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='presets',
        null=True,
        blank=True,
    )

    def __str__(self):
        return 'Preset {}: {}'.format(self.id, self.name)

    class Meta:
        ordering = ['name']


class PresetProblem(models.Model):

    objects = PresetProblemManager()

    preset = models.ForeignKey(
        to=Preset,
        on_delete=models.CASCADE,
        related_name='problem_links',
    )
    problem = models.ForeignKey(
        to=Problem,
        on_delete=models.CASCADE,
        related_name='preset_links',
    )
    position = models.IntegerField(default=0)

    class Meta:
        ordering = ['position']
        unique_together = ('preset', 'problem')


class Complaint(WFModelMixin, TimeStampedModel):
    """
    Жалоба пользователя на задачу
    """

    WIKI_FIELDS_MAP = {
        'comment': 'formatted_comment',
    }

    problem = models.ForeignKey(
        to=Problem,
        on_delete=models.CASCADE,
        related_name='complaints',
    )
    # Оригинальная задача, если problem является дубликатом
    original_problem = models.ForeignKey(
        to=Problem,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
    )
    is_active = models.BooleanField(default=True)
    created_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='created_complaints',
    )
    closed_by = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='closed_complaints',
        null=True,
        blank=True,
    )
    comment = models.TextField(
        default='',
        blank=True,
    )
    formatted_comment = models.TextField(
        default='',
        blank=True,
        null=True,
    )
    kind = models.CharField(
        max_length=64,
        choices=COMPLAINT_KINDS_TRANSLATIONS,
        default='other',
    )
    # Резолюция закрытия жалобы
    # На жалобах, обработанных до FEMIDA-3312, проставлена только на жалобы-дубликаты (FEMIDA-3319)
    resolution = models.CharField(
        max_length=64,
        choices=COMPLAINT_RESOLUTIONS,
        default='',
        blank=True,
    )

    comments = GenericRelation('comments.Comment')

    class Meta:
        ordering = ['-id']
