from django.contrib import admin, messages

from intranet.femida.src.core.admin import pretty_json, describe
from intranet.femida.src.publications import models
from intranet.femida.src.publications.tasks import update_og_image


class PublicationAdmin(admin.ModelAdmin):

    raw_id_fields = (
        'vacancy',
        'public_service',
    )

    search_fields = (
        '=id',
        '=vacancy__id',
        '=form_id',
        'title',
        'public_service__name_ru',
        'public_service__name_en',
    )

    list_display = (
        'id',
        'title',
        'type',
        'status',
        'vacancy_id',
        'form_id',
        'priority',
        'is_chief',
    )

    list_editable = (
        'priority',
    )

    list_filter = (
        'type',
        'status',
        'public_service__name_en',
        'is_chief',
    )

    readonly_fields = (
        'uuid',
        'pretty_bunker_json',
        'vacancy_status',
    )
    exclude = ('bunker_json',)

    actions = ('generate_og_image',)

    def pretty_bunker_json(self, obj):
        return pretty_json(obj.bunker_json)

    def vacancy_status(self, obj):
        return obj.vacancy.status

    @describe('Сгенерировать og:image')
    def generate_og_image(self, request, queryset):
        update_og_image(list(queryset.values_list('id', flat=True)))
        messages.success(request, 'Процесс генерации og:image для объявлений запущен')


class PublicationFacetAdmin(admin.ModelAdmin):

    list_display = (
        'facet',
        'value',
        'lang',
        'publication_ids',
    )

    list_filter = (
        'facet',
        'lang',
    )

    ordering = (
        'facet',
        'lang',
    )


class PublicationSuggestAdmin(admin.ModelAdmin):

    fields = (
        ('text', 'lang', 'priority'),
        'facets',
    )

    list_display = (
        'text',
        'priority',
        'lang',
        'facet_list',
        'is_active',
    )

    list_editable = (
        'priority',
        'is_active',
    )

    list_filter = (
        'lang',
        'is_active',
    )

    search_fields = (
        'text',
        'facets__value',
    )

    filter_horizontal = (
        'facets',
    )

    ordering = (
        '-priority',
        'text',
        'lang',
    )

    def get_queryset(self, request):
        return super().get_queryset(request).prefetch_related('facets')

    def facet_list(self, obj):
        facets_data = obj.facets.values_list('facet', 'value')
        return ' | '.join(
            [f'{facet}:{value}' for facet, value in facets_data]
        )


admin.site.register(models.Publication, PublicationAdmin)
admin.site.register(models.PublicationFacet, PublicationFacetAdmin)
admin.site.register(models.PublicationSuggest, PublicationSuggestAdmin)
