from collections import defaultdict

from requests import get, ConnectionError, Timeout

from django.conf import settings
from django.core.management.base import BaseCommand
from django.db import transaction
from django.db.models import Exists, OuterRef

from intranet.femida.src.publications.choices import (
    PUBLICATION_STATUSES,
    PUBLICATION_LANGUAGES,
    PUBLICATION_TYPES,
)
from intranet.femida.src.publications.models import Publication
from intranet.femida.src.services.models import PublicService
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES
from intranet.femida.src.vacancies.models import SubmissionForm, Vacancy


def _initialize_publication_from_bunker_data(form, service_id_by_slug, errors):
    url = f'http://{settings.BUNKER_HOST}/v1/cat/?node=/vacancies-www/vacancies{form.title}'
    try:
        response = get(url)
    except (ConnectionError, Timeout):
        errors['connection or timeout error'].append(form.id)
        return None
    else:
        if response.status_code != 200:
            errors[f'wrong status code - {response.status_code}'].append(form.id)
            return None

    data = response.json()
    data['submission_form_id'] = form.id

    if data['domain'] in (PUBLICATION_LANGUAGES.ru, PUBLICATION_LANGUAGES.en):
        lang = data['domain']
    else:
        errors[f'unexpected language - {data["domain"]}'].append(form.id)
        lang = PUBLICATION_LANGUAGES.ru

    if form.has_open_vacancies:
        status = PUBLICATION_STATUSES.draft
        vacancy = form.vacancies.exclude(status=VACANCY_STATUSES.closed).first()
    else:
        status = PUBLICATION_STATUSES.archived
        vacancy = form.vacancies.first()

    services_tags = data['services']
    if len(data['services']) > 1:
        errors['more than 1 service'].append(form.id)

    if not services_tags or services_tags[0] not in service_id_by_slug:
        services_tags = ['yandex']
    public_service_id = service_id_by_slug[services_tags[0]]

    publication = Publication(
        type=PUBLICATION_TYPES.external,
        status=status,
        lang=lang,
        form_id=form.forms_constructor_id,
        public_service_id=public_service_id,
        vacancy=vacancy,
        title=data['title'],
        description=data['description'],
        bunker_json=data,
    )
    return publication


class Command(BaseCommand):

    help = 'Create publications from Bunker data'

    def add_arguments(self, parser):
        parser.add_argument('--dry-run', action='store_true')

    @transaction.atomic
    def handle(self, *args, **options):
        publications = []
        forms = (
            SubmissionForm.objects
            .annotate(
                has_open_vacancies=Exists(
                    Vacancy.unsafe
                    .exclude(status=VACANCY_STATUSES.closed)
                    .filter(submission_forms=OuterRef('id'))
                    .values('id')
                )
            )
            .exclude(vacancies__isnull=True)
        )

        service_id_by_slug = dict(PublicService.objects.values_list('slug', 'id'))
        errors = defaultdict(list)

        for form in forms:
            publication = _initialize_publication_from_bunker_data(form, service_id_by_slug, errors)
            if publication:
                publications.append(publication)

        if not options['dry_run']:
            submission_form_ids = set(
                Publication.objects.values_list('bunker_json__submission_form_id', flat=True)
            )
            publications_to_create = []
            for publication in publications:
                submission_form_id = publication.bunker_json['submission_form_id']
                if submission_form_id in submission_form_ids:
                    continue

                publications_to_create.append(publication)

            Publication.objects.bulk_create(publications_to_create)
            print(len(publications_to_create), 'publications created')

        for error_name, sf_ids in errors.items():
            print(f'{len(sf_ids)} errors of type "{error_name}"')
            print('sf ids:', sf_ids, end='\n' * 2)
