# Generated by Django 2.2.24 on 2021-09-01 14:16
from django.db import migrations
from django.db.models import F
from django.conf import settings


def fill_search_vectors(apps, schema_editor):
    Publication = apps.get_model('publications', 'Publication')
    publications = Publication.objects.filter(status='published')
    publications.update(title=F('title'))


class Migration(migrations.Migration):

    dependencies = [
        ('publications', '0009_add_search_vectors'),
    ]

    operations = [
        # Создаём view для связи нескольких таблиц.
        migrations.RunSQL(
            sql="""
                CREATE OR REPLACE VIEW publications_fulltext_search_view AS
                SELECT
                    pub.id AS pub_id, pub.lang, pub.title,
                    pub.short_summary, pub.description, pub.duties, pub.key_qualifications,
                    service.name_ru || ' ' || service.name_en AS service_name,
                    service.name_en AS service_name_en,
                    prof.name || ' ' || prof.name_en AS prof_name, prof.name_en AS prof_name_en,
                    profsphere.name AS sphere_name, profsphere.name_en AS sphere_name_en,
                    array_to_string(ARRAY_AGG(skill.name) FILTER(WHERE skill.is_public = TRUE), ' ') AS skills,
                    array_to_string(ARRAY_AGG(city.name_ru), ' ')
                        || ' '|| array_to_string(ARRAY_AGG(city.name_en), ' ') AS cities,
                    array_to_string(ARRAY_AGG(city.name_en), ' ') AS cities_en
                FROM
                    publications_publication AS pub
                    LEFT JOIN services_publicservice AS service ON pub.public_service_id = service.id
                    LEFT JOIN vacancies_vacancy AS vac ON pub.vacancy_id = vac.id
                    LEFT JOIN vacancies_vacancycity USING (vacancy_id)
                    LEFT JOIN core_city AS city ON vacancies_vacancycity.city_id = city.id
                    LEFT JOIN vacancies_vacancyskill USING (vacancy_id)
                    LEFT JOIN skills_skill AS skill on vacancies_vacancyskill.skill_id = skill.id
                    LEFT JOIN professions_profession AS prof ON vac.profession_id = prof.id
                    LEFT JOIN professions_professionalsphere AS profsphere ON vac.professional_sphere_id = profsphere.id
                GROUP BY
                    pub_id, service_name, service_name_en, prof_name, prof_name_en, sphere_name, sphere_name_en
            """,
            reverse_sql="DROP VIEW IF EXISTS publications_fulltext_search_view"
        ),

        # Индексы для полнотекстового поиска
        migrations.RunSQL(
            sql=(
                "CREATE INDEX publications_publication_full_search_ru_idx "
                "ON publications_publication "
                "USING GIN (search_vector_ru)"
            ),
            reverse_sql="DROP INDEX IF EXISTS publications_publication_full_search_ru_idx",
        ),
        migrations.RunSQL(
            sql=(
                "CREATE INDEX publications_publication_full_search_en_idx "
                "ON publications_publication "
                "USING GIN (search_vector_en)"
            ),
            reverse_sql="DROP INDEX IF EXISTS publications_publication_full_search_en_idx",
        ),

        # Триггер для автоматического пересчёта поисковых векторов при обновлении самих публикаций
        migrations.RunSQL(
            sql=("""
                CREATE FUNCTION publication_update_search_vectors() RETURNS trigger AS $$
                DECLARE raw_search_data RECORD;
                begin
                    SELECT *
                    INTO raw_search_data
                    FROM publications_fulltext_search_view AS search
                    WHERE search.pub_id = NEW.id;

                    NEW.search_vector_ru :=
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.title, '')), 'A') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.service_name, '')), 'B') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.cities, '')), 'B') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.skills, '')), 'B') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.prof_name, '')), 'B') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.sphere_name, '')), 'B') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.short_summary, '')), 'C') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.description, '')), 'C') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.duties, '')), 'D') ||
                        setweight(to_tsvector('{ru_config}', coalesce(raw_search_data.key_qualifications, '')), 'D');

                    NEW.search_vector_en :=
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.title, '')), 'A') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.service_name_en, '')), 'B') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.cities_en, '')), 'B') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.skills, '')), 'B') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.prof_name_en, '')), 'B') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.sphere_name_en, '')), 'B') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.short_summary, '')), 'C') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.description, '')), 'C') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.duties, '')), 'D') ||
                        setweight(to_tsvector('{en_config}', coalesce(raw_search_data.key_qualifications, '')), 'D');

                  return NEW;
                end
                $$ LANGUAGE plpgsql;

                CREATE TRIGGER publication_update_search_vectors_trigger
                BEFORE INSERT OR UPDATE
                ON publications_publication
                FOR EACH ROW EXECUTE PROCEDURE publication_update_search_vectors();

            """.format(
                ru_config=settings.PG_TEXT_SEARCH_CONFIG,
                en_config=settings.PG_TEXT_SEARCH_CONFIG_EN,
            )),
            reverse_sql=("""
                DROP TRIGGER IF EXISTS publication_update_search_vectors_trigger ON publications_publication;
                DROP FUNCTION IF EXISTS publication_update_search_vectors();
            """),
        ),

        # Первоначальное заполнение вектора
        migrations.RunPython(fill_search_vectors, migrations.RunPython.noop),
    ]
