import uuid
from functools import partial

from constance import config
from django.conf import settings
from django.contrib.postgres.fields import JSONField, ArrayField
from django.contrib.postgres.search import SearchVectorField
from django.db import models
from model_utils.models import TimeStampedModel

from intranet.femida.src.attachments.models import upload_to
from intranet.femida.src.publications import choices
from intranet.femida.src.publications.managers import PublicationManager, PublicationSuggestManager
from intranet.femida.src.publications.tasks import (
    update_og_image,
    update_publications_search_vectors,
)
from intranet.femida.src.services.models import PublicService
from intranet.femida.src.utils.s3 import S3Storage
from intranet.femida.src.vacancies.models import Vacancy


def get_default_form_id():
    return config.DEFAULT_PUBLICATION_FORM_ID


class Publication(TimeStampedModel):
    """
    Модель внешнего объявления для Сайта Вакансий
    """
    objects = PublicationManager()

    uuid = models.UUIDField(
        default=uuid.uuid4,
        unique=True,
    )
    vacancy = models.ForeignKey(
        to=Vacancy,
        related_name='publications',
        on_delete=models.PROTECT,
    )
    public_service = models.ForeignKey(
        to=PublicService,
        related_name='publications',
        on_delete=models.PROTECT,
    )
    form_id = models.CharField(
        max_length=100,
        blank=False,
        null=True,
        default=get_default_form_id,
    )
    type = models.CharField(
        max_length=32,
        choices=choices.PUBLICATION_TYPES,
    )
    status = models.CharField(
        max_length=32,
        choices=choices.PUBLICATION_STATUSES,
        default=choices.PUBLICATION_STATUSES.draft,
    )
    priority = models.IntegerField(default=0)
    published_at = models.DateField(null=True, blank=True)
    lang = models.CharField(
        max_length=32,
        choices=choices.PUBLICATION_LANGUAGES,
        default=choices.PUBLICATION_LANGUAGES.ru,
    )

    title = models.CharField(max_length=100)
    short_summary = models.TextField(verbose_name='Краткое описание')

    duties = models.TextField(verbose_name='Что нужно делать', blank=True, null=True)
    key_qualifications = models.TextField(verbose_name='Мы ждем, что вы')
    additional_requirements = models.TextField(verbose_name='Будет плюсом, если вы', blank=True)
    conditions = models.TextField(verbose_name='Условия', blank=True, null=True)
    description = models.TextField(verbose_name='Общее описание', blank=True)

    bunker_json = JSONField(null=True, blank=True)
    is_internship = models.BooleanField(default=False, null=True)
    is_chief = models.BooleanField(default=False, null=True)

    og_image = models.FileField(
        upload_to=partial(upload_to, prefix='publications'),
        max_length=512,
        storage=S3Storage(),
        null=True,
        blank=True,
    )
    search_vector_ru = SearchVectorField(null=True, blank=True)
    search_vector_en = SearchVectorField(null=True, blank=True)

    @property
    def url(self):
        base_url = choices.PUBLICATION_URLS_MAP.get(self.lang, settings.JOBS_COM_URL)
        return f'{base_url}vacancies/{self.id}'

    def __str__(self):
        return f'Publication {self.id}, Vacancy: {self.vacancy_id}'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._update_initial()

    def _update_initial(self):
        self._initial = {
            'title': self.title,
            'short_summary': self.short_summary,
        }

    def save(self, *args, **kwargs):
        is_creation = self.pk is None
        super().save(*args, **kwargs)
        is_og_image_info_changed = (
            self.title != self._initial['title']
            or self.short_summary != self._initial['short_summary']
        )
        if is_creation or is_og_image_info_changed:
            update_og_image([self.pk])

        update_publications_search_vectors.delay(id=self.id)


class PublicationFacet(TimeStampedModel):
    """
    Модель записи со списком публикаций, отфильтрованных по одному фильтру и языку
    """
    facet = models.CharField(
        max_length=32,
        choices=choices.PUBLICATION_FACETS,
        help_text='Фасет фильтрации',
    )
    value = models.CharField(max_length=255, help_text='Slug/ID значения фасета')
    lang = models.CharField(
        max_length=2,
        choices=choices.PUBLICATION_LANGUAGES,
        help_text='Язык',
    )

    publication_ids = ArrayField(
        base_field=models.IntegerField(),
        default=list,
        help_text='Список подходящих под значение фасета публикаций',
    )

    def __str__(self):
        return f'Facet ({self.lang}) {self.facet}: {self.value}'

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=['facet', 'value', 'lang'],
                name='unique_publications_filter_index_fields',
            ),
        ]


class PublicationSuggest(models.Model):

    objects = PublicationSuggestManager()

    text = models.CharField(max_length=64, help_text='Текст, показываемый пользователю', null=False)
    facets = models.ManyToManyField(PublicationFacet, related_name='suggests', blank=True)
    lang = models.CharField(
        max_length=2,
        choices=choices.PUBLICATION_LANGUAGES,
        default=choices.PUBLICATION_LANGUAGES.ru,
        help_text='Язык',
        null=False,
    )
    priority = models.IntegerField(default=0)
    is_active = models.BooleanField(default=True)
