import logging

from collections import defaultdict

from django.utils import timezone

from intranet.femida.src.celery_app import app, NoRetry
from intranet.femida.src.publications.choices import (
    VANADIUM_BATCH_STATUSES,
    VANADIUM_TASK_STATUSES,
    PUBLICATION_STATUSES,
    PUBLICATION_TYPES,
)
from intranet.femida.src.publications.helpers import get_suitable_publications
from intranet.femida.src.utils.files import download
from intranet.femida.src.utils.lock import locked_task
from intranet.femida.src.utils.vanadium import VanadiumAPI, VanadiumError
from intranet.femida.src.yt.base import ProcessedScreenshotYTTable
from intranet.femida.src.yt.tasks import save_screenshot_data_in_yt


logger = logging.getLogger(__name__)


def update_og_image(publication_ids):
    tasks_chain = (
        save_screenshot_data_in_yt.s(publication_ids)
        | send_batch_to_vanadium.s()
        | take_result_table_from_vanadium.signature(countdown=60)
        | process_vanadium_result_table.s()
    )
    tasks_chain.delay()


@app.autoretry_task(max_reties=3)
def send_batch_to_vanadium(table_path):
    data = VanadiumAPI.enqueue_batch(table_path)
    return data


@app.autoretry_task(max_retries=5)
def take_result_table_from_vanadium(batch_data):
    result = VanadiumAPI.check_batch_status(
        batch_data['yt-cluster'],
        batch_data['queue-name'],
        batch_data['batch-name'],
    )
    batch_status = result['batch-status']
    batch_uuid = batch_data['batch-uuid']

    if batch_status in [VANADIUM_BATCH_STATUSES.new, VANADIUM_BATCH_STATUSES.progress]:
        logger.info('Vanadium has not processed batch %s yet', batch_uuid)
        raise VanadiumError(f'Batch {batch_uuid} not processed. Status: {batch_status}')
    if batch_status != VANADIUM_BATCH_STATUSES.succeeded:
        error_info = result['batch-error-info']
        raise NoRetry(f'Vanadium failed to process batch {batch_uuid}. {error_info}')

    return result['batch-output-table-path']


def _get_processed_screenshot_data_from_yt(table_path):
    table = ProcessedScreenshotYTTable(path=table_path)
    data = table.read()
    return data


@app.autoretry_task(max_retries=3)
def process_vanadium_result_table(table_path):
    from intranet.femida.src.publications.models import Publication

    results = {}
    errors = set()
    data = _get_processed_screenshot_data_from_yt(table_path)
    for row in data:
        if row['status'] == VANADIUM_TASK_STATUSES.failure:
            logger.error('%(error-type)s for %(url)s}', row)
            errors.add(row['error-type'])
            continue
        if row['screenshot-variance'] == 0:
            logger.error('Blank screenshot for %(url)s', row)
            errors.add('blank screenshot')
            continue
        publication_id = row['meta']['publication_id']
        results[publication_id] = row['screenshot']
    if errors:
        raise NoRetry(f'Errors occured: {errors}')

    publications = Publication.objects.filter(id__in=results)
    for publication in publications:
        screenshot_url = results[publication.id]
        publication.og_image = download(screenshot_url, 'og_image')
        publication.save(update_fields=['og_image'])


@app.task
@locked_task
def update_publications_search_vectors(**publications_filter):
    from intranet.femida.src.publications.controllers import update_search_vectors

    publications_filter = publications_filter or {
        'status': PUBLICATION_STATUSES.published,
        'type': PUBLICATION_TYPES.external,
    }
    update_search_vectors(**publications_filter)


@app.autoretry_task(max_retries=3)
@locked_task
def fill_publication_facet_table():
    from intranet.femida.src.publications.controllers import get_publication_facet_data
    from intranet.femida.src.publications.models import PublicationFacet

    publication_facet_data = get_publication_facet_data()

    existing_objects = PublicationFacet.objects.all()
    new_keys = set(publication_facet_data)
    updated_objects = []
    new_objects = []
    modified_now = timezone.now()
    for obj in existing_objects:
        facet, value, lang = obj.facet, obj.value, obj.lang
        if (facet, value, lang) in publication_facet_data:
            obj.publication_ids = list(publication_facet_data[(facet, value, lang)])
            obj.modified = modified_now
            updated_objects.append(obj)
            new_keys.discard((facet, value, lang))
        else:
            obj.delete()
    for facet, value, lang in new_keys:
        new_objects.append(
            PublicationFacet(
                facet=facet,
                value=value,
                lang=lang,
                publication_ids=list(publication_facet_data[(facet, value, lang)]),
            )
        )

    PublicationFacet.objects.bulk_update(
        objs=updated_objects,
        fields=('publication_ids', 'modified'),
        batch_size=100,
    )
    PublicationFacet.objects.bulk_create(new_objects, batch_size=100)


@app.autoretry_task(max_retries=3)
@locked_task
def update_suggests_priority():
    from intranet.femida.src.publications.models import Publication, PublicationSuggest
    from intranet.femida.src.publications.counters import FacetCounter

    suggests = PublicationSuggest.objects.prefetch_related('facets').all()
    total_publication_count = Publication.objects.count()
    for suggest in suggests:
        facets = suggest.facets.all()
        initial_data = defaultdict(list)
        for facet in facets:
            initial_data[facet.facet].append(facet.value)
        queryset = get_suitable_publications(
            queryset=Publication.objects.published_external(),
            text=suggest.text,
            lang=suggest.lang,
        )
        available_publication_ids = set(queryset.values_list('id', flat=True))
        counters = FacetCounter(initial_data, available_publication_ids, suggest.lang)
        count = counters.compute_selected_count()
        suggest.priority = count if count is not None else total_publication_count

    PublicationSuggest.objects.bulk_update(
        objs=suggests,
        fields=('priority',),
        batch_size=100,
    )
