from django.contrib import admin, messages
from django.contrib.admin import ModelAdmin
from django.contrib.admin.filters import SimpleListFilter
from django.utils.functional import cached_property

from intranet.femida.src.core.admin import describe
from intranet.femida.src.staff import (
    choices,
    models,
    tasks,
)


class StaffModelAdmin(ModelAdmin):

    actions = ['sync']
    sync_task = None

    def sync(self, request, queryset):
        self.sync_task.delay()
        messages.success(request, 'Синхронизация начата')
    sync.short_description = 'Запустить синхронизацию'


class OfficeAdmin(StaffModelAdmin):

    sync_task = tasks.sync_offices
    readonly_fields = ('id',)
    list_display = (
        'id',
        'name_ru',
        'name_en',
        'city',
        'is_deleted',
    )
    list_select_related = (
        'city',
    )


class OrganizationAdmin(StaffModelAdmin):

    sync_task = tasks.sync_organizations
    list_display = (
        'id',
        'name',
        'startrek_id',
        'is_deleted',
    )
    list_editable = (
        'startrek_id',
    )


class DepartmentTagsFilter(SimpleListFilter):
    title = 'Tags'
    parameter_name = 'tag'

    def lookups(self, request, model_admin):
        return choices.DEPARTMENT_TAGS

    def queryset(self, request, queryset):
        if not self.value():
            return queryset

        queryset = queryset.filter(tags__overlap=[self.value()])
        return queryset


class DepartmentAdmin(StaffModelAdmin):

    sync_task = tasks.sync_departments
    list_display = (
        'id',
        'name',
        'url',
        'group_id',
        'tags',
        'is_deleted',
    )

    search_fields = (
        'id',
        'name',
        'url',
    )

    list_filter = (
        'kind',
        DepartmentTagsFilter,
    )


class ServiceAdmin(StaffModelAdmin):

    sync_task = tasks.sync_services
    list_display = (
        'id',
        'slug',
        'name',
        'url',
        'group_id',
        'is_deleted',
    )

    search_fields = (
        'id',
        'slug',
        'name',
        'url',
    )


class DepartmentUserAdmin(StaffModelAdmin):

    sync_task = tasks.sync_department_users
    raw_id_fields = (
        'department',
        'user',
    )
    list_display = (
        'id',
        'department',
        'user',
        'role',
    )
    search_fields = (
        '=department__id',
        '=department__vacancies__id',
        '=department__url',
        '=user__username',
    )
    list_filter = (
        'role',
    )


class DepartmentAdaptationAdmin(StaffModelAdmin):

    raw_id_fields = (
        'department',
    )
    list_display = (
        'id',
        'department',
        'issue_type',
        'queue',
    )
    search_fields = (
        '=department__id',
    )


class ValueStreamAdmin(ModelAdmin):

    list_display = (
        'id',
        'slug',
        'name_ru',
        'is_active',
    )

    search_fields = (
        'id',
        'slug',
        'name_ru',
        'startrek_id',
    )


class GeographyAdmin(StaffModelAdmin):

    sync_task = tasks.sync_geographies

    list_display = (
        'id',
        'url',
        'name_ru',
        'oebs_code',
        'startrek_id',
        'is_deleted',
    )

    search_fields = (
        'id',
        'url',
        'name_ru',
        'name_en',
        'oebs_code',
    )


class StaffSyncAdmin(ModelAdmin):

    list_display = ('name', 'synced_at')
    actions = (
        'sync',
    )

    @cached_property
    def known_sync_functions(self):
        from intranet.femida.src.beamery import tasks as beamery
        from intranet.femida.src.offers import tasks as offers

        return {
            'beamery_s3_candidates': beamery.upload_candidates_to_beamery_task.si(False).delay,
            'beamery_s3_vacancies': beamery.upload_vacancies_to_beamery_task.si(False).delay,
            'beamery_s3_notes': beamery.upload_notes_to_beamery_task.si(False).delay,
            'beamery_s3_applications': beamery.upload_applications_to_beamery_task.si(False).delay,
            'newhire_offers': offers.sync_newhire_offers_task.delay,
        }

    @describe('Запустить синхронизацию')
    def sync(self, request, queryset):
        for instance in queryset:
            if instance.name in self.known_sync_functions:
                self.known_sync_functions[instance.name]()
                messages.success(request, f'Синхронизация "{instance.name}" начата')
            else:
                messages.error(request, f'Синхронизация "{instance.name}" недоступна')


admin.site.register(models.Organization, OrganizationAdmin)
admin.site.register(models.Office, OfficeAdmin)
admin.site.register(models.Department, DepartmentAdmin)
admin.site.register(models.Service, ServiceAdmin)
admin.site.register(models.DepartmentUser, DepartmentUserAdmin)
admin.site.register(models.DepartmentAdaptation, DepartmentAdaptationAdmin)
admin.site.register(models.ValueStream, ValueStreamAdmin)
admin.site.register(models.Geography, GeographyAdmin)
admin.site.register(models.StaffSync, StaffSyncAdmin)
