from dataclasses import dataclass
from datetime import date
import logging
import requests
import waffle

from django.conf import settings

from intranet.femida.src.core.exceptions import FemidaError
from intranet.femida.src.utils.tvm2_client import get_service_ticket

logger = logging.getLogger(__name__)


class BPRegistryError(FemidaError):
    """
    Стандартная ошибка при походах в реестр
    """


class BPRegistryUnrecoverableError(BPRegistryError):
    """
    Ошибка, которую не полечит ретрай.
    Т.е. транзакцию не получится создать точно.
    """

    def __init__(
        self,
        message='',
        error_messages_ru=None,
        error_messages_en=None,
        staff_response=None,
        *args,
    ):
        super().__init__(message, *args)
        self.error_messages_ru = error_messages_ru
        self.error_messages_en = error_messages_en
        self.staff_response = staff_response


class BPRegistryOebsError(BPRegistryUnrecoverableError):
    """
    Ошибка, которая возникла при отправке данных в OeBS (Я.Найм)
    """
    def __init__(self, message='', oebs_internal_errors=None, *args):
        super().__init__(message, *args)
        self.oebs_internal_errors = oebs_internal_errors or []


@dataclass
class RewardSchemeRequest:
    department_id: int
    occupation_staff_id: str
    budget_position_id: int
    is_internship: bool
    grade_level: int or None = None
    prof_level: int or None = None
    is_main_work_place: bool or None = None
    contract_term: int or None = None
    contract_term_date: date or None = None


@dataclass
class PlacementExistenceRequest:
    office_id: int
    organization_id: int


class BPRegistryAPI:
    """
    Класс для походов в реестр изменений бюджетной позиции,
    который живёт в Стаффе.
    """
    @classmethod
    def _raise_for_oebs(cls, response):
        try:
            result = response.json() if response.content else {}
            raise BPRegistryOebsError(
                message='BPRegistry OeBS error',
                oebs_internal_errors=result['oebs_internal_errors'],
            )
        except (ValueError, KeyError):
            pass

    @classmethod
    def _auth_headers(cls, use_tvm=False):
        if use_tvm:
            return {
                'X-Ya-Service-Ticket': get_service_ticket(settings.TVM_STAFF_CLIENT_ID),
            }

        return {'Authorization': 'OAuth %s' % settings.FEMIDA_ROBOT_TOKEN}

    @classmethod
    def _request(cls, method, url, params=None, data=None, use_tvm=False):
        try:
            response = requests.request(
                method=method,
                url=settings.STAFF_OEBS_URL + url,
                params=params,
                json=data,
                headers=cls._auth_headers(use_tvm),
                verify=settings.YANDEX_INTERNAL_CERT_PATH,
                timeout=settings.STAFF_TIMEOUT,
                allow_redirects=False,
            )
        except requests.exceptions.RequestException:
            logger.exception('BPRegistry API is not responding')
            raise BPRegistryError('BPRegistry API is not responding')

        if 300 <= response.status_code < 400:
            logger.error('BPRegistry API responded with status redirect')
            raise BPRegistryError('BPRegistry API responded with redirect')

        if not response.ok:
            logger.error(
                'BPRegistry API responded with status %d: %s',
                response.status_code,
                response.content.decode('utf-8'),
            )
            # Клиентская ошибка. Можно считать ее неисправимой,
            # потому что ретраи не помогут. Это либо конфликт,
            # либо нужно править данные
            if 400 <= response.status_code < 500:
                # Сейчас реестр обязан отдавать хоть какие-то ошибки при 4хх
                staff_response = response.json() if response.content else {}
                errors_section = staff_response.get('errors', {})
                messages_ru = errors_section.get('messages_ru')
                messages_en = errors_section.get('messages_en')

                raise BPRegistryUnrecoverableError(
                    'BPRegistry API responded with status %d' % response.status_code,
                    error_messages_ru=messages_ru,
                    error_messages_en=messages_en,
                    staff_response=staff_response,
                )
            cls._raise_for_oebs(response)
            raise BPRegistryError('BPRegistry API responded with status %d' % response.status_code)

        try:
            result = response.json() if response.content else {}
        except ValueError as e:
            logger.error('Error in parsing json from BPRegistry API: %s' % response.content)
            raise BPRegistryError(*e.args)

        return result

    @classmethod
    def _post(cls, url, data=None, use_tvm=False):
        return cls._request('POST', url, data=data, use_tvm=use_tvm)

    @classmethod
    def _get(cls, url, params=None, use_tvm=False):
        return cls._request('GET', url, params=params, use_tvm=use_tvm)

    @classmethod
    def create_transaction(cls, data):
        if not waffle.switch_is_active('enable_bp_registry'):
            return None
        result = cls._post('budget-position-api/attach-to-vacancy', data)
        return result['id']

    @classmethod
    def create_bp_assignment(cls, data):
        if not waffle.switch_is_active('enable_bp_registry'):
            return {}
        return cls._post('budget-position-api/create-assignment', data)

    @classmethod
    def get_review_scheme(cls, department_id, grade_level, occupation_staff_id):
        if not waffle.switch_is_active('enable_scheme_requests'):
            return {}
        params = {
            'grade_level': grade_level,
            'occupation': occupation_staff_id,
            'department': department_id,
        }
        return cls._get(
            'budget-position-api/review-scheme',
            params=params,
            use_tvm=True,
        )

    @classmethod
    def get_bonus_scheme(cls, department_id, grade_level, occupation_staff_id):
        if not waffle.switch_is_active('enable_scheme_requests'):
            return {}
        params = {
            'grade_level': grade_level,
            'occupation': occupation_staff_id,
            'department': department_id,
        }
        return cls._get(
            'budget-position-api/bonus-scheme',
            params=params,
            use_tvm=True,
        )

    @classmethod
    def get_reward_scheme(cls, request: RewardSchemeRequest):
        if not waffle.switch_is_active('enable_scheme_requests'):
            return {}

        params = {
            'grade_level': request.grade_level,
            'occupation': request.occupation_staff_id,
            'department': request.department_id,
            'budget_position_code': request.budget_position_id if request.budget_position_id != -1 else None,
            'is_internship': request.is_internship,
            'professional_level': request.prof_level,
            'is_main_work_place': request.is_main_work_place,
            'contract_term': request.contract_term,
            'contract_term_date': request.contract_term_date and request.contract_term_date.isoformat(),
        }
        return cls._get(
            'budget-position-api/reward-scheme',
            params=params,
            use_tvm=True,
        )

    @classmethod
    def check_placement_exists(cls, request: PlacementExistenceRequest) -> bool:
        if not waffle.switch_is_active('enable_placement_check'):
            return True

        params = {'organization': request.organization_id, 'office': request.office_id}
        try:
            response = cls._get('budget-position-api/check-placement', params=params, use_tvm=True)
        except BPRegistryUnrecoverableError:
            logger.info('Either organization or office is invalid')
            return False

        return response.get('exists')
