import waffle

from django.conf import settings
from django.contrib.auth import get_user_model

from intranet.femida.src.staff.models import DepartmentUser

from . import choices


User = get_user_model()


def safe_getitem(data, path, default=None):
    try:
        for key in path:
            data = data[key]
    except (KeyError, IndexError, TypeError):
        return default
    return data


def normalize_user_data(user_data):
    return {
        'username': safe_getitem(user_data, ['login'], default=''),
        'staff_id': safe_getitem(user_data, ['id']),
        'uid': safe_getitem(user_data, ['uid'], default=''),
        'email': safe_getitem(user_data, ['work_email'], default=''),
        # TODO: FEMIDA-813
        'first_name': safe_getitem(user_data, ['name', 'first', 'ru'], default='')[:30],
        'first_name_en': safe_getitem(user_data, ['name', 'first', 'en'], default='')[:30],
        'last_name': safe_getitem(user_data, ['name', 'last', 'ru'], default='')[:30],
        'last_name_en': safe_getitem(user_data, ['name', 'last', 'en'], default='')[:30],
        'is_dismissed': safe_getitem(user_data, ['official', 'is_dismissed'], default=False),
        'gender': safe_getitem(user_data, ['personal', 'gender'], default=''),
        'phone': safe_getitem(user_data, ['phones', 0, 'number'], default=''),
        'work_phone': safe_getitem(user_data, ['work_phone']),
        'lang': safe_getitem(user_data, ['language', 'ui'], default='ru'),
        'timezone': safe_getitem(user_data, ['environment', 'timezone'], default='Europe/Moscow'),
        'department_id': safe_getitem(user_data, ['department_group', 'department', 'id']),
        'is_intern': safe_getitem(user_data, ['official', 'is_trainee']),
        'chief_id': safe_getitem(user_data, ['chief', 'login']),
        'position_ru': safe_getitem(user_data, ['official', 'position', 'ru']),
    }


def _get_sorted_department_users_by_role(department, role):
    """
    Возвращает список сотрудников с ролью `role`
    для подразделения `department` и всех родительских подразделений.
    Список отсортирован снизу-вверх по ветке.
    """
    if department is None:
        return []
    department_ids = department.ancestors + [department.id]
    return sorted(
        DepartmentUser.objects
        .filter(
            department__in=department_ids,
            role=role,
            is_direct=True,
        )
        .select_related(
            'user',
            'department',
        ),
        key=lambda x: -len(x.department.ancestors)
    )


def _get_users_by_role_from_closest_department(department, role):
    """
    Возвращает сотрудников с ролью `role`
    из ближайшего подразделения, по ветке вверх от подразделения `department`,
    где есть сотрудники с такой ролью
    """
    department_users = _get_sorted_department_users_by_role(department, role)
    if not department_users:
        return []
    closest_department_id = department_users[0].department_id
    return [d.user for d in department_users if d.department_id == closest_department_id]


def get_hr_partners(department):
    """
    Возвращает непосредственных HR-партнеров для подразделения.
    Таковыми являются ближайшие HR-партнеры по ветке вверх.
    """
    return _get_users_by_role_from_closest_department(
        department=department,
        role=choices.DEPARTMENT_ROLES.hr_partner,
    )


def get_hr_analysts(department):
    """
    Возвращает непосредственных HR-аналитиков для подразделения.
    Таковыми являются ближайшие HR-аналитики по ветке вверх.
    """
    return _get_users_by_role_from_closest_department(
        department=department,
        role=choices.DEPARTMENT_ROLES.hr_analyst,
    )


def get_recruitment_partners(department):
    """
    Возвращает непосредственных рекрутмент-партнёров для подразделения.
    Таковыми являются ближайшие рекрутмент-партнёры по ветке вверх.
    """
    return _get_users_by_role_from_closest_department(
        department=department,
        role=choices.DEPARTMENT_ROLES.recruitment_partner,
    )


def get_department_chief(department):
    """
    Возвращает прямого руководителя либо ближайшего по ветве вверх
    """
    if waffle.switch_is_active('enable_head_sync'):
        du = (
            DepartmentUser.objects
            .filter(
                department=department,
                role=choices.DEPARTMENT_ROLES.chief,
                is_closest=True,
            )
            .first()
        )
    else:
        department_users = _get_sorted_department_users_by_role(
            department=department,
            role=choices.DEPARTMENT_ROLES.chief,
        )
        du = department_users[0] if department_users else None
    if du:
        return du.user


def get_department_chiefs_chain(department, from_current_direction_only=False):
    """
    Возвращает список всех руководителей
    для текущего подразделения по ветке вверх.
    :param department: подразделение
    :param from_current_direction_only:
      если True, отдает только руководителей из текущего направления
    """
    department_users = _get_sorted_department_users_by_role(
        department=department,
        role=choices.DEPARTMENT_ROLES.chief,
    )
    result = []
    for dep_user in department_users:
        result.append(dep_user.user)
        is_direction = dep_user.department.kind == choices.DEPARTMENT_KINDS.direction
        if from_current_direction_only and is_direction:
            break

    return result


def is_department_chief(department, user):
    return (
        DepartmentUser.objects
        .filter(
            department=department,
            user=user,
            role=choices.DEPARTMENT_ROLES.chief,
        )
        .exists()
    )


def get_hr_analyst_groups(department):
    groups = [settings.OK_HR_ANALYSTS_GROUP_URL]
    if department.is_in_tree(settings.YANDEX_MARKET_DEPARTMENT_ID):
        groups.append(settings.OK_MARKET_HR_ANALYSTS_GROUP_URL)
    return groups
