import logging

from django.contrib.postgres.fields import ArrayField
from django.db.models import Q, QuerySet, Manager, OuterRef, IntegerField

from intranet.femida.src.core.db import ArraySubquery, Cardinality, PythonConverterExpression
from intranet.femida.src.staff.choices import DEPARTMENT_KINDS
from intranet.femida.src.utils.translation import get_name_field


logger = logging.getLogger(__name__)


class OrganizationQuerySet(QuerySet):

    def alive(self):
        # Считаем организации без привязки к СТ неактивными,
        # т.к. использование таких организаций вызовет ошибки.
        return (
            self
            .filter(is_deleted=False)
            .exclude(startrek_id='')
        )


class TreeQuerySetMixin:

    def in_tree(self, root_id, include_self=True):
        return self.in_trees(
            root_ids=[root_id],
            include_self=include_self,
        )

    def in_trees(self, root_ids, include_self=True):
        q = Q(ancestors__overlap=root_ids)
        if include_self:
            q |= Q(id__in=root_ids)
        return self.filter(q)


class DepartmentQuerySet(QuerySet, TreeQuerySetMixin):

    def alive(self, is_alive=True):
        return self.filter(is_deleted=not is_alive)

    def with_direction(self):
        """
        Аннотирует queryset множеством всех направлений,
        в которые входит подразделение.
        На уровне модели из них выбирается самое ближайшее
        к подразделению направление.
        Можно бы было всё вынести в запрос,
        но сортировка в подзапросе по вычисленному на лету уровню
        точно негативно скажется на скорости.

        Пример использования:
        > department = Department.objects.with_direction().get(id=1)
        > direction_id = department.direction_id
        """
        return self.annotate(_directions=self._directions_annotation)

    @property
    def _directions_annotation(self):
        direction_ids = (
            self.model.objects
            .filter(kind=DEPARTMENT_KINDS.direction)
            .values('id')
        )
        subquery = ArraySubquery(
            direction_ids
            .filter(id__any=OuterRef('ancestors')),
            output_field=ArrayField(IntegerField()),
        )
        return PythonConverterExpression(subquery, convert_function=set)


class OfficeQuerySet(QuerySet):

    def alive(self):
        return self.filter(is_deleted=False)

    def exclude_groups(self):
        return self.exclude(is_group=True)


class GeographyQuerySet(QuerySet, TreeQuerySetMixin):

    def alive(self, is_alive=True):
        return self.filter(is_deleted=not is_alive)

    def exclude_roots(self):
        return self.exclude(ancestors=[])

    def order_by_tree(self):
        return (
            self.annotate(level=Cardinality('ancestors'))
            .order_by('level', get_name_field())
        )


OrganizationManager = Manager.from_queryset(OrganizationQuerySet)
DepartmentManager = Manager.from_queryset(DepartmentQuerySet)
OfficeManager = Manager.from_queryset(OfficeQuerySet)
GeographyManager = Manager.from_queryset(GeographyQuerySet)
