from typing import Optional

import pytz

from contextlib import contextmanager
from datetime import datetime

from django.conf import settings
from django.contrib.postgres.fields.array import ArrayField
from django.db import models
from django.utils import timezone
from django.utils.functional import cached_property

from intranet.femida.src.core.models import (
    I18NLocalizedNameModelMixin,
    I18NModelMixin,
    I18NNameModelMixin,
)
from intranet.femida.src.staff import choices
from intranet.femida.src.startrek.choices import ISSUE_TYPES
from intranet.femida.src.staff.managers import (
    DepartmentManager,
    GeographyManager,
    OfficeManager,
    OrganizationManager,
)


class Organization(I18NLocalizedNameModelMixin, models.Model):

    objects = OrganizationManager()

    name = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255, null=True)
    is_deleted = models.BooleanField(default=False)
    country_code = models.CharField(max_length=2, null=True)

    startrek_id = models.CharField(max_length=255, blank=True)

    @property
    def is_russian(self):
        return self.country_code == 'RU'

    def __str__(self):
        return self.name


class Office(I18NModelMixin, models.Model):

    objects = OfficeManager()
    localized_fields = ('name', 'address')

    name_ru = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255)

    address_ru = models.CharField(max_length=255)
    address_en = models.CharField(max_length=255)
    city = models.ForeignKey(
        to='core.City',
        on_delete=models.CASCADE,
        null=True,
    )
    is_deleted = models.BooleanField(default=False)

    is_group = models.NullBooleanField(
        default=False,
        help_text=(
            'Флаг для группы офисов, наподобие Красной Розы. '
            'Такие офисы создаются руками в Фемиде и больше нигде в этом виде не используются'
        ),
    )
    grouped_office_ids = ArrayField(
        base_field=models.IntegerField(),
        default=list,
        null=True,
        help_text='ID реальных офисов, объединённых в группу',
    )

    def __str__(self):
        return self.name_ru

    class Meta:
        db_table = 'offices_office'


class Group(models.Model):
    """
    Базовая абстрактная модель для сущностей, синхронизируемых из staff-api через группы
    """
    url = models.CharField(max_length=255)
    name = models.CharField(max_length=255)
    group_id = models.IntegerField()
    is_deleted = models.BooleanField(default=False)

    class Meta:
        abstract = True


class Department(I18NLocalizedNameModelMixin, Group):

    objects = DepartmentManager()

    name_en = models.CharField(max_length=255, null=True)
    kind = models.CharField(
        max_length=16,
        choices=choices.DEPARTMENT_KINDS,
        default='',
        blank=True,
        null=True,
    )
    ancestors = ArrayField(models.IntegerField(), default=list)
    tags = ArrayField(
        models.CharField(
            max_length=255,
            choices=choices.DEPARTMENT_TAGS,
        ),
        default=list,
        blank=True,
        null=True,
    )

    def is_in_trees(self, tree_ids):
        return bool((set(self.ancestors) | {self.id}) & set(tree_ids))

    def is_in_tree(self, tree_id):
        return self.is_in_trees({tree_id})

    @cached_property
    def direction_id(self):
        """
        id направления, к которому отностися данное подразделение.
        Чтобы использовать, нужно получить подразделение, используя
        queryset-метод `with_direction()`.

        > Department.objects.with_direction().get(id=10)
        """
        if not hasattr(self, '_directions'):
            raise AttributeError(
                "'Department' object has no attribute 'direction_id'. "
                "Hint: maybe you forgot to use '.with_direction()'"
            )

        # Если подразделение является направлением, отдаём собственный id
        if self.kind == choices.DEPARTMENT_KINDS.direction:
            return self.id

        # Находим ближайшее направление по ветке вверх
        for i in reversed(self.ancestors):
            if i in self._directions:
                return i

        # Если у подразделения нет направления до самого корня,
        # считаем направлением 3-й узел сверху.
        # Если сверху меньше 3-х узлов,
        # считаем, что само подразделение является направлением
        return self.ancestors[2] if len(self.ancestors) > 2 else self.id

    @cached_property
    def ancestor_departments(self):
        return sorted(
            Department.objects.filter(id__in=self.ancestors),
            key=lambda x: len(x.ancestors),
        )

    @cached_property
    def chain(self):
        return self.ancestor_departments + [self]

    @cached_property
    def business_unit_or_experiment(self):
        tags = {
            choices.DEPARTMENT_TAGS.business_unit,
            choices.DEPARTMENT_TAGS.experiment,
        }
        for department in reversed(self.chain):
            if set(department.tags) & tags:
                return department
        return None

    def __str__(self):
        return 'Department %d: %s' % (self.id, self.name)

    class Meta:
        db_table = 'departments_department'


class Service(Group):

    slug = models.CharField(max_length=255, null=True, db_index=True)

    def __str__(self):
        return f'Service {self.id} ({self.slug}): {self.name}'

    class Meta:
        db_table = 'services_service'


class DepartmentUser(models.Model):
    """
    Таблица в которой хранится инфа о том,
    является ли user руководителем (необязательно непосредственным, но любым по цепочке наверх)
    или hr-партнером подразделения department.
    """
    department = models.ForeignKey(
        to=Department,
        on_delete=models.CASCADE,
        related_name='department_users',
    )
    user = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='department_users',
    )
    role = models.CharField(max_length=32, choices=choices.DEPARTMENT_ROLES)

    # Признак непосредственного отношения к подразделению.
    is_direct = models.BooleanField(default=False)
    # Признак ближайшего руководителя к этому подразделению.
    # Если is_direct == True, то is_closest автоматически тоже True.
    # Но есть случаи, когда у подразделения нет руководителя. В таком случае is_closest == True
    # будет у того, кто ближе по цепочке подразделений.
    is_closest = models.BooleanField(default=False)

    def __str__(self):
        return 'DepartmentUser %d: (%s, %s)' % (self.id, self.department.name, self.user.username)

    class Meta:
        db_table = 'departments_departmentuser'


class DepartmentAdaptation(models.Model):
    department = models.OneToOneField(
        to=Department,
        on_delete=models.CASCADE,
        related_name='adaptation',
    )
    issue_type = models.CharField(max_length=64, choices=ISSUE_TYPES)
    queue = models.CharField(max_length=64)


# TODO: вынести из приложения staff
class StaffSync(models.Model):
    """
    Таблица, в которой хранится дата последнего синка.
    """
    name = models.CharField(max_length=255)
    synced_at = models.DateTimeField()

    def __str__(self):
        return self.name

    @classmethod
    @contextmanager
    def sync(cls, name, track_start=True):
        sync_entry, _ = cls.objects.get_or_create(
            name=name,
            # Note: Здесь умышленно берется не datetime.min,
            # чтобы от дефолтной даты можно было отнимать вменяемое кол-во времени
            defaults={'synced_at': datetime(1000, 1, 1, tzinfo=pytz.utc)},
        )
        synced_at = timezone.now()

        yield sync_entry.synced_at

        sync_entry.synced_at = synced_at if track_start else timezone.now()
        sync_entry.save()


class ValueStream(I18NNameModelMixin, models.Model):

    slug = models.CharField(max_length=255)
    staff_id = models.IntegerField()
    is_active = models.BooleanField(default=True)

    name_ru = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255)

    oebs_product_id = models.IntegerField(null=True)
    startrek_id = models.IntegerField(null=True, blank=True)

    @property
    def service(self) -> Optional[Service]:
        # TODO: Create a one2one relation with Service
        # Оказывается, для стаффа abc-сервисы не связаны с основным продуктом через id
        # Они связаны через slug
        return Service.objects.filter(slug=self.slug).first()

    def __str__(self):
        return self.name_ru


class Geography(I18NNameModelMixin, models.Model):

    objects = GeographyManager()

    url = models.CharField(max_length=255)
    name_ru = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255)
    oebs_code = models.CharField(unique=True, null=True, max_length=32)
    startrek_id = models.IntegerField(null=True, blank=True)
    is_deleted = models.BooleanField(default=False)

    ancestors = ArrayField(models.IntegerField(), default=list)

    kind = models.CharField(
        max_length=16,
        choices=choices.GEOGRAPHY_KINDS,
        default='',
        blank=True,
    )

    def is_in_trees(self, tree_ids):
        return bool((set(self.ancestors) | {self.id}) & set(tree_ids))

    def is_in_tree(self, tree_id):
        return self.is_in_trees({tree_id})

    def __str__(self):
        return 'Geography %d: %s' % (self.id, self.name)
