import logging

from collections import namedtuple
from typing import Set

from constance import config
from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction
from django.db.models import F
from django.utils import timezone
from django.utils.functional import cached_property

from ids.registry import registry

from intranet.femida.src.actionlog.tasks import touch_candidates_task
from intranet.femida.src.candidates.bulk_upload.uploaders import CandidateStaffUploader
from intranet.femida.src.candidates.choices import CONTACT_SOURCES
from intranet.femida.src.candidates.models import Candidate, CandidateContact
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.core.models import City
from intranet.femida.src.isearch.tasks import bulk_push_candidates_to_isearch
from intranet.femida.src.offers.choices import OFFER_STATUSES
from intranet.femida.src.offers.models import Position, Offer
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.staff.contacts import fetch_contacts_from_person
from intranet.femida.src.staff.models import (
    Service,
    Department,
    DepartmentUser,
    Office,
    Organization,
    ValueStream,
    Geography,
)
from intranet.femida.src.staff.choices import DEPARTMENT_ROLES, GEOGRAPHY_KINDS
from intranet.femida.src.staff.helpers import (
    safe_getitem,
    normalize_user_data,
)
from intranet.femida.src.staff.value_streams import get_value_streams
from intranet.femida.src.startrek.utils import get_startrek_field_values
from intranet.femida.src.stats.utils import str_to_moscow_date
from intranet.femida.src.utils.strings import camel_to_kebab
from intranet.femida.src.utils.tvm2_client import get_service_ticket
from intranet.femida.src.vacancies.memberships import create_vacancy_memberships


logger = logging.getLogger(__name__)
User = get_user_model()


_prepared_dep_user_fields = (
    'department_id',
    'user_id',
    'role',
    'is_direct',
    'is_closest',
)


PreparedDepartmentUserBase = namedtuple(
    typename='PreparedDepartmentUserBase',
    field_names=_prepared_dep_user_fields,
)


class PreparedDepartmentUser(PreparedDepartmentUserBase):

    @cached_property
    def instance(self):
        return DepartmentUser(**self._asdict())


class StaffSynchronizerBase:

    resource_type = None
    model_class = None

    def __init__(self):
        self.repo = registry.get_repository(
            service='staff',
            resource_type=self.resource_type,
            user_agent='femida',
            service_ticket=get_service_ticket(settings.TVM_STAFF_API_CLIENT_ID),
            timeout=settings.STAFF_API_TIMEOUT,
        )

    def sync(self):
        raise NotImplementedError


class GroupSynchronizer(StaffSynchronizerBase):

    group_type = None
    resource_type = 'group'
    fields = (
        'id',
        'type',
        'name',
        'service',
        'department',
        'url',
        'is_deleted',
    )

    def has_anything_changed(self, instance, item):
        for field_name in item:
            if getattr(instance, field_name) != item[field_name]:
                return True
        return False

    def normalize_item(self, item):
        return {
            'id': safe_getitem(item, [self.group_type, 'id']),
            'group_id': item['id'],
            'name': item['name'],
            'url': item['url'],
            'is_deleted': item['is_deleted'],
        }

    def sync(self):
        params = {
            '_limit': 1000,
            '_sort': 'id',
            '_fields': ','.join(self.fields),
            'type': self.group_type,
            'is_deleted': 'true,false',
        }
        data = self.repo.getiter(params)
        mapping = {i.id: i for i in self.model_class.objects.all()}

        created_counter = 0
        updated_counter = 0
        logger.info(
            'Start %ss sync. Total number of items: %d',
            self.group_type,
            data.total,
        )
        for item in data:
            normalized_item = self.normalize_item(item)
            # staff-api как минимум в тестинге может отдавать группы, у которых нет
            # ни id service, ни id департамента
            if normalized_item['id'] is None:
                continue

            instance = mapping.get(normalized_item['id']) or self.model_class()

            if not self.has_anything_changed(instance, normalized_item):
                continue

            if instance.id:
                updated_counter += 1
            else:
                created_counter += 1

            for field_name, value in normalized_item.items():
                setattr(instance, field_name, value)
            instance.save()

        logger.info(
            '%ss sync completed. Created: %d, Updated: %d',
            self.group_type,
            created_counter,
            updated_counter,
        )


class ServiceSynchronizer(GroupSynchronizer):

    model_class = Service
    group_type = 'service'

    def normalize_item(self, item):
        normalized_item = super().normalize_item(item)
        url = normalized_item['url']
        prefix = 'svc_'
        normalized_item['slug'] = url[len(prefix):] if url.startswith(prefix) else url
        return normalized_item


class DepartmentSynchronizer(GroupSynchronizer):

    model_class = Department
    group_type = 'department'
    fields = GroupSynchronizer.fields + (
        'ancestors.department.id',
        'department.name.full.en',
    )

    def normalize_item(self, item):
        normalized_item = super().normalize_item(item)
        normalized_item['ancestors'] = [i['department']['id'] for i in item['ancestors']]
        normalized_item['kind'] = safe_getitem(item, ['department', 'kind', 'slug'], '')
        normalized_item['name_en'] = (
            safe_getitem(item, ['department', 'name', 'full', 'en'])
            or item['name']
        )
        return normalized_item


class DepartmentUserSyncronizer(StaffSynchronizerBase):

    model_class = DepartmentUser
    resource_type = 'group'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.prepared_dep_users = set()

    @cached_property
    def username_to_id_map(self):
        return dict(User.objects.values_list('username', 'id'))

    @cached_property
    def existing_dep_users_map(self):
        return {
            PreparedDepartmentUser(*item[1:]): item[0]
            for item in self.model_class.objects.values_list('id', *_prepared_dep_user_fields)
        }

    def _get_department_chain(self, group):
        """
        :param group: департамент (group типа department) из выдачи staff-api
        :return: цепочка подразделений вверх, включая искомое
        """
        return [group['department']] + [a['department'] for a in reversed(group['ancestors'])]

    def _is_prepared_dep_user_suitable(self, item):
        """
        Возвращает True, если связь подразделение-пользователь подходящая.
        В БД мы храним связи с уникальными (department_id, user_id, role).
        Иногда пользователь может одновременно иметь
        и прямое, и косвенное отношение к подразделению.
        Для прямого отношения приоритет выше, поэтому подходящими считаем:
        - любое прямое отношение (is_direct=True);
        - косвенное отношение, если не существует прямого;
        """
        if item.is_direct:
            return True

        reverse_item = item._replace(is_direct=True, is_closest=True)
        if reverse_item not in self.prepared_dep_users:
            return True

        return False

    def _handle_group(self, group):
        """
        Формирует необходимые PreparedDepartmentUser для каждого департамента в выдаче staff-api
        """
        target_dep_id = group['department']['id']
        department_chain = self._get_department_chain(group)
        # Роли, для которых уже проставили is_closest = True для искомого департамента
        roles_with_closest = set()

        for department in department_chain:
            is_direct = target_dep_id == department['id']
            # Роли для флага is_closest апдейтим, когда пройдемся по всем хэдам родительского
            # департамента, так как в одном департаменте может быть несколько человек в одной роли
            new_roles_with_closest = set()
            for head in department['heads']:
                role = head['role']
                if role not in DEPARTMENT_ROLES:
                    continue

                is_closest = role not in roles_with_closest
                user_id = self.username_to_id_map.get(head['person']['login'])
                prepared_dep_user = PreparedDepartmentUser(
                    department_id=target_dep_id,
                    user_id=user_id,
                    role=role,
                    is_direct=is_direct,
                    is_closest=is_closest,
                )
                if self._is_prepared_dep_user_suitable(prepared_dep_user):
                    self.prepared_dep_users.add(prepared_dep_user)
                    new_roles_with_closest.add(role)
            roles_with_closest |= new_roles_with_closest

    def sync(self):
        """
        key - это (department_id, user_id, role) - уникальное сочетание полей для DepartmentUser
        """
        logger.info('Start DepartmentUser sync...')

        fields = (
            'department.id',
            'department.heads.role',
            'department.heads.person.login',
            'ancestors.department.id',
            'ancestors.department.heads.role',
            'ancestors.department.heads.person.login',
        )
        params = {
            '_limit': 200,
            'type': 'department',
            '_fields': ','.join(fields),
            '_sort': 'id',
        }
        data = self.repo.getiter(params)
        for group in data:
            self._handle_group(group)

        dep_user_ids_to_delete = {
            dep_user_id for dep_user, dep_user_id in self.existing_dep_users_map.items()
            if dep_user not in self.prepared_dep_users
        }

        with transaction.atomic():
            # Удаляются не только department_users, но и по каскаду соответствущие memberships
            _total, deletion_details = (
                DepartmentUser.objects.filter(id__in=dep_user_ids_to_delete).delete()
            )

            dep_users_to_create = self.prepared_dep_users - set(self.existing_dep_users_map)
            DepartmentUser.objects.bulk_create([d.instance for d in dep_users_to_create])

            inserted_count = create_vacancy_memberships()

        logger.info(
            'DepartmentUser sync completed. Created: %d. Deleted: %d',
            len(dep_users_to_create),
            deletion_details.get('staff.DepartmentUser', 0),
        )
        logger.info(
            'VacancyMembership sync completed. Created: %d. Deleted: %d ',
            inserted_count,
            deletion_details.get('vacancies.VacancyMembership', 0),
        )


class UserSyncronizer(StaffSynchronizerBase):

    model_class = User
    resource_type = 'person'

    USER_FIELDS = (
        'username',
        'staff_id',
        'uid',
        'email',
        'first_name',
        'first_name_en',
        'last_name',
        'last_name_en',
        'is_dismissed',
        'gender',
        'phone',
        'work_phone',
        'lang',
        'timezone',
        'department_id',
        'is_intern',
        'chief_id',
        'position_ru',
    )

    def has_anything_changed(self, user, user_data):
        return any(getattr(user, f) != user_data[f] for f in self.USER_FIELDS)

    @staticmethod
    def extract_contacts(user, candidate_id):
        return {
            (contact['type'], candidate_id, contact['account_id'])
            for contact in fetch_contacts_from_person(user)
        }

    @cached_property
    def candidate_sync_department_ids(self) -> Set[int]:
        root_ids = config.STAFF_CANDIDATE_SYNC_DEPARTMENT_IDS.strip()
        root_ids = [int(i) for i in root_ids.split(',')] if root_ids else []
        return set(
            Department.objects
            .in_trees(root_ids)
            .values_list('id', flat=True)
        )

    def sync(self):
        fields = (
            'id',
            'login',
            'uid',
            'work_email',
            'name.first',
            'name.last',
            'official.is_dismissed',
            'official.is_robot',
            'official.is_trainee',
            'official.organization.id',
            'official.position.ru',
            'personal.gender',
            'personal.birthday',
            'phones',
            'accounts',
            'yandex.login',
            'work_phone',
            'language.ui',
            'environment.timezone',
            'department_group.department.id',
            'chief.login',
        )
        users_data = self.repo.getiter({
            '_limit': 1000,
            '_fields': ','.join(fields),
            '_sort': 'id',
        })
        username_user_id_map = {u.username: u for u in User.objects.all()}
        username_cand_id_map = dict(
            Candidate.unsafe.alive().exclude(login='').values_list('login', 'id')
        )
        logger.info('Total number of users: {}'.format(users_data.total))
        created_counter = 0
        updated_counter = 0
        self.actual_contact_tuples = set()
        usernames_with_changed_is_dismissed = []
        candidate_sync_data = []

        username_blacklist = config.STAFF_CANDIDATE_SYNC_USERNAME_BLACKLIST.strip().split(',')
        username_blacklist = {i.strip() for i in username_blacklist}

        # Note: получаем логины кандидатов, которые приняли оффер.
        # Сотрудников с такими логинами нам не надо добавлять в нашу базу кандидатов
        # через регулярный синк. У таких кандидатов логин проставится при закрытии оффера.
        username_blacklist |= set(
            Offer.unsafe
            .filter(status=OFFER_STATUSES.accepted)
            .values_list('username', flat=True)
        )

        for user_data in users_data:
            try:
                normalized_user = normalize_user_data(user_data)
                username = normalized_user['username']
                department_id = normalized_user['department_id']
                user = username_user_id_map.get(username) or User()

                if username in username_cand_id_map:
                    self.actual_contact_tuples |= self.extract_contacts(
                        user=user_data,
                        candidate_id=username_cand_id_map[username],
                    )

                is_candidate_sync_needed = (
                    username not in username_cand_id_map
                    and department_id in self.candidate_sync_department_ids
                    and username not in username_blacklist
                    and not safe_getitem(user_data, ['official', 'is_robot'], True)
                )
                if is_candidate_sync_needed:
                    candidate_sync_data.append(user_data)

                if user.id and user.is_dismissed != normalized_user['is_dismissed']:
                    usernames_with_changed_is_dismissed.append(username)

                if self.has_anything_changed(user, normalized_user):
                    if user.id:
                        updated_counter += 1
                    else:
                        created_counter += 1
                    for field_name in self.USER_FIELDS:
                        setattr(user, field_name, normalized_user[field_name])
                    user.save()
            except Exception:
                logger.exception('Unhandled exception in sync_users')

        logger.info(
            'Users synchronization completed, '
            '%d new users created, %d users updated',
            created_counter, updated_counter,
        )

        self.sync_contacts()

        # Загружаем в базу кандидатов Фемиды сотрудников со Стаффа
        if candidate_sync_data:
            uploader = CandidateStaffUploader(candidate_sync_data)
            uploader.upload()

        self.push_candidates(usernames_with_changed_is_dismissed)

    def push_candidates(self, logins):
        """
        Если у юзера поменялся флаг is_dismissed, нужно запушить соответствующих кандидатов в поиск
        и в другие системы
        """
        candidate_ids = list(
            Candidate.unsafe
            .exclude(login='')
            .filter(login__in=logins)
            .values_list('id', flat=True)
        )
        touch_candidates_task.delay(candidate_ids)
        bulk_push_candidates_to_isearch.delay(candidate_ids)

    def sync_contacts(self):
        contacts_qs = (
            CandidateContact.objects
            .filter(candidate__is_duplicate=False)
            .exclude(candidate__login='')
            .values('id', 'type', 'candidate_id', 'normalized_account_id')
        )
        db_contacts_map = {
            (c['type'], c['candidate_id'], c['normalized_account_id']): c['id'] for c in contacts_qs
        }
        new_contacts = []

        logger.info('%d user contacts to sync.', len(self.actual_contact_tuples))

        for key in self.actual_contact_tuples:
            if key in db_contacts_map:
                continue

            contact_type, candidate_id, account_id = key
            new_contacts.append(CandidateContact(
                type=contact_type,
                candidate_id=candidate_id,
                account_id=account_id,
                normalized_account_id=account_id,
                source=CONTACT_SOURCES.staff,
            ))

        created_contacts = CandidateContact.objects.bulk_create(new_contacts)
        for c in created_contacts:
            self.actual_contact_tuples.add((c.type, c.candidate_id, c.normalized_account_id))

        contact_ids_to_delete = []
        staff_contacts_qs = (
            CandidateContact.objects
            .filter(source=CONTACT_SOURCES.staff, is_active=True)
            .values('id', 'type', 'candidate_id', 'normalized_account_id')
        )
        for contact in staff_contacts_qs:
            key = (contact['type'], contact['candidate_id'], contact['normalized_account_id'])
            if key not in self.actual_contact_tuples:
                contact_ids_to_delete.append(contact['id'])
        CandidateContact.objects.filter(id__in=contact_ids_to_delete).delete()

        logger.info(
            'User contacts synchronization completed, %d contacts were created, '
            '%d contacts were deleted',
            len(new_contacts), len(contact_ids_to_delete),
        )


class OrganizationSynchronizer(StaffSynchronizerBase):

    model_class = Organization
    resource_type = 'organization'
    fields = (
        'id',
        'name',
        'name_en',
        'is_deleted',
        'country_code',
    )

    def normalize_item(self, item):
        return {
            'name': item['name'],
            'name_en': item['name_en'],
            'is_deleted': item['is_deleted'],
            'country_code': item['country_code'],
        }

    def sync(self):
        params = {
            '_limit': 100,
            '_fields': ','.join(self.fields),
            '_sort': 'id',
            'is_deleted': 'true,false',
        }
        data = self.repo.getiter(params)
        created_counter = 0
        updated_counter = 0
        logger.info('Start organizations sync. Total number of items: %d', data.total)
        for item in data:
            normalized_item = self.normalize_item(item)
            organization, created = self.model_class.objects.update_or_create(
                id=item['id'],
                defaults=normalized_item,
            )
            created_counter += 1 if created else 0
            updated_counter += 0 if created else 1

        self.sync_startrek_ids()

        logger.info(
            'Organizations sync completed. Created: %d, Updated: %d',
            created_counter,
            updated_counter,
        )

    def sync_startrek_ids(self):
        """
        Проставляет автоматом startrek_id для ЮЛ, название которых
        полностью совпадает с одним из ЮЛ в Трекере.
        Возможно, приложение `staff` – не самое удачное место для этой синхронизации,
        но делать ещё один синк рядом кажется лишним.
        """
        try:
            startrek_organizations = set(get_startrek_field_values('legalEntity'))
        except Exception as exc:
            logger.warning('Skipping sync with startrek for organizations', exc_info=exc)
            return
        organizations = self.model_class.objects.filter(is_deleted=False, startrek_id='')
        organization_ids = [o.id for o in organizations if o.name in startrek_organizations]
        qs = self.model_class.objects.filter(id__in=organization_ids)
        qs.update(startrek_id=F('name'))
        logger.info('Set startrek_id for %d organizations', len(organization_ids))


class OfficeSynchronizer(StaffSynchronizerBase):

    model_class = Office
    resource_type = 'office'
    fields = (
        'id',
        'name',
        'address',
        'city.id',
        'city.name',
        'is_deleted',
    )

    def normalize_item(self, item):
        city_id = safe_getitem(item, ['city', 'id'])
        if not city_id:
            raise City.DoesNotExist
        return {
            'name_ru': item['name']['ru'],
            'name_en': item['name']['en'],
            'address_ru': item['address']['ru'],
            'address_en': item['address']['en'],
            'city': City.objects.get(staff_id=city_id),
            'is_deleted': item['is_deleted'],
        }

    def sync(self):
        params = {
            '_limit': 1000,
            '_fields': ','.join(self.fields),
            '_sort': 'id',
            'is_deleted': 'true,false',
        }
        data = self.repo.getiter(params)
        created_counter = 0
        updated_counter = 0
        logger.info('Start offices sync. Total number of items: %d', data.total)
        for item in data:
            try:
                normalized_item = self.normalize_item(item)
            except City.DoesNotExist:
                # Не синкаем офисы, для которых не были заведены города
                if safe_getitem(item, ['city', 'name', 'ru']):
                    logger.warning(
                        'Offices sync. City does not exist %d, %s',
                        item['city']['id'],
                        item['city']['name']['ru'],
                    )
                else:
                    logger.warning('Offices sync. Office does not have a city')
                continue

            qs = self.model_class.objects.exclude_groups()
            office, created = qs.update_or_create(
                id=item['id'],
                defaults=normalized_item,
            )
            created_counter += 1 if created else 0
            updated_counter += 0 if created else 1

        logger.info(
            'Offices sync completed. Created: %d, Updated: %d',
            created_counter,
            updated_counter,
        )


class PositionSynchronizer(StaffSynchronizerBase):

    model_class = Position
    resource_type = 'position'
    fields = (
        'id',
        'name',
        'end_date',
    )

    def normalize_item(self, item):
        end_date = item.get('end_date')
        is_deleted = bool(end_date and str_to_moscow_date(end_date) < timezone.now())
        return {
            'name_ru': item['name']['ru'],
            'name_en': item['name']['en'],
            'is_deleted': is_deleted,
        }

    def sync(self):
        data = self.repo.getiter({
            '_limit': 1000,
            '_fields': ','.join(self.fields),
            '_sort': 'id',
        })

        created_counter = 0
        updated_counter = 0
        logger.info(
            'Start positions sync. Total number of items: %d',
            data.total,
        )
        for item in data:
            normalized_item = self.normalize_item(item)

            position, created = self.model_class.objects.update_or_create(
                staff_id=item['id'],
                defaults=normalized_item,
            )
            created_counter += 1 if created else 0
            updated_counter += 0 if created else 1

        logger.info(
            'Positions sync completed. Created: %d, Updated: %d',
            created_counter,
            updated_counter,
        )


class ProfessionSynchronizer(StaffSynchronizerBase):

    model_class = Profession
    resource_type = 'occupation'
    fields = (
        'id',
        'description.ru',
        'description.en',
        'end_date',
    )

    def normalize_item(self, item):
        end_date = item.get('end_date')
        is_active = bool(not end_date or str_to_moscow_date(end_date) > timezone.now())
        return {
            'name': item['description']['ru'],
            'name_en': item['description']['en'],
            'is_active': is_active,
        }

    def sync(self):
        data = self.repo.getiter({
            '_limit': 150,
            '_fields': ','.join(self.fields),
            '_sort': 'id',
        })

        created_counter = 0
        updated_counter = 0
        logger.info('Start professions sync. Total number of items: %d', data.total)
        fetched_ids = []
        for item in data:
            normalized_item = self.normalize_item(item)

            profession, created = self.model_class.objects.get_or_create(
                staff_id=item['id'],
                defaults=dict(
                    normalized_item,
                    is_active=False,
                    professional_sphere_id=settings.DEFAULT_PROF_SPHERE_ID,
                    slug=camel_to_kebab(item['id']) if item['id'] else None,
                ),
            )
            if not created:
                normalized_item['is_active'] &= (
                    profession.professional_sphere_id != settings.DEFAULT_PROF_SPHERE_ID
                )
                update_instance(profession, normalized_item)

            created_counter += created
            updated_counter += not created

            fetched_ids.append(item['id'])

        updated_counter += (
            Profession.objects
            .exclude(staff_id__in=fetched_ids)
            .update(is_active=False)
        )

        logger.info(
            'Professions sync completed. Created: %d, Updated: %d',
            created_counter,
            updated_counter,
        )


class ValueStreamSynchronizer:

    def normalize_item(self, item):
        return {
            'slug': item['slug'],
            'name_ru': item['name'],
            'name_en': item['name_en'],
            'oebs_product_id': item['oebs_product_id'],
            'startrek_id': item['st_translation_id'],
            'is_active': item['is_active'],
        }

    def sync(self):
        updated_counter = 0
        created_counter = 0

        data = get_value_streams()

        logger.info('Start value streams sync. Total number of items: %d', len(data))
        for item in data:
            normalized_item = self.normalize_item(item)
            value_stream, created = ValueStream.objects.update_or_create(
                staff_id=item['id'],
                defaults=normalized_item,
            )
            created_counter += created
            updated_counter += not created

        logger.info(
            'Value streams sync completed. Created: %d, Updated: %d',
            created_counter,
            updated_counter,
        )


class GeographySynchronizer(StaffSynchronizerBase):

    resource_type = 'geography'

    fields = (
        'id',
        'is_deleted',
        'url',
        'name',
        'oebs_code',
        'st_translation_id',
        'ancestors.id',
    )

    CIS_GEOGRAPHY_URL = 'geo_cis'

    def normalize_item(self, item, cis_geography_id):
        name = item['name']['full']
        return {
            'url': item['url'],
            'name_ru': name['ru'],
            'name_en': name['en'] or name['ru'],
            'oebs_code': item['oebs_code'],
            'startrek_id': item['st_translation_id'],
            'is_deleted': item['is_deleted'],
            'ancestors': [ancestor['id'] for ancestor in item['ancestors']],
            'kind': self._resolve_kind(item, cis_geography_id),
        }

    def sync(self):
        params = {
            '_limit': 1000,
            '_fields': ','.join(self.fields),
            '_sort': 'id',
            'is_deleted': 'true,false',
        }
        data = list(self.repo.getiter(params))
        updated_counter = 0
        created_counter = 0
        cis_geography_id = self._get_cis_geography_id(data)

        logger.info('Start geography sync. Total number of items: %d', len(data))
        for item in data:
            normalized_item = self.normalize_item(item, cis_geography_id)
            _, created = Geography.objects.update_or_create(
                id=item['id'],
                defaults=normalized_item,
            )
            created_counter += created
            updated_counter += not created

        logger.info(
            'Geography sync completed. Created: %d, Updated: %d',
            created_counter,
            updated_counter,
        )

    def _get_cis_geography_id(self, data):
        return next((g['id'] for g in data if g['url'] == self.CIS_GEOGRAPHY_URL), None)

    @staticmethod
    def _resolve_kind(item, cis_geography_id):
        if item['id'] == cis_geography_id:
            return GEOGRAPHY_KINDS.rus

        ancestors = [ancestor['id'] for ancestor in item['ancestors']]

        if cis_geography_id in ancestors:
            return GEOGRAPHY_KINDS.rus

        return GEOGRAPHY_KINDS.international
