from dataclasses import dataclass
from datetime import datetime
from typing import Any


@dataclass
class Trigger:

    id: int
    name: str
    webhooks: list[str]
    success: bool

    @classmethod
    def build(cls, trigger_entry):
        trigger = trigger_entry['trigger']
        return cls(
            id=trigger.id,
            name=trigger.name,
            webhooks=[i['endpoint'] for i in trigger.actions if i['type'] == 'Webhook'],
            success=trigger_entry['success'],
        )


@dataclass
class ChangelogEntry:

    type: str
    updated_at: datetime
    updated_by: str
    status_from: Any = None
    status_to: Any = None
    executed_triggers: list[Trigger] = None

    @staticmethod
    def _format_status(status):
        return f'{status.key}/{status.name}' if status else '0'

    @property
    def transition(self):
        if not self.status_to:
            return None
        return '{status_from} → {status_to}'.format(
            status_from=self._format_status(self.status_from),
            status_to=self._format_status(self.status_to),
        )


def get_short_issue_changelog(issue):
    """
    Отдаёт в сокращенном виде историю изменений заданного тикета.
    Отдаётся только информация о переходах, а также о сработавших webhook-триггерах
    """
    result = []

    for st_entry in issue.changelog.get_all():
        entry = ChangelogEntry(
            type=st_entry.type,
            updated_at=datetime.fromisoformat(st_entry.updatedAt.split('+')[0]),
            updated_by=st_entry.updatedBy.login,
        )

        status_changes = (f for f in st_entry.fields if f['field'].id == 'status')
        status_change = next(status_changes, None)
        if status_change:
            entry.status_from = status_change['from']
            entry.status_to = status_change['to']

        if getattr(st_entry, 'executedTriggers', None):
            entry.executed_triggers = [Trigger.build(t) for t in st_entry.executedTriggers]

        result.append(entry)

    return result
