import logging
from uuid import uuid4

from django.conf import settings
from django.utils.functional import cached_property
from ids.exceptions import BackendError
from ylog.context import log_context

from intranet.femida.src.startrek.tasks import execute_issue_operation_task
from intranet.femida.src.startrek.utils import (
    get_issue,
    StartrekError,
    TransitionDoesNotExist,
    TransitionFailed,
)
from intranet.femida.src.utils.switches import is_startrek_operations_enabled


logger = logging.getLogger(__name__)


class IssueBaseOperation:
    """
    Базовый класс для операций в Трекере.
    Помимо выполнения самой операции, в определённое поле
    в Трекере записывается ID этой операции,
    чтобы при повторных попытках можно было узнать,
    не выполнялась ли эта операция ранее.
    Актульно при таймаутах, когда Фемида считает, что операция не удалась,
    а Трекер тем временем завершает действие.

    Есть 2 способа запуска:
    - operation() - прямой запуск операции
    - operation.delay() - откладывание операции для запуска через celery
    """
    def __init__(self, key, operation_id=None):
        self._key = key
        self._operation_id = operation_id or uuid4().hex

    @cached_property
    def _issue(self):
        return get_issue(self._key)

    def _add_operation_id(self, fields):
        if not is_startrek_operations_enabled():
            return fields
        operations_field = settings.STARTREK_OPERATIONS_FIELD
        assert operations_field not in fields, 'field `%s` is reserved' % operations_field
        fields[operations_field] = {'add': [self._operation_id]}
        return fields

    def _log_exception(self, exc, description):
        ctx = {
            'class': self.__class__.__name__,
            'key': self._key,
            'id': self._operation_id,
        }
        with log_context(tracker_issue_operation=ctx):
            logger.exception('{}\n{}'.format(description, exc.extra))

    def execute(self, **fields):
        """
        Основная логика операции.
        Нужно описать в дочерних классах
        """
        raise NotImplementedError

    def __call__(self, **fields):
        """
        Прямой запуск операции
        """
        fields = self._add_operation_id(fields)
        try:
            self.execute(**fields)
        except BackendError as exc:
            self._log_exception(exc, 'Error processing an issue')
            raise StartrekError(*exc.args)

    def si(self, **fields):
        """
        Immutable signature основной таски.
        Сейчас используется для возможности запуска операций в chain'e
        """
        return execute_issue_operation_task.si(
            operation_class=self.__class__.__name__,
            key=self._key,
            operation_id=self._operation_id,
            **fields
        )

    def delay(self, **fields):
        """
        Отложенный запуск операции
        """
        self.si(**fields).delay()

    @cached_property
    def is_completed(self):
        """
        Проверка на то, что операция уже выполнилась успешно ранее
        """
        if not is_startrek_operations_enabled():
            return False
        return self._operation_id in getattr(self._issue, settings.STARTREK_OPERATIONS_FIELD, [])


class IssueTransitionOperation(IssueBaseOperation):
    """
    Операция перевода статуса в тикете в Трекере
    """
    def execute(self, transition, **fields):
        try:
            transition = self._issue.transitions.get(transition)
        except BackendError as exc:
            self._log_exception(exc, 'Transition %s failed' % transition)
            if exc.status_code == 404:
                raise TransitionDoesNotExist
            raise TransitionFailed
        else:
            transition.execute(**fields)

    def __call__(self, transition, **fields):
        super().__call__(transition=transition, **fields)

    def delay(self, transition, **fields):
        super().delay(transition=transition, **fields)


class IssueUpdateOperation(IssueBaseOperation):
    """
    Операция редактирования тикета
    """
    def execute(self, **fields):
        self._issue.update(**fields)


class IssueTransitionOrUpdateOperation(IssueTransitionOperation):
    """
    Операция перевода статуса в тикете в Трекере,
    которая в случае отсутствия нужного перехода не падает,
    а просто обновляет тикет тем, что есть (поля, коммент и т.п.)
    """
    def execute(self, transition, **fields):
        try:
            super().execute(transition, **fields)
        except TransitionDoesNotExist:
            self._issue.update(**fields)


class IssueCommentOperation(IssueUpdateOperation):
    """
    Операция добавления коммента в тикет.
    Работает не через стандартное создание коммента,
    а через редактирование тикета, потому что помимо коммента
    в тикет добавляется ID операции.
    """
    def execute(self, **fields):
        if not is_startrek_operations_enabled():
            self._issue.comments.create(**fields['comment'])
            return
        super().execute(**fields)

    def __call__(self, text, **fields):
        fields['text'] = text
        super().__call__(comment=fields)

    def delay(self, text, **fields):
        super().delay(text=text, **fields)
