import logging

from typing import List

from django.conf import settings
from ids.registry import registry
from ids.exceptions import BackendError
from yandex_tracker_client.objects import SeekablePaginatedList


logger = logging.getLogger(__name__)


class ContractTypesEnum:

    fixed = 'Срочный трудовой договор'
    open = 'Бессрочный трудовой договор'


class ProLevelEnum:

    intern = 'Стажер'
    junior = 'Младший'
    middle = 'Специалист'
    senior = 'Старший'
    lead = 'Ведущий'
    expert = 'Эксперт'

    intern_en = 'Intern'
    junior_en = 'Junior'
    middle_en = 'Middle'
    senior_en = 'Senior'
    lead_en = 'Lead'
    expert_en = 'Expert'


class StatusEnum:

    draft = 'draft'
    in_progress = 'inProgress'
    new = 'new'
    open = 'open'
    hr_approval = 'hrapproval'
    resolved = 'resolved'
    on_hold = 'onHold'
    closed = 'closed'
    validated = 'validated'


class ResolutionEnum:

    fixed = 'fixed'
    declined = 'declined'
    wont_fix = 'won\'tFix'
    transfer = 'transfer'


class InternshipEnum:

    yes = 'yes'
    no = 'no'


class TransitionEnum:

    added_to_staff = 'addedToStaff'
    agree_offer = 'agreeOffer'
    approved_by_head = 'approvedByHead'
    cancel = 'cancel'
    close = 'close'
    close_internal = 'closeInternal'
    closed = 'closed'
    confirmed = 'confirmed'
    declined = 'declined'
    employment = 'employment'
    freeze = 'freeze'
    hr_approval = 'hrapproval'
    in_progress = 'inProgress'
    need_info = 'need_info'
    new = 'new'
    offer_accepted = 'offerAccepted'
    offer_declined = 'offerDeclined'
    on_hold = 'onHold'
    open = 'open'
    refused = 'refused'
    resolve = 'resolve'
    start_progress = 'start_progress'
    validated = 'validated'


class IssueTypeEnum:

    new_employee = 'newEmployee'
    remove_process = 'removeProcess'
    service_request = 'serviceRequest'


class ApprovementStatusEnum:

    in_progress = 'Запущено'
    suspended = 'Приостановлено'
    approved = 'Согласовано'
    declined = 'Отменено'


class SalarySystemEnum:

    fixed = 'Фиксированная'
    piecework = 'Сдельная'
    hourly = 'Почасовая'

    fixed_en = 'Fixed'
    piecework_en = 'Piecework'
    hourly_en = 'Hourly'


class StartrekError(Exception):
    code = 'startrek_error'
    message = 'startrek_error'


class KnownStartrekError(StartrekError):
    pass


class IssueDoesNotExist(KnownStartrekError):
    message = 'startrek_issue_not_found'


class IssuePermissionDenied(KnownStartrekError):
    message = 'startrek_issue_permission_denied'


class TransitionFailed(KnownStartrekError):
    message = 'startrek_transition_failed'


class IssueInvalidStatus(KnownStartrekError):
    message = 'startrek_issue_status_invalid'


class TransitionDoesNotExist(TransitionFailed):
    pass


def _get_repository(resource_type, **kwargs):
    return registry.get_repository(
        service='startrek2',
        resource_type=resource_type,
        user_agent='femida',
        oauth_token=settings.FEMIDA_ROBOT_TOKEN,
        **kwargs
    )


def _get_issues_repository(**kwargs):
    return _get_repository('issues', **kwargs)


def _get_fields_repository():
    return _get_repository('fields')


def _get_field(field_name):
    repo = _get_fields_repository()
    return repo.startrek_client_resourses.get(field_name)


def get_startrek_field_values(field_name) -> List:
    field = _get_field(field_name)
    return getattr(field, 'optionsProvider', {}).get('values', [])


def _fetch_issue_fields(issue):
    """
    Загрузить в тикет информацию о типах его полей.

    Библиотека ids отдает тикет без мета-информации о типах его полей.
    При первом обращении к какому-либо полю производится запрос о типах всех полей.
    Данная функция позволяет предзагрузить их и сгруппировать все запросы в трекер в одном месте.
    """
    try:
        issue.key
    except BackendError as exc:
        logger.exception('Error fetching issue fields from StarTrek API')
        raise StartrekError(*exc.args)


def _get_issue(issues, key):
    try:
        issue = issues.startrek_client_resourses.get(key)
    except BackendError as exc:
        if exc.status_code == 404:
            raise IssueDoesNotExist
        elif exc.status_code == 403:
            raise IssuePermissionDenied

        logger.exception('Error getting issue from StarTrek API')
        raise StartrekError(*exc.args)
    else:
        if isinstance(issue, SeekablePaginatedList):
            raise IssueDoesNotExist
        return issue


def get_issue(key):
    issues = _get_issues_repository()
    issue = _get_issue(issues, key)
    _fetch_issue_fields(issue)
    return issue


def _process_issue(keys, callback, callback_description=''):
    if not keys:
        return

    if isinstance(keys, str):
        keys = [keys]

    issues = _get_issues_repository()

    for key in keys:
        issue = _get_issue(issues, key)
        if not issue:
            continue

        try:
            callback(issue)
        except BackendError as exc:
            logger.exception(
                'Error processing the issue %s (%s)\n%s',
                key, callback_description, exc.extra,
            )
            raise StartrekError(*exc.args)


def create_issue(queue, summary, description, author=None, timeout=None, **fields):
    """
    Создает тикет в Startrek
    """

    repository_params = {}
    if timeout:
        repository_params['timeout'] = timeout

    issues = _get_issues_repository(**repository_params)

    params = dict(fields)
    params.update({
        'queue': queue,
        'summary': summary,
        'description': description,
    })
    if author is not None:
        params['author'] = author

    try:
        issue = issues.create(**params)
    except BackendError as exc:
        logger.exception('Error during create issue')
        raise StartrekError(*exc.args)

    _fetch_issue_fields(issue)
    return issue


def add_comment(keys, text, **fields):
    """
    Отправляет комментарий в тикет Startrek
    """
    def callback(issue):
        issue.comments.create(text=text, **fields)

    _process_issue(
        keys=keys,
        callback=callback,
        callback_description='adding comment',
    )


def update_issue(keys, **fields):
    """
    Редактирует тикет Startrek
    """
    def callback(issue):
        issue.update(**fields)

    _process_issue(
        keys=keys,
        callback=callback,
        callback_description='updating',
    )
