import logging

from django.conf import settings
from django.utils.functional import cached_property

from intranet.femida.src.core.models import City
from intranet.femida.src.candidates.choices import CONSIDERATION_STATUSES_TRANSLATIONS
from intranet.femida.src.staff.models import Department
from intranet.femida.src.interviews.choices import (
    APPLICATION_RESOLUTIONS_TRANSLATIONS,
    INTERVIEW_TYPES_TRANSLATIONS,
)
from intranet.femida.src.offers.choices import ST_PROGRAMMING_LANGUAGES, SOURCES_TRANSLATIONS
from intranet.femida.src.professions.models import Profession, ProfessionalSphere
from intranet.femida.src.vacancies.choices import VACANCY_RESOLUTIONS_TRANSLATIONS
from intranet.femida.src.stats import choices
from intranet.femida.src.stats.registry import register_dictionary


logger = logging.getLogger(__name__)


class Dictionary:
    """
    TODO: Добавить описание словарям (description в upload)
    """
    name = None
    is_dynamic = False

    @cached_property
    def api(self):
        from intranet.femida.src.stats.statface import statface
        full_name = 'femida_%s' % self.name
        return statface.get_stat_dict(full_name)

    def download(self):
        """
        Загружает словарь из Statface
        :return:
        """
        return self.api.download()

    def upload(self):
        """
        Заливает словарь на Statface
        """
        return self.api.upload(self.get())

    def get(self):
        """
        Формирует словарь
        :return:
        """
        return {}


@register_dictionary
class DepartmentsDict(Dictionary):

    name = 'departments_dict'
    is_dynamic = True

    def get(self):
        return dict(Department.objects.values_list('id', 'name'))


@register_dictionary
class BooleanDict(Dictionary):

    name = 'boolean'

    def get(self):
        return {
            'True': 'Да',
            'False': 'Нет',
        }


@register_dictionary
class CitiesDict(Dictionary):

    name = 'cities_dict'
    is_dynamic = True

    def get(self):
        return dict(City.objects.values_list('id', 'name_ru'))


@register_dictionary
class HiringTypesDict(Dictionary):

    name = 'hiring_types_dict'

    def get(self):
        return {
            'regular': 'Обычный',
            'rotation': 'Ротация',
        }


@register_dictionary
class ProfessionalSpheresDict(Dictionary):

    name = 'professional_spheres_dict'
    is_dynamic = True

    def get(self):
        return dict(ProfessionalSphere.objects.values_list('id', 'name'))


@register_dictionary
class ProfessionsAndSpheresDict(Dictionary):

    name = 'professions_and_spheres_dict'
    is_dynamic = True

    def get(self):
        result = {}
        for _id, name in ProfessionalSphere.objects.values_list('id', 'name'):
            result['S%d' % _id] = name
        ARROW = '\u2192'
        for _id, s, p in Profession.objects.values_list('id', 'professional_sphere__name', 'name'):
            result['P%d' % _id] = '%s %s %s' % (s, ARROW, p)
        return result


@register_dictionary
class ProgrammingLanguagesDict(Dictionary):

    name = 'programming_languages'

    def get(self):
        result = dict(ST_PROGRAMMING_LANGUAGES)
        result['other_see_comments'] = 'Other'
        return result


@register_dictionary
class TimeslotsDict(Dictionary):

    name = 'timeslots_dict'

    def get(self):
        return choices.TIMESLOTS._display_map


@register_dictionary
class VacancyResolutionsDict(Dictionary):

    name = 'vacancy_resolutions_dict'

    def get(self):
        return VACANCY_RESOLUTIONS_TRANSLATIONS


@register_dictionary
class VacancyStartingPointsDict(Dictionary):

    name = 'vacancy_starting_points'

    def get(self):
        return choices.VACANCY_STARTING_POINTS._display_map


@register_dictionary
class VacancyTypesDict(Dictionary):

    name = 'vacancy_types_dict'

    def get(self):
        return {
            'regular': 'Обычная',
            'internship': 'Стажировка',
        }


@register_dictionary
class ConsiderationSourcesDict(Dictionary):

    name = 'consideration_sources'

    def get(self):
        return SOURCES_TRANSLATIONS


@register_dictionary
class ConsiderationResolutionsDict(Dictionary):

    name = 'consideration_resolutions'

    def get(self):
        return APPLICATION_RESOLUTIONS_TRANSLATIONS


@register_dictionary
class ConsiderationStatusesDict(Dictionary):

    name = 'consideration_statuses'

    def get(self):
        return CONSIDERATION_STATUSES_TRANSLATIONS


@register_dictionary
class AATypeDict(Dictionary):

    name = 'aa_types'

    def get(self):
        return settings.AA_TYPES_TRANSLATIONS


@register_dictionary
class InterviewTypesDict(Dictionary):

    name = 'interview_types'

    def get(self):
        return INTERVIEW_TYPES_TRANSLATIONS
