from collections import namedtuple

from django.contrib.auth import get_user_model
from django.db.models import Q
from django.utils.functional import cached_property

from intranet.femida.src.actionlog.models import LogRecord
from intranet.femida.src.stats import enums
from intranet.femida.src.stats.fetchers.base import ReportDataFetcher


User = get_user_model()


_user_types = (
    'anonymous',
    'recruiter',
    'other',
)
USER_TYPES = namedtuple('USER_TYPES', _user_types)(*_user_types)


class ActionlogDataFetcher(ReportDataFetcher):

    def get_data(self):
        actions_by_user_type = {}
        for user_type in USER_TYPES:
            actions_by_user_type[user_type] = {k: 0 for k in self._action_names}
            actions_by_user_type[user_type][enums.StatKeys.all] = 0

        log_records = (
            self._get_log_records()
            .values_list('action_name', 'user__username')
        )

        for action_name, username in log_records:
            user_type = self._get_user_type(username)
            count_by_actions = actions_by_user_type[user_type]
            count_by_actions[action_name] += 1
            count_by_actions[enums.StatKeys.all] += 1

        result = []
        for initiator_type in actions_by_user_type:
            for action_name in actions_by_user_type[initiator_type]:
                result.append({
                    'fielddate': self.fielddate,
                    'initiator_type': initiator_type,
                    'action_name': action_name,
                    'count': actions_by_user_type[initiator_type][action_name],
                })

        return result

    @cached_property
    def _action_names(self):
        return sorted(
            LogRecord.objects
            .values_list('action_name', flat=True)
            .distinct()
        )

    @cached_property
    def _recruiters(self):
        is_recruiter_q = (
            Q(user_permissions__codename='recruiter_perm')
            | Q(groups__permissions__codename='recruiter_perm')
        )
        return set(
            User.objects
            .filter(is_recruiter_q)
            .values_list('username', flat=True)
        )

    def _get_log_records(self):
        filter_params = {}
        if self.from_dt is not None:
            filter_params['action_time__gte'] = self.from_dt
        if self.to_dt is not None:
            filter_params['action_time__lte'] = self.to_dt
        return LogRecord.objects.filter(**filter_params)

    def _get_user_type(self, username):
        if username is None:
            return USER_TYPES.anonymous
        if username in self._recruiters:
            return USER_TYPES.recruiter
        return USER_TYPES.other
