import logging
import itertools

from collections import defaultdict

from django.conf import settings
from django.contrib.auth import get_user_model
from django.utils import timezone
from django.utils.functional import cached_property
from statface_client import DAILY_SCALE, WEEKLY_SCALE

from intranet.femida.src.professions.models import Profession
from intranet.femida.src.staff.models import Department
from intranet.femida.src.stats import enums
from intranet.femida.src.stats.utils import (
    get_beginning_of_moscow_day,
    str_to_moscow_date,
    StaffUnit,
    ProfessionUnit,
)


logger = logging.getLogger(__name__)

User = get_user_model()


class ReportDataFetcher:

    dt_format = '%Y-%m-%d'
    report_name = None  # Название отчета в URL
    title = None  # Человекочитаемое название отчета
    config_name = None
    scale = DAILY_SCALE
    delta = None

    def __init__(self, config, fielddate=None, delta=None, **options):
        self.config = config
        self.options = options
        self._setup_dates(fielddate, delta)

    def _setup_dates(self, fielddate, delta):
        periods = {
            WEEKLY_SCALE: timezone.timedelta(days=7),
            DAILY_SCALE: timezone.timedelta(days=1),
        }

        if fielddate is None:
            self.to_dt = get_beginning_of_moscow_day(timezone.now())
            period_start = self.to_dt - periods[self.scale]
            self.fielddate = period_start.strftime(self.dt_format)
        else:
            self.fielddate = fielddate
            dt = str_to_moscow_date(fielddate)
            if self.scale == WEEKLY_SCALE:
                time_until_period_end = timezone.timedelta(days=7 - dt.weekday())
            else:
                time_until_period_end = timezone.timedelta(days=1)

            self.to_dt = dt + time_until_period_end

        if delta is not None:
            self.from_dt = self.to_dt - delta
        else:
            self.from_dt = self.to_dt - timezone.timedelta(days=1)

    @cached_property
    def dimensions(self):
        result = self.config.dimensions.copy()
        result.pop('fielddate')
        return result

    @cached_property
    def measures(self):
        return self.config.measures

    def get_data(self):
        return {}


class HierarchicReportDataFetcher(ReportDataFetcher):
    """
    Умеет агрегировать данные по дереву департаментов (1) и по измерениям (2).
    (1) Для измерения departament или staffunit (дерево департаментов с user в качестве листа)
    показатели аггрегируются вверх по иерархии.
    (2) Для измерений с несколькими значеними (тип вакансии, проф. сфера) добавляетя значение ALL
    """
    # id корневых департаментов. Если какой-то департамент не внутри исходных, то он
    # считается как UNKNOWN_DEPARTMENT
    department_ids = [
        settings.YANDEX_DEPARTMENT_ID,
        settings.EXTERNAL_DEPARTMENT_ID,
    ]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.initialize_result()
        self.username_to_department_id = dict(User.objects.values_list('username', 'department'))

    def initialize_result(self):
        """
        Инициализируем словарь, в котором будут содержаться все полученные данные.
        Ключи словаря - кортежы срезов
        Значения - словари с показателями
        """
        self.result = defaultdict(lambda: {m: 0 for m in self.measures})

    @cached_property
    def dep_ancestors_map(self):
        """
        Словарь - dep_id: ancestors
        """
        result = dict(
            Department.objects
            .in_trees(self.department_ids)
            .values_list('id', 'ancestors')
        )
        result[enums.StatKeys.unknown_department] = []
        return result

    @cached_property
    def dep_url_to_id(self):
        return dict(
            Department.objects
            .in_trees(self.department_ids)
            .values_list('url', 'id')
        )

    @cached_property
    def profession_to_prof_spheres_map(self):
        """
        Словарь - profession: professional_sphere
        """
        return dict(
            Profession.objects
            .values_list(
                'id',
                'professional_sphere_id',
            )
        )

    def get_related_keys(self, key):
        """
        Для каждого кортежа со срезом получаем все связанные срезы, которые так же нужно учесть
        """
        assert len(key) == len(self.dimensions)

        dimensions = []
        for item in key:
            if isinstance(item, ProfessionUnit):
                dimensions.append(self.get_prof_units_chain(item))
            elif isinstance(item, StaffUnit):
                dimensions.append(self.get_staff_units_chain(item))
            else:
                dimensions.append([item, enums.StatKeys.all])

        return itertools.product(*dimensions)

    def get_staff_units_chain(self, staff_unit):
        """
        :type staff_unit: StaffUnit
        :return: Восстановленный к корню список подразделений/сотрудников
        """
        username = None
        if staff_unit is None or staff_unit.id == enums.StatKeys.unknown_department:
            department_id = enums.StatKeys.unknown_department
        elif staff_unit.type == enums.StaffUnitTypes.department:
            department_id = staff_unit.id
        else:
            department_id = self.username_to_department_id[staff_unit.id]
            username = staff_unit.id

        if department_id not in self.dep_ancestors_map:
            department_id = enums.StatKeys.unknown_department

        result = [
            StaffUnit(dep_id, enums.StaffUnitTypes.department)
            for dep_id in self.dep_ancestors_map[department_id]
        ]
        result.append(StaffUnit(department_id, enums.StaffUnitTypes.department))

        if username and department_id != enums.StatKeys.unknown_department:
            result.append(StaffUnit(username, enums.StaffUnitTypes.user))
        return result

    def get_prof_units_chain(self, prof_unit):
        result = [enums.StatKeys.all, prof_unit]
        if prof_unit.type == enums.ProfessionUnitTypes.profession:
            result.append(
                ProfessionUnit(
                    id=self.profession_to_prof_spheres_map[prof_unit.id],
                    type=enums.ProfessionUnitTypes.professional_sphere,
                )
            )
        return result

    def get_transformed_item(self, key):
        """
        Ключ-значение из преподсчитанного словаря трансформирует в строку для аплоада в Stat
        """
        result = {
            'fielddate': self.fielddate,
        }
        result.update(self.get_transformed_dimensions(key))
        result.update(self.get_transformed_measures(key))
        return result

    def get_transformed_dimensions(self, key):
        result = {}
        for item, (name, _type) in zip(key, self.dimensions.items()):
            if isinstance(item, ProfessionUnit):
                result[name] = item.key
            elif isinstance(item, StaffUnit):
                result[name] = [node.id for node in self.get_staff_units_chain(item)]
            else:
                result[name] = item
        return result

    def get_transformed_measures(self, key):
        return self.result[key]

    def get_transformed_measures_with_periods(self, key, non_period_measures=(),
                                              time_scale=enums.TimeScales.days):
        result = {}
        for k, v in self.result[key].items():
            if k in non_period_measures:
                result[k] = v
            else:
                td = v[0] / v[1] if v[1] else timezone.timedelta()
                result[k] = td.total_seconds() / time_scale
        return result

    def collect_data(self):
        raise NotImplementedError

    def get_data(self):
        self.collect_data()
        return [self.get_transformed_item(i) for i in self.result]
