import itertools

from collections import namedtuple, defaultdict, Counter
from datetime import timedelta

from django.conf import settings
from django.contrib.auth import get_user_model
from django.utils import timezone
from django.utils.dateparse import parse_datetime
from django.db.models import F, Q, OuterRef, Value, CharField, Exists, BooleanField
from django.utils.functional import cached_property

from intranet.femida.src.actionlog.models import Snapshot
from intranet.femida.src.candidates.models import (
    Candidate,
    CandidateResponsible,
    Consideration,
    ConsiderationHistory,
)
from intranet.femida.src.candidates.choices import (
    CANDIDATE_STATUSES,
    CONSIDERATION_EXTENDED_STATUSES,
)
from intranet.femida.src.candidates.controllers import get_active_consideration_subquery
from intranet.femida.src.core.db import RowsToList
from intranet.femida.src.interviews.models import Interview, Application
from intranet.femida.src.interviews.choices import (
    INTERVIEW_TYPES,
    APPLICATION_PROPOSAL_STATUSES,
    APPLICATION_SOURCES,
)
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.staff.models import Department
from intranet.femida.src.stats.fetchers.base import ReportDataFetcher, HierarchicReportDataFetcher
from intranet.femida.src.stats import choices, enums
from intranet.femida.src.stats.utils import (
    HIRING_TYPES,
    ROTATION_HIRING_TYPE_MAP,
    VacancyTypeExpression,
    StaffUnit,
)
from intranet.femida.src.core.db import MapperExpression


User = get_user_model()


_measures = (
    'screenings_finished',
    'regulars_finished',
    'finals_finished',
    'offers_sent',
    'offers_accepted',
    'offers_closed',
)
MEASURES = namedtuple('MEASURES', _measures)(*_measures)

_stats_vacancy_types = (
    'regular',
    'internship',
)
STATS_VACANCY_TYPES = namedtuple('STATS_VACANCY_TYPES', _stats_vacancy_types)(*_stats_vacancy_types)


INTERVIEW_TYPE_MEASURE_MAP = {
    INTERVIEW_TYPES.screening: MEASURES.screenings_finished,
    INTERVIEW_TYPES.regular: MEASURES.regulars_finished,
    INTERVIEW_TYPES.final: MEASURES.finals_finished,
}

OFFER_ACTION_MEASURE_MAP = {
    'offer_send': MEASURES.offers_sent,
    'offer_accept': MEASURES.offers_accepted,
    'offer_close': MEASURES.offers_closed,
}


class CandidatesFunnelDataFetcher(ReportDataFetcher):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.result = defaultdict(lambda: {m: set() for m in MEASURES})
        # считаем только для подразделений в ветке Яндекс
        self.departments_qs = Department.objects.in_tree(settings.YANDEX_DEPARTMENT_ID)

    def get_data(self):
        self._collect_data()
        data = []
        for key in self.result:
            d, p, v, h = key
            row = self.result[key]
            measures = {k: len(v) for k, v in row.items()}
            data.append(dict(
                measures,
                fielddate=self.fielddate,
                department=list(self._get_departments_chain(d)),
                professional_sphere=str(p),
                vacancy_type=v,
                hiring_type=h,
            ))
        return data

    def _collect_data(self):
        collections = [
            self._get_interviews(),
            self._get_offers('offer_send'),
            self._get_offers('offer_accept'),
            self._get_offers('offer_close'),
        ]

        for collection in collections:
            for keys in collection:
                self._count_candidate(*keys)

    @cached_property
    def departments(self):
        """
        Словарь - dep_id: ancestors
        """
        result = dict(self.departments_qs.values_list('id', 'ancestors'))
        result[enums.StatKeys.unknown_department] = []
        return result

    def _get_departments_chain(self, department_id):
        """
        Все предки подразделения + само подразделение
        """
        if department_id not in self.departments:
            department_id = enums.StatKeys.unknown_department
        return itertools.chain(self.departments[department_id], [department_id])

    def _get_departments_q(self, prefix=''):
        """
        Условие поиска по подразделениям.
        Ищем только по подразделениям в ветке "Яндекс"
        и если подразделение не указано совсем.
        """
        return (
            Q(**{prefix + 'department__isnull': True})
            | Q(**{prefix + 'department__in': self.departments_qs})
        )

    def _count_candidate(self, department_id, prof_sphere_id, vacancy_type, hiring_type,
                         measure, candidate_id):
        dimensions = [
            self._get_departments_chain(department_id),
            [prof_sphere_id, enums.StatKeys.all],
            [vacancy_type, enums.StatKeys.all],
            [hiring_type, enums.StatKeys.all],
        ]
        for d, p, v, h in itertools.product(*dimensions):
            self.result[(d, p, v, h)][measure].add(candidate_id)

    def _get_snapshot_ids(self, obj_str, action_name):
        return (
            Snapshot.objects
            .filter(
                obj_str=obj_str,
                log_record__action_name=action_name,
                log_record__action_time__gte=self.from_dt,
                log_record__action_time__lt=self.to_dt,
            )
            .values_list('obj_id', flat=True)
        )

    # Коллекции.
    # Все коллекции должны возвращать iterable со списками с форматом:
    # подразделение, проф.сфера, тип вакансии, показатель, id кандидата

    def _get_interviews(self):
        return (
            Interview.unsafe
            .filter(
                self._get_departments_q('application__vacancy__'),
                finished__gte=self.from_dt,
                finished__lt=self.to_dt,
                type__in=INTERVIEW_TYPE_MEASURE_MAP.keys(),
            )
            .values_list(
                'application__vacancy__department_id',
                'application__vacancy__professional_sphere_id',
                VacancyTypeExpression('application__vacancy__type'),
                MapperExpression(
                    'consideration__is_rotation',
                    mapping=ROTATION_HIRING_TYPE_MAP,
                    output_field=BooleanField(),
                ),
                MapperExpression(
                    'type',
                    mapping=INTERVIEW_TYPE_MEASURE_MAP,
                    output_field=CharField(),
                ),
                'candidate_id',
            )
        )

    def _get_offers(self, action_name):
        offer_ids = self._get_snapshot_ids('offer', action_name)
        measure = OFFER_ACTION_MEASURE_MAP.get(action_name)
        return (
            Offer.unsafe
            .filter(
                self._get_departments_q('vacancy__'),
                id__in=offer_ids,
            )
            .values_list(
                'vacancy__department_id',
                'vacancy__professional_sphere_id',
                VacancyTypeExpression('vacancy__type'),
                # Для офферов считаем, что любой наём считается внешним
                Value(HIRING_TYPES.regular, CharField(max_length=32)),
                Value(measure, CharField(max_length=32)),
                'candidate_id',
            )
        )


class CandidatesQueueDataFetcher(HierarchicReportDataFetcher):

    def initialize_result(self):
        self.result = defaultdict(lambda: {m: set() for m in self.measures})

    def _get_usernames(self, instance):
        if instance['responsibles_attr']:
            return {i['username'] for i in instance['responsibles_attr']}
        return set()

    def _get_last_update_type(self, dt):
        now = timezone.now()
        if dt > now - timedelta(days=14):
            return choices.TIMESLOTS.less_than_2_weeks
        elif dt > now - timedelta(days=28):
            return choices.TIMESLOTS.between_2_and_4_weeks
        else:
            return choices.TIMESLOTS.more_than_4_weeks

    def get_transformed_measures(self, key):
        return {
            'count': len(self.result[key]['count']),
        }

    def _get_qs(self):
        qs = (
            Candidate.unsafe.alive()
            .filter(status=CANDIDATE_STATUSES.in_progress)
        )
        qs = (
            qs.annotate(
                responsibles_attr=RowsToList(
                    CandidateResponsible.objects
                    .filter(candidate_id=OuterRef('id'))
                    .values(username=F('user__username'))
                ),
                active_consideration=get_active_consideration_subquery(),
            )
            .values(
                'id',
                'modified',
                'responsibles_attr',
                'active_consideration',
            )
        )
        return qs

    def collect_data(self):
        candidates = self._get_qs()
        for cand in candidates:
            usernames = self._get_usernames(cand)
            last_update_type = self._get_last_update_type(cand['modified'])
            if usernames:
                keys = []
                for username in usernames:
                    keys.extend(self.get_related_keys((
                        StaffUnit(username, enums.StaffUnitTypes.user),
                        last_update_type,
                    )))
            else:
                keys = self.get_related_keys((
                    StaffUnit(enums.StatKeys.unknown_department, enums.StaffUnitTypes.department),
                    last_update_type,
                ))

            for key in keys:
                self.result[key]['count'].add(cand['id'])


class ConsiderationsQueueDataFetcher(CandidatesQueueDataFetcher):
    """
    Этот отчет должен заменить CandidatesQueueDataFetcher.
    Но пока пусть будет сбоку.
    """
    def collect_data(self):
        candidates = self._get_qs()
        for cand in candidates:
            usernames = self._get_usernames(cand)
            last_update_type = self._get_last_update_type(cand['modified'])
            if cand['active_consideration'] is not None:
                extended_status = cand['active_consideration']['extended_status']
            else:
                extended_status = enums.StatKeys.unknown
            if usernames:
                keys = []
                for username in usernames:
                    keys.extend(self.get_related_keys((
                        StaffUnit(username, enums.StaffUnitTypes.user),
                        last_update_type,
                        extended_status,
                    )))
            else:
                keys = self.get_related_keys((
                    StaffUnit(enums.StatKeys.unknown_department, enums.StaffUnitTypes.department),
                    last_update_type,
                    extended_status,
                ))

            for key in keys:
                self.result[key]['count'].add(cand['id'])


class ProposalsConversionDataFetcher(ReportDataFetcher):

    def _get_qs(self):
        appls_qs = (
            Application.unsafe
            .filter(
                consideration=OuterRef('id'),
                source=APPLICATION_SOURCES.proposal,
            )
            .annotate(
                has_interviews=Exists(
                    Interview.unsafe
                    .filter(
                        application=OuterRef('id'),
                        state=Interview.STATES.finished,
                        type=INTERVIEW_TYPES.regular,
                    )
                    .values('id')
                )
            )
            .values(
                'id',
                'proposal_status',
                'has_interviews',
            )
        )

        return (
            Consideration.unsafe
            .filter(
                state=Consideration.STATES.archived,
                finished__gte=self.from_dt,
                finished__lt=self.to_dt,
                id__in=(
                    Application.unsafe
                    .filter(source=APPLICATION_SOURCES.proposal)
                    .values('consideration_id')
                )
            )
            .annotate(appls=RowsToList(appls_qs))
            .values(
                'id',
                'appls',
            )
        )

    def get_data(self):
        considerations = self._get_qs()

        data = defaultdict(list)
        cons_count = 0
        for cons in considerations:
            cons_count += 1
            appls = cons['appls']
            total_count = accept_count = reject_count = has_interviews_count = 0
            for appl in appls:
                total_count += 1
                if appl['proposal_status'] == APPLICATION_PROPOSAL_STATUSES.accepted:
                    accept_count += 1
                elif appl['proposal_status'] == APPLICATION_PROPOSAL_STATUSES.rejected:
                    reject_count += 1
                if appl['has_interviews']:
                    has_interviews_count += 1
            total_count = float(total_count)

            data['appls_count'].append(total_count)
            data['accuracy'].append(accept_count / total_count)
            data['inaccuracy'].append(reject_count / total_count)
            if accept_count:
                data['interview_prob'].append(has_interviews_count / float(accept_count))

        result = {
            'fielddate': self.fielddate,
        }
        for measure in self.measures:
            if measure == 'cons_count':
                result[measure] = cons_count
            else:
                avg = sum(data[measure]) / len(data[measure]) if data[measure] else 0
                if measure == 'appls_count':
                    result[measure] = avg
                else:
                    result[measure] = avg * 100
        return result


class ProposalsFastConversionDataFetcher(ReportDataFetcher):

    def _get_handled_appl_ids(self):
        return set(
            Snapshot.objects
            .filter(
                log_record__action_name__in=(
                    'application_accept_proposal',
                    'application_reject_proposal',
                ),
                log_record__action_time__gte=self.from_dt,
                log_record__action_time__lt=self.to_dt,
                obj_str='application',
            )
            .values_list('obj_id', flat=True)
        )

    def get_data(self):
        appls_qs = (
            Application.unsafe
            .filter(
                source=APPLICATION_SOURCES.proposal,
                created__gte=self.from_dt,
                created__lt=self.to_dt,
            )
            .values('id', 'proposal_status', 'proposal_factors', 'consideration_id')
            .order_by('consideration_id')
        )
        handled_appl_ids = self._get_handled_appl_ids()
        cons_count = 0
        data = defaultdict(lambda: defaultdict(list))
        for cons_id, appls in itertools.groupby(appls_qs, lambda x: x['consideration_id']):
            counts = defaultdict(Counter)
            cons_count += 1
            for appl in appls:
                is_first_proposal = appl['proposal_factors'].get('is_first_proposal', True)
                counts[is_first_proposal]['total_count'] += 1
                counts[enums.StatKeys.all]['total_count'] += 1
                # Учитываем только те прет-ва, действие над которым было сделано в учитываемое время
                if appl['id'] not in handled_appl_ids:
                    continue

                for state in (is_first_proposal, enums.StatKeys.all):
                    if appl['proposal_status'] == APPLICATION_PROPOSAL_STATUSES.accepted:
                        counts[state]['accept_count'] += 1
                    elif appl['proposal_status'] == APPLICATION_PROPOSAL_STATUSES.rejected:
                        counts[state]['reject_count'] += 1

            for state in counts.keys():
                total_count = float(counts[state]['total_count'])
                data[state]['appls_count'].append(total_count)
                data[state]['accuracy'].append(counts[state]['accept_count'] / total_count)
                data[state]['inaccuracy'].append(counts[state]['reject_count'] / total_count)

        results = []
        for state in data.keys():
            state_data = data[state]
            result = {
                'fielddate': self.fielddate,
                'first_proposal': state,
            }
            for measure in self.measures:
                if measure == 'cons_count':
                    result[measure] = cons_count
                else:
                    avg = sum(state_data[measure]) / len(state_data[measure])
                    if measure == 'appls_count':
                        result[measure] = avg
                    else:
                        result[measure] = avg * 100
            results.append(result)
        return results


class CandidateProcessingSpeedDataFetcher(ReportDataFetcher):

    def _get_qs(self):
        return (
            Consideration.unsafe
            .filter(
                finished__gte=self.from_dt,
                finished__lt=self.to_dt,
            )
            .annotate(
                history=RowsToList(
                    ConsiderationHistory.objects
                    .filter(
                        consideration_id=OuterRef('id'),
                    )
                    .order_by('changed_at')
                    .values(
                        'status',
                        'changed_at',
                    )
                ),
            )
            .values(
                'id',
                'started',
                'finished',
                'history',
            )
        )

    def _process_cons(self, cons, durations):
        cons_started = cons['started']
        cons_finished = cons['finished']
        screening_assigned_time = None
        screening_finished_time = None
        onsite_assigned_time = None
        onsite_finished_time = None

        durations['start_to_finish'].append(cons_finished - cons_started)
        for item in cons['history'] or []:
            changed_at = parse_datetime(item['changed_at'])
            # Если назначили скайп
            if (item['status'] == CONSIDERATION_EXTENDED_STATUSES.screening_assigned
                    and screening_assigned_time is None):
                screening_assigned_time = changed_at
                durations['start_to_screening'].append(screening_assigned_time - cons_started)

            # Если скайп был назначен и завершился
            if (item['status'] == CONSIDERATION_EXTENDED_STATUSES.screening_finished
                    and screening_assigned_time
                    and screening_finished_time is None):
                screening_finished_time = changed_at
                durations['screening'].append(screening_finished_time - screening_assigned_time)

            # Если очки были назначены
            if (item['status'] == CONSIDERATION_EXTENDED_STATUSES.interview_assigned
                    and onsite_assigned_time is None):
                onsite_assigned_time = changed_at
                durations['start_to_onsite'].append(onsite_assigned_time - cons_started)
                # и скайп был завершен
                if screening_finished_time:
                    durations['screening_to_onsite'].append(
                        onsite_assigned_time - screening_finished_time
                    )

            # Если очки были завершены
            if (item['status'] == CONSIDERATION_EXTENDED_STATUSES.interview_finished
                    and onsite_finished_time is None):
                onsite_finished_time = changed_at
                durations['onsite_to_finish'].append(cons_finished - onsite_finished_time)
                # и очки были назначены
                if onsite_assigned_time:
                    durations['onsite'].append(onsite_finished_time - onsite_assigned_time)

    def get_data(self):
        qs = self._get_qs()
        durations = defaultdict(list)
        for cons in qs:
            self._process_cons(cons, durations)

        result = {
            'fielddate': self.fielddate,
        }
        for key, value in durations.items():
            result[key + '_count'] = len(value)
            avg = sum(value, timedelta()) / len(value)
            result[key + '_avg'] = float(avg.total_seconds()) / (24 * 60 * 60)
        return result
