from django.db.models import OuterRef, Exists

from intranet.femida.src.core.db.expressions import RowToDict
from intranet.femida.src.offers.choices import OFFER_STATUSES
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.candidates.models import Consideration
from intranet.femida.src.candidates.choices import CONSIDERATION_STATUSES
from intranet.femida.src.stats.fetchers.base import HierarchicReportDataFetcher

from intranet.femida.src.stats.utils import VacancyTypeExpression, ProfessionUnit
from intranet.femida.src.stats import enums


class ConsiderationSourcesToOfferDataFetcher(HierarchicReportDataFetcher):

    def _get_qs(self):
        qs = (
            Consideration.unsafe
            .filter(
                state=CONSIDERATION_STATUSES.archived,
                finished__gte=self.from_dt,
                finished__lt=self.to_dt,
                is_rotation=False,
            )
            .exclude(source='')
            .annotate(
                offer=RowToDict(
                    Offer.unsafe
                    .filter(
                        application__consideration=OuterRef('id'),
                        status__in=(OFFER_STATUSES.rejected, OFFER_STATUSES.closed),
                    )
                    .values(
                        'profession_id',
                        'grade',
                        vacancy_type=VacancyTypeExpression('vacancy__type'),
                    )[:1]
                ),
            )
            .filter(
                offer__isnull=False,
            )
            .values(
                'source',
                'offer',
            )
        )
        return qs

    def collect_data(self):
        qs = self._get_qs()
        for cons in qs:
            offer = cons['offer']
            keys = self.get_related_keys((
                cons['source'],
                ProfessionUnit(offer['profession_id'], enums.ProfessionUnitTypes.profession),
                offer['grade'],
                offer['vacancy_type'],
            ))
            for key in keys:
                self.result[key]['count'] += 1


class ConsiderationSourcesToNoOfferDataFetcher(HierarchicReportDataFetcher):

    def _get_qs(self):
        return (
            Consideration.unsafe
            .filter(
                state=CONSIDERATION_STATUSES.archived,
                finished__gte=self.from_dt,
                finished__lt=self.to_dt,
            )
            .exclude(resolution='')
            .exclude(source='')
            .annotate(
                is_offer_exist=Exists(
                    Offer.unsafe
                    .filter(
                        application__consideration=OuterRef('id'),
                        status__in=(OFFER_STATUSES.rejected, OFFER_STATUSES.closed),
                    )
                    .values('id')
                )
            )
            .filter(is_offer_exist=False)
            .values(
                'source',
                'resolution',
            )
        )

    def collect_data(self):
        qs = self._get_qs()
        for cons in qs:
            keys = self.get_related_keys((
                cons['source'],
                cons['resolution'],
            ))
            for key in keys:
                self.result[key]['count'] += 1
