from collections import defaultdict
from copy import deepcopy
from datetime import timedelta
from itertools import groupby

from django.conf import settings
from django.db.models import Count

from intranet.femida.src.hire_orders.models import HireOrder, HireOrderHistory
from intranet.femida.src.hire_orders.choices import HIRE_ORDER_STATUSES
from intranet.femida.src.stats.fetchers.base import HierarchicReportDataFetcher
from intranet.femida.src.stats import enums
from intranet.femida.src.stats.utils import StaffUnit
from statface_client.constants import WEEKLY_SCALE


class HireOrdersQueueDataFetcher(HierarchicReportDataFetcher):

    department_ids = [
        settings.OUTSTAFF_DEPARTMENT_ID,
    ]

    def collect_data(self):
        queryset = (
            HireOrder.objects
            .exclude(status=HIRE_ORDER_STATUSES.closed)
            .values_list(
                'status',
                'raw_data__vacancy__department',
            )
            .annotate(count=Count('id'))
        )
        for status, department_url, count in queryset:
            department_id = self.dep_url_to_id.get(department_url)
            staff_unit = StaffUnit(department_id, enums.StaffUnitTypes.department)
            keys = self.get_related_keys((staff_unit,))
            for key in keys:
                self.result[key][status] += count


class HireOrdersProcessingSpeedDataFetcher(HierarchicReportDataFetcher):

    scale = WEEKLY_SCALE
    department_ids = [
        settings.OUTSTAFF_DEPARTMENT_ID,
    ]

    def initialize_result(self):
        default_measures = {
            m: 0 if m in self.non_period_measures else [timedelta(), 0]
            for m in self.measures
        }
        self.result = defaultdict(lambda: deepcopy(default_measures))

    def collect_data(self):
        queryset = (
            HireOrderHistory.objects
            .filter(
                hire_order__status=HIRE_ORDER_STATUSES.closed,
                hire_order_id__in=(
                    HireOrderHistory.objects
                    .filter(
                        changed_at__gte=self.from_dt,
                        changed_at__lt=self.to_dt,
                        status=HIRE_ORDER_STATUSES.closed,
                    )
                    .values('hire_order')
                )
            )
            .order_by('hire_order_id', 'changed_at')
            .values(
                'hire_order_id',
                'status',
                'resolution',
                'changed_at',
                'hire_order__raw_data__vacancy__department',
            )
        )
        for hire_order_id, group in groupby(queryset, lambda x: x['hire_order_id']):
            group_list = list(group)
            item = group_list[0]
            department_url = item['hire_order__raw_data__vacancy__department']
            department_id = self.dep_url_to_id.get(department_url)
            staff_unit = StaffUnit(department_id, enums.StaffUnitTypes.department)
            keys = list(self.get_related_keys((staff_unit,)))

            for key in keys:
                resolution = group_list[-1]['resolution']
                self.result[key][f'{resolution}_count'] += 1

            for i in range(len(group_list) - 1):
                curr, next = group_list[i], group_list[i + 1]
                for key in keys:
                    measure = curr['status'] + '_avg'
                    self.result[key][measure][0] += next['changed_at'] - curr['changed_at']
                    self.result[key][measure][1] += 1

    @property
    def non_period_measures(self):
        return {m for m in self.measures if not m.endswith('avg')}

    def get_transformed_measures(self, key):
        return self.get_transformed_measures_with_periods(
            key=key,
            non_period_measures=self.non_period_measures,
            time_scale=enums.TimeScales.seconds,
        )
