from datetime import timedelta

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db.models import F, BooleanField

from intranet.femida.src.actionlog.models import Snapshot
from intranet.femida.src.interviews.choices import (
    INTERVIEW_TYPES,
    INTERVIEW_GRADABLE_TYPES,
    AA_TYPES,
)
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.stats.fetchers.base import ReportDataFetcher, HierarchicReportDataFetcher
from intranet.femida.src.stats.utils import (
    VacancyTypeExpression,
    ROTATION_HIRING_TYPE_MAP,
    StaffUnit,
    ProfessionUnit,
)
from intranet.femida.src.stats import enums, choices
from intranet.femida.src.users.utils import filter_users_by_aa_type
from intranet.femida.src.core.db import MapperExpression


User = get_user_model()


class InterviewersWorkloadDataFetcher(HierarchicReportDataFetcher):

    department_ids = [
        settings.YANDEX_DEPARTMENT_ID,
    ]

    def collect_data(self):
        one_month_ago = self.to_dt - timedelta(days=30)
        six_months_ago = self.to_dt - timedelta(days=6 * 30)

        types = list(INTERVIEW_GRADABLE_TYPES._db_values) + [INTERVIEW_TYPES.final]
        interviews = (
            Interview.unsafe
            .alive()
            .filter(type__in=types, created__gte=six_months_ago)
            .values(
                'type',
                'created',
                'event_start_time',
                'finished',
                'interviewer__username',
            )
        )
        for interview in interviews:
            keys = self.get_related_keys((
                StaffUnit(interview['interviewer__username'], enums.StaffUnitTypes.user),
            ))

            measures = []

            interview_date = interview['event_start_time'] or interview['finished']
            is_future_interview = (
                interview['created'] >= self.to_dt
                or interview_date is None
                or interview_date >= self.to_dt
            )
            is_final = interview['type'] == INTERVIEW_TYPES.final

            if is_future_interview:
                prefix = 'final' if is_final else 'total'
                measures.append(prefix + '_next')
            else:
                if interview_date >= one_month_ago:
                    measures.append(interview['type'] + '_30d')

                if not is_final:
                    if interview_date >= six_months_ago:
                        measures.append('total_6m')

                    if interview_date >= one_month_ago:
                        measures.append('total_30d')

            for key in keys:
                for measure in measures:
                    self.result[key][measure] += 1

        for key in self.result:
            # Получаем примерно среднее за 7 дней
            self.result[key]['rate'] = self.result[key]['total_30d'] // 4


class AAInterviewersDataFetcher(ReportDataFetcher):

    def _get_interviews_data(self, aa_type, interviewers):
        return (
            Interview.unsafe.alive()
            .filter(
                type=INTERVIEW_TYPES.aa,
                aa_type=aa_type,
                created__gte=self.six_months_ago,
                interviewer__username__in=interviewers,
            )
            .values(
                'interviewer__username',
                'created',
                'event_start_time',
                'finished',
                'state',
            )
        )

    def get_data(self):
        self.week_ago = self.to_dt - timedelta(days=7)
        self.two_months_ago = self.to_dt - timedelta(days=60)
        self.six_months_ago = self.to_dt - timedelta(days=180)

        data = []
        for aa_type in AA_TYPES._db_values:
            data.extend(self._get_data_for_aa_type(aa_type))
        return data

    def _get_data_for_aa_type(self, aa_type):
        res = []
        aa_usernames = list(
            filter_users_by_aa_type(
                queryset=(
                    User.objects
                    .filter(is_dismissed=False)
                    .values_list('username', flat=True)
                ),
                aa_type=aa_type,
            )
        )
        interviews_data = self._get_interviews_data(aa_type, aa_usernames)
        aa_interviewers_counts = {
            username: {
                'next_7d_count': 0,
                'prev_7d_count': 0,
                'prev_60d_count': 0,
                'prev_180d_count': 0,
            }
            for username in aa_usernames
        }
        for i in interviews_data:
            username = i['interviewer__username']
            user_counts = aa_interviewers_counts[username]
            interview_date = i['event_start_time'] or i['finished']

            if i['created'] >= self.to_dt or interview_date is None or interview_date >= self.to_dt:
                user_counts['next_7d_count'] += 1
                continue

            if interview_date >= self.six_months_ago:
                user_counts['prev_180d_count'] += 1
            if interview_date >= self.two_months_ago:
                user_counts['prev_60d_count'] += 1
            if interview_date >= self.week_ago:
                user_counts['prev_7d_count'] += 1

        for username, data in aa_interviewers_counts.items():
            data['username'] = username
            data['rate'] = data['prev_60d_count'] / 8.0  # Чтобы получить примерно среднее за 7 дней
            data['fielddate'] = self.fielddate
            data['aa_type'] = aa_type
            res.append(data)

        return res


class InterviewCountsDataFetcher(HierarchicReportDataFetcher):

    department_ids = [
        settings.YANDEX_DEPARTMENT_ID,
    ]

    def collect_data(self):
        interviews = (
            Interview.unsafe
            .filter(
                state=Interview.STATES.finished,
                finished__gte=self.from_dt,
                finished__lt=self.to_dt
            )
            .values(
                'type',
                department_id=F('interviewer__department_id'),
                prof_sphere_id=F('application__vacancy__professional_sphere_id'),
                vacancy_type=VacancyTypeExpression('application__vacancy__type'),
                hiring_type=MapperExpression(
                    'consideration__is_rotation',
                    mapping=ROTATION_HIRING_TYPE_MAP,
                    output_field=BooleanField(),
                ),
            )
        )
        for interview in interviews:
            if interview['type'] == INTERVIEW_TYPES.aa:
                prof_sphere = settings.DEVELOPMENT_PROF_SPHERE_ID
            else:
                prof_sphere = interview['prof_sphere_id'] or enums.StatKeys.unknown
            keys = self.get_related_keys((
                StaffUnit(interview['department_id'], enums.StaffUnitTypes.department),
                prof_sphere,
                interview['vacancy_type'],
                interview['hiring_type'],
            ))
            measures = [interview['type'] + '_count', 'total_count']
            for key in keys:
                for measure in measures:
                    self.result[key][measure] += 1


class InterviewsProcessingSpeedDataFetcher(HierarchicReportDataFetcher):

    INTERVALS = (
        (timedelta(minutes=15), choices.TIMESLOTS.less_than_15_min),
        (timedelta(minutes=60), choices.TIMESLOTS.between_15_and_60_min),
        (timedelta(hours=4), choices.TIMESLOTS.between_1_and_4_hours),
        (timedelta(hours=12), choices.TIMESLOTS.between_4_and_12_hours),
        (timedelta(hours=36), choices.TIMESLOTS.between_12_and_36_hours),
        (timedelta.max, choices.TIMESLOTS.more_than_36_hours),
    )

    def _get_interval(self, delta):
        for _max, _str in self.INTERVALS:
            if delta <= _max:
                return _str

    def collect_data(self):
        interviews = (
            Interview.unsafe
            .filter(
                state=Interview.STATES.finished,
                finished__gte=self.from_dt,
                finished__lt=self.to_dt,
                type__in=(INTERVIEW_TYPES.regular, INTERVIEW_TYPES.aa),
            )
            .values(
                'id',
                'finished',
                'event_start_time',
                interviewer_username=F('interviewer__username'),
            )
        )
        estimation_snapshot_times_dict = _get_estimation_snapshot_times_dict(self.from_dt)
        for interview in interviews:
            if interview['event_start_time'] is None:
                continue
            interview['estimated'] = (
                estimation_snapshot_times_dict.get(interview['id'])
                or interview['finished']
            )
            # TODO: Хранить event_finish_time, потому что не всегда 1 час
            event_finish_time = interview['event_start_time'] + timedelta(hours=1)

            staff_unit = StaffUnit(interview['interviewer_username'], enums.StaffUnitTypes.user)
            for status in ('finished', 'estimated'):
                if interview[status] < event_finish_time:
                    continue
                interval = self._get_interval(interview[status] - event_finish_time)
                keys = self.get_related_keys((staff_unit, interval))
                for key in keys:
                    self.result[key][status + '_count'] += 1


class InterviewsProcessingSpeedDetailedDataFetcher(HierarchicReportDataFetcher):

    def collect_data(self):
        # Берем секции, завершенные за последние InterviewsProcessingSpeedDetailedReport.delta дней,
        # и вычисляем среди них, как долго они завершались, и как долго они оценивались
        qs = (
            Interview.unsafe
            .filter(
                state=Interview.STATES.finished,
                finished__gte=self.from_dt,
                finished__lt=self.to_dt,
            )
            .exclude(type=INTERVIEW_TYPES.hr_screening)
            .values(
                'id',
                'finished',
                'event_start_time',
                'type',
                interviewer_username=F('interviewer__username'),
                profession_id=F('application__vacancy__profession_id'),
            )
        )
        estimation_snapshot_times_dict = _get_estimation_snapshot_times_dict(self.from_dt)
        for interview in qs:
            staff_unit = StaffUnit(interview['interviewer_username'], enums.StaffUnitTypes.user)
            profession = (
                ProfessionUnit(interview['profession_id'], enums.ProfessionUnitTypes.profession)
                if interview['profession_id']
                else enums.StatKeys.unknown
            )
            keys = self.get_related_keys((
                staff_unit,
                profession,
                interview['type'],
            ))
            interview['estimated'] = (
                estimation_snapshot_times_dict.get(interview['id'])
                # Любое завершенное считаем оцененным
                or interview['finished']
            )
            for key in keys:
                cell = self.result[key]
                cell['finished_count'] += 1
                if interview['event_start_time'] is None:
                    continue
                event_date = interview['event_start_time'].date()
                for status in ('estimated', 'finished'):
                    processing_interval_days = (interview[status].date() - event_date).days
                    if processing_interval_days < 0:
                        cell[f'{status}_before_calendar_event_time'] += 1
                    elif processing_interval_days == 0:
                        cell[f'{status}_same_day_count'] += 1
                    elif processing_interval_days == 1:
                        cell[f'{status}_next_day_count'] += 1
                    else:
                        cell[f'{status}_after_next_day_count'] += 1


class InterviewEventTimeFetcher(HierarchicReportDataFetcher):

    def collect_data(self):
        qs = (
            Interview.unsafe
            .alive()
            .filter(
                created__gte=self.from_dt,
                created__lt=self.to_dt,
            )
            .exclude(type=INTERVIEW_TYPES.hr_screening)
            .values(
                'created',
                'event_start_time',
                'type',
                profession_id=F('application__vacancy__profession_id'),
                vacancy_type=F('application__vacancy__type'),
            )
        )
        for interview in qs:
            profession = (
                ProfessionUnit(interview['profession_id'], enums.ProfessionUnitTypes.profession)
                if interview['profession_id']
                else enums.StatKeys.unknown
            )
            keys = self.get_related_keys((
                profession,
                interview['type'],
                interview['vacancy_type'] or enums.StatKeys.unknown,
            ))
            for key in keys:
                self.result[key]['all_count'] += 1
                self.result[key]['no_event_time_count'] += (
                    interview['event_start_time'] is None
                )
                self.result[key]['event_time_before_created_count'] += (
                    interview['event_start_time'] is not None
                    and interview['event_start_time'] < interview['created']
                )


def _get_estimation_snapshot_times_dict(from_dt):
    return dict(
        Snapshot.objects.filter(
            log_record__action_time__gte=from_dt,
            log_record__action_name='interview_estimate',
            obj_str='interview',
        )
        .values_list(
            'obj_id',
            'log_record__action_time',
        )
    )
