from collections import defaultdict

from django.conf import settings
from django.db.models import F
from django.utils.functional import cached_property

from intranet.femida.src.actionlog.models import Snapshot
from intranet.femida.src.core.db import JsonF
from intranet.femida.src.offers.choices import (
    OFFER_STATUSES,
    DOCUMENT_TYPES,
    OFFER_DOCS_PROCESSING_STATUSES as DOCS_STATUSES,
    OFFER_DOCS_PROCESSING_RESOLUTIONS as DOCS_RESOLUTIONS,
    OFFER_DOCS_PROCESSING_SKIPPED_RESOLUTIONS as DOCS_SKIPPED_RESOLUTIONS,
)
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.stats import enums
from intranet.femida.src.stats.fetchers.base import ReportDataFetcher, HierarchicReportDataFetcher
from intranet.femida.src.stats.utils import (
    get_beginning_of_moscow_day,
    VacancyTypeExpression,
    StaffUnit,
    ProfessionUnit,
)


class EmployersDataFetcher(ReportDataFetcher):

    def _get_snapshots(self, from_dt, to_dt):
        filter_params = {
            'obj_str': 'offer',
            'log_record__action_name': 'offer_close',
        }
        if from_dt is not None:
            filter_params['log_record__action_time__gte'] = from_dt
        if to_dt is not None:
            filter_params['log_record__action_time__lte'] = to_dt
        return Snapshot.objects.filter(**filter_params)

    def _get_offers_data(self, offer_ids):
        return set(
            Offer.unsafe
            .filter(id__in=offer_ids)
            .exclude(current_company='')
            .exclude(candidate__login='')
            .values_list(
                'id',
                'candidate__login',
                'current_company',
            )
        )

    def get_data(self):
        res = []
        snapshots = self._get_snapshots(self.from_dt, self.to_dt)
        offer_dt_map = dict(snapshots.values_list('obj_id', 'log_record__action_time'))
        offers_data = self._get_offers_data(list(offer_dt_map))
        for _id, login, current_company in offers_data:
            res.append({
                'fielddate': get_beginning_of_moscow_day(offer_dt_map[_id]).strftime('%Y-%m-%d'),
                'username': login,
                'employer': current_company,
            })
        return res


class ExternalHireEfficiencyDataFetcher(HierarchicReportDataFetcher):

    def _process_offer(self, offer):
        if offer['profession_id'] is not None:
            profession = ProfessionUnit(
                offer['profession_id'],
                enums.ProfessionUnitTypes.profession,
            )
        else:
            profession = enums.StatKeys.unknown

        keys = list(
            self.get_related_keys((
                StaffUnit(offer['creator_username'], enums.StaffUnitTypes.user),
                profession,
                offer['vacancy_type'],
            ))
        )
        actions = self.offer_to_actions_map[offer['id']]
        for key in keys:
            for action in actions:
                self.result[key][self.actions_to_measure_map[action]] += 1

    def _get_snapshots(self):
        return (
            Snapshot.objects
            .filter(
                obj_str='offer',
                log_record__action_name__in=self.actions_to_measure_map.keys(),
                log_record__action_time__gte=self.from_dt,
                log_record__action_time__lt=self.to_dt,
            )
            .values_list(
                'obj_id',
                'log_record__action_name',
            )
        )

    def collect_data(self):
        self.actions_to_measure_map = {
            'offer_send': 'sent_count',
            'offer_accept': 'accepted_count',
            'offer_close': 'closed_count',
        }
        snapshots = self._get_snapshots()
        self.offer_to_actions_map = defaultdict(set)
        for offer_id, action_name in snapshots:
            self.offer_to_actions_map[offer_id].add(action_name)

        offers = (
            Offer.unsafe
            .filter(id__in=self.offer_to_actions_map.keys())
            .values(
                'id',
                'profession_id',
                creator_username=F('creator__username'),
                vacancy_type=VacancyTypeExpression('vacancy__type'),
            )
        )
        for offer in offers:
            self._process_offer(offer)


class OfferDocsProcessingFunnelDataFetcher(HierarchicReportDataFetcher):
    """
    FEMIDA-5665: Воронка обработки документов
    """
    department_ids = [
        settings.YANDEX_DEPARTMENT_ID,
        settings.OUTSTAFF_DEPARTMENT_ID,
    ]
    valid_statuses = (
        OFFER_STATUSES.accepted,
        OFFER_STATUSES.closed,
        OFFER_STATUSES.rejected,
        OFFER_STATUSES.deleted,
    )
    _unified_statuses = {
        OFFER_STATUSES.deleted: OFFER_STATUSES.rejected,
    }

    def collect_data(self):
        for offer in self._queryset:
            keys = self.get_related_keys((
                StaffUnit(offer.department_id, enums.StaffUnitTypes.department),
                bool(offer.docs_request_count),
            ))
            for key in keys:
                self._calculate(offer, key)

    def _calculate(self, offer, key):
        status = self._unified_statuses.get(offer.status, offer.status)
        dp_status = offer.docs_processing_status
        dp_resolution = offer.docs_processing_resolution
        is_person_created = offer.oebs_person_id is not None
        is_russian_passport = offer.document_type == DOCUMENT_TYPES.russian_passport

        def count(measure, condition=True, value=1):
            if condition:
                self.result[key][measure] += value

        count('total')
        count('total_sent_to_yang', dp_resolution not in DOCS_SKIPPED_RESOLUTIONS)
        count(f'{status}_{dp_status}', dp_status != DOCS_STATUSES.finished)
        count(dp_resolution, dp_status == DOCS_STATUSES.finished)
        count(f'{dp_resolution}_person_created', (
            is_person_created
            and dp_resolution in (DOCS_RESOLUTIONS.processed, DOCS_RESOLUTIONS.expired)
        ))
        count('processed_invalid_passport', (
            dp_resolution == DOCS_RESOLUTIONS.processed
            and not is_russian_passport
        ))
        count('processed_person_creation_failed', (
            dp_resolution == DOCS_RESOLUTIONS.processed
            and not is_person_created
            and is_russian_passport
        ))

    @cached_property
    def _queryset(self):
        return (
            Offer.unsafe
            .filter(
                id__in=self._checked_offer_ids,
                status__in=self.valid_statuses,
            )
            .annotate(
                document_type=JsonF('passport_data', 'document_type'),
            )
            .values_list(
                'status',
                'docs_processing_status',
                'docs_processing_resolution',
                'oebs_person_id',
                'department_id',
                'docs_request_count',
                'document_type',
                named=True,
            )
        )

    @cached_property
    def _checked_offer_ids(self):
        return set(
            Snapshot.objects
            .filter(
                obj_str='offer',
                log_record__action_name='offer_check_oebs_login',
                log_record__action_time__gte=self.from_dt,
                log_record__action_time__lt=self.to_dt,
            )
            .values_list('obj_id', flat=True)
        )
