from collections import defaultdict
from datetime import timedelta

from django.contrib.auth import get_user_model
from django.db.models import F
from django.utils import timezone

from intranet.femida.src.candidates.models import CandidateSubmission
from intranet.femida.src.candidates.choices import SUBMISSION_STATUSES
from intranet.femida.src.candidates.submissions.helpers import get_submission_vacancies_qs
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES
from intranet.femida.src.stats.fetchers.base import HierarchicReportDataFetcher
from intranet.femida.src.stats import choices, enums
from intranet.femida.src.stats.utils import StaffUnit


User = get_user_model()


class SubmissionsQueueDataFetcher(HierarchicReportDataFetcher):

    def initialize_result(self):
        self.result = defaultdict(lambda: {m: set() for m in self.measures})

    def _get_usernames(self, instance):
        usernames = set()
        for vacancy in get_submission_vacancies_qs(instance):
            if vacancy.status != VACANCY_STATUSES.in_progress:
                continue
            for recruiter in vacancy.recruiters:
                usernames.add(recruiter.username)
        return usernames

    def _get_lifetime(self, dt):
        now = timezone.now()
        if dt > now - timedelta(days=1):
            return choices.TIMESLOTS.less_than_1_day
        elif dt > now - timedelta(days=3):
            return choices.TIMESLOTS.between_1_and_3_days
        elif dt > now - timedelta(days=7):
            return choices.TIMESLOTS.between_3_and_7_days
        else:
            return choices.TIMESLOTS.more_than_7_days

    def get_transformed_measures(self, key):
        return {
            'count': len(self.result[key]['count']),
        }

    def collect_data(self):
        submissions = (
            CandidateSubmission.unsafe
            .filter(status=SUBMISSION_STATUSES.new)
            .prefetch_related('form__vacancies__memberships__member')
        )
        for submission in submissions:
            usernames = self._get_usernames(submission)
            lifetime = self._get_lifetime(submission.created)
            if usernames:
                keys = []
                for username in usernames:
                    keys.extend(self.get_related_keys((
                        StaffUnit(username, enums.StaffUnitTypes.user),
                        lifetime,
                    )))
            else:
                keys = self.get_related_keys((
                    StaffUnit(enums.StatKeys.unknown_department, enums.StaffUnitTypes.department),
                    lifetime,
                ))

            for key in keys:
                self.result[key]['count'].add(submission.id)


class SubmissionsProcessingSpeedDataFetcher(HierarchicReportDataFetcher):

    def _get_measure(self, submission):
        diff = submission['closed_at'] - submission['created']

        if diff < timedelta(days=1):
            return choices.TIMESLOTS.less_than_1_day
        if diff < timedelta(days=3):
            return choices.TIMESLOTS.between_1_and_3_days
        if diff < timedelta(days=7):
            return choices.TIMESLOTS.between_3_and_7_days
        else:
            return choices.TIMESLOTS.more_than_7_days

    def collect_data(self):
        submissions = (
            CandidateSubmission.unsafe
            .filter(
                closed_at__gte=self.from_dt,
                closed_at__lt=self.to_dt,
                form__is_internship=False,
            )
            .values(
                'created',
                'closed_at',
                responsible_username=F('responsible__username'),
            )
        )
        for submission in submissions:
            keys = self.get_related_keys((
                StaffUnit(submission['responsible_username'], enums.StaffUnitTypes.user),
            ))
            measure = self._get_measure(submission)
            measures = [measure, 'total_count']
            for key in keys:
                for measure in measures:
                    self.result[key][measure] += 1
