from itertools import groupby
from collections import defaultdict
from copy import deepcopy
from datetime import timedelta

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db.models import F, OuterRef, Min, Subquery, DateTimeField, Case, When, CharField, Value
from django.db.models.functions import Coalesce
from django.utils.dateparse import parse_datetime

from intranet.femida.src.offers.choices import OFFER_STATUSES
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES
from intranet.femida.src.vacancies.models import Vacancy, VacancyHistory
from intranet.femida.src.core.db import RowToDict, RowsToList
from intranet.femida.src.stats.fetchers.base import ReportDataFetcher, HierarchicReportDataFetcher
from intranet.femida.src.stats import choices, enums
from intranet.femida.src.stats.utils import VacancyTypeExpression, StaffUnit, ProfessionUnit

User = get_user_model()


GRADE_RANGES = (
    (0, 10, '<=10'),
    (11, 13, '11-13'),
    (14, 15, '14-15'),
    (16, 16, '16'),
    (17, 18, '17-18'),
    (19, 100, '19+'),
)


def _get_grade_range(field):
    conditions = [
        When(**{
            field + '__gte': _min,
            field + '__lte': _max,
            'then': Value(v),
        })
        for _min, _max, v in GRADE_RANGES
    ]

    return Case(
        output_field=CharField(),
        default=Value(enums.StatKeys.unknown),
        *conditions
    )


class VacancyCountsDataFetcher(HierarchicReportDataFetcher):

    department_ids = [
        settings.YANDEX_DEPARTMENT_ID,
        settings.OUTSTAFF_DEPARTMENT_ID,
    ]

    def initialize_result(self):
        self.result = defaultdict(lambda: {m: 0 for m in self.measures})

    def collect_data(self):
        status_to_measure_map = {
            VACANCY_STATUSES.on_approval: 'on_approval_count',
            VACANCY_STATUSES.in_progress: 'in_progress_count',
            VACANCY_STATUSES.suspended: 'suspended_count',
            VACANCY_STATUSES.offer_processing: 'offer_processing_count',
            VACANCY_STATUSES.offer_accepted: 'offer_accepted_count',
        }
        vacancies = (
            Vacancy.unsafe
            .filter(status__in=status_to_measure_map.keys())
            .values(
                'department_id',
                'status',
                vacancy_type=VacancyTypeExpression('type'),
            )
        )
        for vacancy in vacancies:
            keys = self.get_related_keys((
                StaffUnit(vacancy['department_id'], enums.StaffUnitTypes.department),
                vacancy['vacancy_type'],
            ))
            measures = [status_to_measure_map[vacancy['status']]]
            for key in keys:
                for measure in measures:
                    self.result[key][measure] += 1


class VacancyProcessingSpeedDataFetcher(HierarchicReportDataFetcher):

    department_ids = [
        settings.YANDEX_DEPARTMENT_ID,
        settings.OUTSTAFF_DEPARTMENT_ID,
    ]

    def initialize_result(self):
        # сумма периодов / количество закрытий
        default_measures = {
            m: [timedelta(), 0] for m in self.measures if m != 'close_count'
        }
        default_measures['close_count'] = 0
        self.result = defaultdict(lambda: deepcopy(default_measures))

    def collect_data(self):
        qs = (
            VacancyHistory.objects
            .filter(
                vacancy__status=VACANCY_STATUSES.closed,
                vacancy_id__in=(
                    VacancyHistory.objects
                    .filter(
                        changed_at__gte=self.from_dt,
                        changed_at__lt=self.to_dt,
                        status=VACANCY_STATUSES.closed,
                    )
                    .values('vacancy')
                )
            )
            .order_by('vacancy_id', 'changed_at')
            .values(
                'changed_at',
                'vacancy_id',
                'status',
                department_id=F('vacancy__department_id'),
                vacancy_type=VacancyTypeExpression('vacancy__type'),
                vacancy_resolution=F('vacancy__resolution'),
                prof_sphere_id=F('vacancy__professional_sphere_id'),
            )
        )
        valid_statuses = [
            VACANCY_STATUSES.on_approval,
            VACANCY_STATUSES.suspended,
            VACANCY_STATUSES.in_progress,
            VACANCY_STATUSES.offer_processing,
            VACANCY_STATUSES.offer_accepted,
        ]
        for vacancy_id, group in groupby(qs, lambda x: x['vacancy_id']):
            group_list = list(group)
            item = group_list[0]
            keys = list(
                self.get_related_keys((
                    StaffUnit(item['department_id'], enums.StaffUnitTypes.department),
                    item['vacancy_type'] or enums.StatKeys.unknown,
                    item['vacancy_resolution'] or enums.StatKeys.unknown,
                    item['prof_sphere_id'] or enums.StatKeys.unknown,
                ))
            )

            for key in keys:
                self.result[key]['close_count'] += 1

            for i in range(len(group_list) - 1):
                curr, next = group_list[i], group_list[i + 1]

                if curr['status'] not in valid_statuses:
                    continue

                for key in keys:
                    measure = curr['status'] + '_avg'
                    self.result[key][measure][0] += next['changed_at'] - curr['changed_at']
                    self.result[key][measure][1] += 1

    def get_transformed_measures(self, key):
        return self.get_transformed_measures_with_periods(
            key=key,
            non_period_measures=('close_count',),
        )


class VacanciesDurationDataFetcher(ReportDataFetcher):
    """
    Это странный отчет, потому что он пока единственный, где в измерениях не считается значение ALL.
    То есть значения измерений здесь - это часть целого, а непересекающиеся по логике варианты.
    """
    DURATION_INTERVALS = (
        (timedelta(60), '0-60'),
        (timedelta(90), '61-90'),
        (timedelta(120), '91-120'),
        (timedelta(180), '121-180'),
        (timedelta.max, '181+'),
    )

    def _get_duration_interval(self, duration):
        for _max, _str in self.DURATION_INTERVALS:
            if duration <= _max:
                return _str

    def _get_qs(self):
        next_changed_at_qs = (
            VacancyHistory.objects
            .filter(
                vacancy=OuterRef('vacancy_id'),
                changed_at__gt=OuterRef('changed_at'),
            )
            .values('vacancy_id')
            .annotate(_aggr=Min('changed_at'))
            .values('_aggr')
        )

        vacancy_history_qs = (
            VacancyHistory.objects
            .filter(vacancy=OuterRef('id'))
            .annotate(
                next_changed_at=Coalesce(
                    Subquery(next_changed_at_qs, DateTimeField()),
                    self.to_dt,
                ),
            )
            .order_by('changed_at')
            .values(
                'status',
                'changed_at',
                'next_changed_at',
            )
        )

        approved_at_qs = (
            VacancyHistory.objects
            .filter(
                vacancy=OuterRef('id'),
                status=VACANCY_STATUSES.in_progress,
            )
            .values('vacancy')
            .annotate(_aggr=Min('changed_at'))
            .values('_aggr')
        )

        return (
            Vacancy.unsafe
            .filter(
                status__in=(
                    VACANCY_STATUSES.on_approval,
                    VACANCY_STATUSES.in_progress,
                    VACANCY_STATUSES.suspended,
                    VACANCY_STATUSES.offer_processing,
                ),
            )
            .values(
                'id',
                'created',
                approved_at=Subquery(approved_at_qs, DateTimeField()),
                history=RowsToList(vacancy_history_qs),
                vacancy_type=VacancyTypeExpression('type'),
            )
        )

    def _process_vacancy(self, vacancy):
        suspense_time = timedelta()
        history = vacancy['history'] or []
        vacancy_type = vacancy['vacancy_type']

        for item in history:
            if item['status'] != VACANCY_STATUSES.suspended:
                continue
            changed_at = parse_datetime(item['changed_at'])
            next_changed_at = parse_datetime(item['next_changed_at'])
            suspense_time += next_changed_at - changed_at

        starting_points = {
            choices.VACANCY_STARTING_POINTS.created: vacancy['created'],
        }
        if vacancy['approved_at']:
            starting_points[choices.VACANCY_STARTING_POINTS.approved] = vacancy['approved_at']

        keys = []
        for starting_point, dt in starting_points.items():
            delta = self.to_dt - dt
            for vacancy_type in [vacancy_type, enums.StatKeys.all]:
                keys.extend([
                    (
                        starting_point,
                        False,  # Не исключаем приостановки
                        self._get_duration_interval(delta),
                        vacancy_type,
                    ),
                    (
                        starting_point,
                        True,  # Исключаем приостановки
                        self._get_duration_interval(delta - suspense_time),
                        vacancy_type,
                    ),
                ])

        for key in keys:
            self.result[key] += 1

    def collect_data(self):
        self.result = defaultdict(int)

        qs = self._get_qs()
        for vacancy in qs:
            self._process_vacancy(vacancy)

    def get_data(self):
        self.collect_data()
        data = []
        for key in self.result:
            starting_point, exclude_suspense_time, duration_interval, vacancy_type = key
            data.append({
                'fielddate': self.fielddate,
                'starting_point': starting_point,
                'exclude_suspense_time': exclude_suspense_time,
                'duration_interval': duration_interval,
                'vacancy_type': vacancy_type,
                'vacancies_count': self.result[key],
            })
        return data


class OfferVacancyProcessingSpeedDataFetcher(HierarchicReportDataFetcher):

    department_ids = [
        settings.YANDEX_DEPARTMENT_ID,
        settings.OUTSTAFF_DEPARTMENT_ID,
    ]

    def initialize_result(self):
        # сумма периодов пребывания в статусе / количество пребываний в статусе
        default_measures = {
            m: [timedelta(), 0] for m in self.measures if m != 'close_count'
        }
        default_measures['close_count'] = 0
        self.result = defaultdict(lambda: deepcopy(default_measures))

    def _get_qs(self):
        closed_recently_vacancy_ids_qs = (
            VacancyHistory.objects
            .filter(
                changed_at__gte=self.from_dt,
                changed_at__lt=self.to_dt,
                status=VACANCY_STATUSES.closed,
            )
            .values('vacancy')
        )

        last_closed_offer_qs = (
            Offer.unsafe
            .filter(
                status=OFFER_STATUSES.closed,
                vacancy=OuterRef('id'),
                # Игнорим мигрированные офферы
                grade__isnull=False,
            )
            .order_by('-modified')
            .values(
                'department_id',
                'profession_id',
                'programming_language',
                vacancy_type=VacancyTypeExpression('vacancy__type'),
                grade_range=_get_grade_range('grade'),
                city_id=F('office__city_id'),
            )[:1]
        )

        vacancy_history_qs = (
            VacancyHistory.objects
            .filter(vacancy=OuterRef('id'))
            .order_by('changed_at')
            .values(
                'changed_at',
                'status',
            )
        )

        return (
            Vacancy.unsafe
            .annotate(
                offer=RowToDict(last_closed_offer_qs),
                history=RowsToList(vacancy_history_qs),
            )
            .filter(
                status=VACANCY_STATUSES.closed,
                offer__isnull=False,
                id__in=closed_recently_vacancy_ids_qs,
            )
            .values(
                'id',
                'offer',
                'history',
            )
        )

    def _process_vacancy(self, vacancy, valid_statuses):
        offer = vacancy['offer']
        history = vacancy['history']
        keys = list(
            self.get_related_keys((
                StaffUnit(offer['department_id'], enums.StaffUnitTypes.department),
                ProfessionUnit(offer['profession_id'], enums.ProfessionUnitTypes.profession),
                offer['vacancy_type'] or enums.StatKeys.unknown,
                offer['programming_language'] or enums.StatKeys.unknown,
                offer['city_id'] or enums.StatKeys.unknown,
                offer['grade_range'],
            ))
        )

        for key in keys:
            self.result[key]['close_count'] += 1

        _first = history[0]
        _first['changed_at'] = parse_datetime(_first['changed_at'])
        for i in range(len(history) - 1):
            _curr, _next = history[i], history[i + 1]
            _next['changed_at'] = parse_datetime(_next['changed_at'])

            if _curr['status'] not in valid_statuses:
                continue

            for key in keys:
                measure = _curr['status'] + '_avg'
                self.result[key][measure][0] += _next['changed_at'] - _curr['changed_at']
                self.result[key][measure][1] += 1

    def collect_data(self):
        qs = self._get_qs()
        valid_statuses = [
            VACANCY_STATUSES.on_approval,
            VACANCY_STATUSES.suspended,
            VACANCY_STATUSES.in_progress,
            VACANCY_STATUSES.offer_processing,
            VACANCY_STATUSES.offer_accepted,
        ]
        for vacancy in qs:
            self._process_vacancy(vacancy, valid_statuses)

    def get_transformed_measures(self, key):
        return self.get_transformed_measures_with_periods(
            key=key,
            non_period_measures=('close_count',),
        )
