import datetime
import pytz

from collections import namedtuple

from django.db.models import CharField, Value
from django.utils import timezone

from intranet.femida.src.core.db.expressions import MapperExpression
from intranet.femida.src.vacancies.choices import VACANCY_TYPES
from intranet.femida.src.stats import enums


StaffUnit = namedtuple('StaffUnit', ['id', 'type'])
ProfessionUnitBase = namedtuple('ProfessionUnitBase', ['id', 'type'])


class ProfessionUnit(ProfessionUnitBase):

    @property
    def key(self):
        return '{type}{id}'.format(**self._asdict())


def get_beginning_of_moscow_day(dt):
    """
    :param dt: aware datetime
    :return: Start of moscow datetime
    """
    assert dt.tzinfo is not None
    tz = pytz.timezone('Europe/Moscow')
    current_dt = dt.astimezone(tz)
    return current_dt.replace(hour=0, minute=0, second=0, microsecond=0)


def str_to_moscow_date(date_str):
    dt = timezone.datetime.strptime(date_str, '%Y-%m-%d')
    tz = pytz.timezone('Europe/Moscow')
    return tz.localize(dt)


def _strptime_or_date(dt, date_str=None):
    """
    Возвращает datetime в MSK из строки date_str
    или начало дня для dt, если date_str is None
    :param date_str: строка формата %Y-%m-%d
    """
    if date_str is None:
        dt = get_beginning_of_moscow_day(dt)
    else:
        dt = str_to_moscow_date(date_str)
    return dt


def strptime_or_yesterday(date_str=None):
    yesterday = timezone.now() - datetime.timedelta(days=1)
    return _strptime_or_date(yesterday, date_str)


def strptime_or_today(date_str=None):
    today = timezone.now()
    return _strptime_or_date(today, date_str)


VACANCY_TYPES_MAP = {
    VACANCY_TYPES.new: 'regular',
    VACANCY_TYPES.replacement: 'regular',
    VACANCY_TYPES.pool: 'regular',
    VACANCY_TYPES.internship: 'internship',
}

_hiring_types = (
    'regular',
    'rotation',
)
HIRING_TYPES = namedtuple('HIRING_TYPES', _hiring_types)(*_hiring_types)


ROTATION_HIRING_TYPE_MAP = {
    True: HIRING_TYPES.rotation,
    False: HIRING_TYPES.regular,
    None: HIRING_TYPES.regular,
}


class VacancyTypeExpression(MapperExpression):

    output_field = CharField()

    def __init__(self, field_name, **kwargs):
        super().__init__(
            field_name=field_name,
            mapping=VACANCY_TYPES_MAP,
            default=Value(enums.StatKeys.unknown),
            **kwargs
        )
