import logging
import requests
import smtplib
import waffle
import xml.etree.ElementTree as ET
import yenv

from typing import Optional
from urllib.parse import urlencode

from constance import config
from django.conf import settings
from django.core.mail import EmailMessage, get_connection
from django.core.mail.message import make_msgid

from intranet.femida.src.celery_app import app, get_retry_countdown


logger = logging.getLogger(__name__)


def get_emails_whitelist():
    emails = config.EMAILS_WHITELIST.strip()
    if not emails:
        return []
    return [e.strip() for e in emails.split(',')]


def _get_attachments(attachment_ids):
    from intranet.femida.src.attachments.models import Attachment

    if not attachment_ids:
        return []
    attachments = (
        Attachment.objects
        .filter(id__in=attachment_ids)
        .only('name', 'attached_file', 'content_type')
    )
    return [(
        a.name,
        a.attached_file.read(),
        a.content_type if a.content_type else 'plain/text',
    ) for a in attachments]


@app.task(max_retries=5)
def send_email(subject, body, to, **kwargs) -> Optional[str]:
    """
    Отправляет email.
    В случае некоторых ошибок возвращает строку с деталями ошибки
    """
    is_sms = kwargs.setdefault('is_sms', False)
    headers = kwargs.setdefault('headers', {})
    from_email = kwargs.setdefault('from_email', settings.DEFAULT_FROM_EMAIL)
    attachment_ids = kwargs.setdefault('attachment_ids', [])
    attachments = _get_attachments(attachment_ids)
    cc = kwargs.setdefault('cc', [])
    bcc = kwargs.setdefault('bcc', [])
    original_to = list(to)
    message_id = kwargs.get('message_id')
    in_reply_to = kwargs.get('in_reply_to')
    reply_to = kwargs.get('reply_to')

    if is_sms:
        body += '<br/>\nOriginal SMS To: %s' % to
        to = [settings.DEBUG_EMAIL]
    elif yenv.type != 'production':
        body += '<br/>\nOriginal To: %s' % to
        emails_whitelist = set(get_emails_whitelist())
        to = list(set(to) & emails_whitelist)
        to.append(settings.DEBUG_EMAIL)

        if cc:
            body += '<br/>\nOriginal Cc: %s' % cc
            cc = list(set(cc) & emails_whitelist)
        if bcc:
            body += '<br/>\nOriginal Bcc: %s' % bcc
            bcc = list(set(bcc) & emails_whitelist)

    # Сами генерируем Message-ID, чтобы в случае ретрая отправлялось то же самое письмо
    if 'Message-ID' not in headers:
        headers['Message-ID'] = message_id or make_msgid()

    if in_reply_to and 'In-Reply-To' not in headers:
        headers['In-Reply-To'] = in_reply_to

    if reply_to and 'Reply-To' not in headers:
        headers['Reply-To'] = reply_to

    connection = get_connection()
    try:
        connection.open()
    except Exception:
        logger.exception('Cannot reach EMAIL_BACKEND %s', settings.EMAIL_BACKEND)

    msg = EmailMessage(
        subject=subject,
        body=body,
        from_email=from_email,
        to=to,
        headers=headers,
        attachments=attachments,
        cc=cc,
        bcc=bcc,
        connection=connection,
    )
    msg.content_subtype = 'html'

    try:
        msg.send()
    except smtplib.SMTPRecipientsRefused as exc:
        logger.warning('Invalid email recipients: %s', to, exc_info=exc)
        if not waffle.switch_is_active('enable_message_status_failed'):
            raise
        return f'Recipients refused: {exc.recipients}'
    except Exception as exc:
        if isinstance(exc, smtplib.SMTPServerDisconnected):
            logger.error('SMTP server disconnected, but email may be sent.')
            # Если словили эту ошибку, то ретраить будем максимум последний раз, чтобы не заспамить
            # https://st.yandex-team.ru/MAILADM-5207
            send_email.max_retries = 1

        logger.exception('Error during send email with Message-ID %s', headers['Message-ID'])
        # Если письмо внешнее и ретраи не помогли, напишем в лог об этом
        if kwargs.get('is_external') and send_email.request.retries >= send_email.max_retries:
            logger.info(
                'May be candidate will not receive the email! '
                'From: %s, To: %s, Subject: %s, Message-ID: %s',
                from_email, to, subject, headers['Message-ID']
            )
        send_email.retry(
            countdown=get_retry_countdown(send_email.request.retries),
            exc=exc,
            args=[subject, body, original_to],
            kwargs=kwargs,
        )
    else:
        attach_str = ', '.join([a[0] for a in attachments]) or '-'
        logger.info(
            'Sent email %s to %s from %s, Message-ID: %s, attachments: %s',
            subject, to, from_email, headers['Message-ID'], attach_str,
        )
    finally:
        connection.close()


@app.task(max_retries=5)
def send_sms(phone, text, sender='femida'):
    if yenv.type != 'production':
        send_email(subject='', body=text, to=[phone], is_sms=True)
        return

    params = {
        'sender': sender,
        'utf8': 1,
        'phone': phone,
        'text': text.encode('utf-8'),
    }

    url = '{host}/sendsms?{params}'.format(
        host=settings.SMS_PASSPORT_HOST,
        params=urlencode(params),
    )

    try:
        response = requests.get(url)
    except requests.RequestException as exc:
        send_sms.retry(
            countdown=get_retry_countdown(send_sms.request.retries),
            exc=exc,
        )
        return

    tree = ET.fromstring(response.content)

    message_sent = tree.find('message-sent')
    if message_sent is not None:
        sms_id = message_sent.attrib['id']
        logger.info('SMS was sent with id %s', sms_id)
    else:
        error = tree.find('error')
        if error is not None:
            error_msg = error.text
            logger.info('SMS was not sent with error `%s`', error_msg)
        else:
            logger.error(
                'Unknown response format from %s: %s',
                settings.SMS_PASSPORT_HOST,
                response.content,
            )
