from django.conf import settings
from django.contrib.auth.models import AbstractUser
from django.db import models
from django.utils.functional import cached_property

from intranet.femida.src.core.choices import GENDER_CHOICES, LANGUAGES
from intranet.femida.src.interviews.choices import AA_TYPES


class User(AbstractUser):
    staff_id = models.IntegerField(blank=True, null=True)
    uid = models.CharField(max_length=30, blank=True, db_index=True)
    first_name_en = models.CharField(max_length=30, null=True, blank=True)
    last_name_en = models.CharField(max_length=150, null=True, blank=True)
    is_dismissed = models.BooleanField(default=False)
    gender = models.CharField(max_length=6, choices=GENDER_CHOICES)
    phone = models.CharField(max_length=30, blank=True)
    lang = models.CharField(max_length=2, null=True, blank=True, choices=LANGUAGES, default='ru')
    timezone = models.CharField(max_length=32, default='Europe/Moscow')
    work_phone = models.PositiveIntegerField(null=True, blank=True)
    department = models.ForeignKey(
        to='staff.Department',
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name='users',
    )
    is_intern = models.NullBooleanField()
    chief = models.ForeignKey(
        to='self',
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        to_field='username',
        related_name='subordinates',
    )
    position_ru = models.CharField(max_length=150, null=True, blank=True)

    @cached_property
    def subordinates_rec(self):
        subordinates_ids = set(
            subordinate.pk
            for subordinate in self.subordinates.all()
            if not subordinate.is_dismissed
        )
        layer = subordinates_ids
        while layer:
            layer = set(
                User.objects
                    .filter(chief__pk__in=layer, is_dismissed=False)
                    .exclude(pk__in=subordinates_ids)
                    .values_list('pk', flat=True)
            )
            subordinates_ids |= layer
        return User.objects.filter(pk__in=subordinates_ids)

    @cached_property
    def recruitment_partner(self):
        if self.position_ru == 'Рекрутмент партнер':
            return self
        return getattr(self.chief, 'recruitment_partner', None)

    @property
    def is_hrbp(self):
        """
        FEMIDA-7431
        """
        return self.has_perm('permissions.hrbp_perm')

    @property
    def is_auditor(self):
        """
        FEMIDA-7440
        """
        return self.has_perm('permissions.auditor_perm')

    @property
    def is_recruiting_manager(self):
        return self.has_perm('permissions.recruiting_manager_perm')

    @property
    def is_recruiter(self):
        return self.has_perm('permissions.recruiter_perm')

    @property
    def is_recruiter_assessor(self):
        """
        FEMIDA-5284 Рекрутер-асессор
        """
        return self.has_perm('permissions.recruiter_assessor_perm')

    @property
    def is_publication_manager(self):
        """
        FEMIDA-6092
        Это право ограничивает видимость полей вакансии в админке
        """
        if self.is_superuser:
            return False
        return self.has_perm('permissions.publication_manager_perm')

    @property
    def is_reference_coordinator(self):
        """
        Координатор рекомендаций - Инна :)
        """
        return self.has_perm('permissions.reference_coordinator_perm')

    @property
    def is_screening_reviewer(self):
        return self.has_perm('permissions.screening_reviewer_perm')

    @property
    def is_aa(self):
        return self.aa_type is not None

    @property
    def is_aa_canonical(self):
        return self.has_perm('permissions.aa_perm')

    @property
    def is_robot_femida(self):
        return self.uid == settings.FEMIDA_ROBOT_UID

    @property
    def aa_type(self):
        # Note: плохое место, потому что в теории
        # человек может быть AA-собесудеющим нескольких типов
        for aa_type, _ in AA_TYPES:
            if getattr(self, f'is_aa_{aa_type}', False):
                return aa_type

    @property
    def is_analyst(self):
        return self.has_perm('permissions.analyst_perm')

    @property
    def is_moderator(self):
        return self.has_perm('problems.change_complaint')

    def get_full_name(self, lang=LANGUAGES.ru):
        if lang == LANGUAGES.ru:
            return super().get_full_name()
        return self.get_full_name_en()

    def get_full_name_en(self):
        return f'{self.first_name_en} {self.last_name_en}'.strip()

    def __getattr__(self, item):
        if item.startswith('is_aa_'):
            aa_type = item[6:]
            if aa_type in AA_TYPES:
                return self.has_perm(f'permissions.aa_{aa_type}_perm')
        raise AttributeError(f"'User' object has no attribute '{item}'")

    class Meta:
        db_table = 'auth_user'


class UserAbsence(models.Model):

    user = models.ForeignKey(User, on_delete=models.CASCADE)
    date = models.DateField()

    def __str__(self):
        return 'UserAbsence: {username}, {date}'.format(
            username=self.user.username,
            date=self.date.strftime('%Y-%m-%d'),
        )

    class Meta:
        unique_together = ('user', 'date')


def get_robot_femida():
    return User.objects.get(uid=settings.FEMIDA_ROBOT_UID)
