import logging
import requests

from collections import defaultdict

from django.conf import settings
from json.decoder import JSONDecodeError
from requests.exceptions import RequestException
from time import time, sleep

from intranet.femida.src.core.exceptions import FemidaError
from intranet.femida.src.utils.itertools import get_chunks


logger = logging.getLogger(__name__)


class BegemotAPIException(FemidaError):
    pass


class BegemotAPI:

    _begemot_last_request = time()
    request_url = settings.BEGEMOT_URL
    request_params = {}
    request_timeout = 0.1

    required_fields = set()

    @classmethod
    def _request(cls, user_params, timeout=None):
        timeout = timeout or cls.request_timeout
        cls.wait_rps_cooldown()
        try:
            response = requests.get(
                url=cls.request_url,
                params=cls.request_params | user_params,
                timeout=timeout,
            )
        except RequestException:
            raise BegemotAPIException('Begemot request error')

        try:
            data = response.json()
        except JSONDecodeError:
            response_content = response.content.decode('utf-8')
            raise BegemotAPIException(f'Begemot response parsing error: {response_content}')

        missing_fields = cls.required_fields - set(data)
        if missing_fields:
            raise BegemotAPIException(
                f'Wrong response format: missing fields {missing_fields} in response {data}',
            )
        return data

    @classmethod
    def wait_rps_cooldown(cls):
        """
        Временное решение чтобы не завалить Бегемот высоким rps.
        Если будет нужен высокий rps, то его надо будет согласовывать с Бегемотом.
        """
        now = time()
        delta_t = now - cls._begemot_last_request
        if delta_t < 1 / settings.BEGEMOT_MAX_RPS:
            sleep_time = 1 / settings.BEGEMOT_MAX_RPS - delta_t
            logger.info('Begemot rps exceeded. Sleep: %s', sleep_time)
            sleep(sleep_time)
        cls._begemot_last_request = now


class MisspellAPI(BegemotAPI):

    MSOPT_IGNORE_NICKNAMES = 0x0100

    status_code_corrected = 201
    status_codes_no_correction = (200, 202)

    good_hints = (10000, 8000)

    request_url = settings.MISSPELL_URL
    request_params = {
        'wizclient': 'jobs',
        'options': MSOPT_IGNORE_NICKNAMES,
        'ie': 'utf-8',
        'format': 'json',
    }

    required_fields = {'code', 'r', 'text'}

    local_rules = {
        "с++": "c++",
        "С++": "C++",
    }

    @classmethod
    def get_misspell_spellcheck(cls, text, timeout=None) -> (str, bool):
        user_params = {'text': text}
        try:
            misspell_json = cls._request(user_params, timeout)
        except BegemotAPIException as e:
            logger.exception(e.message)
            return text, False

        if misspell_json.get('code') in cls.status_codes_no_correction:
            return text, False

        misspell_success = (
            int(misspell_json['code']) == cls.status_code_corrected
            and int(misspell_json['r']) in cls.good_hints
        )
        if misspell_success:
            return misspell_json['text'], True
        return text, False

    @classmethod
    def get_local_rules_spellcheck(cls, text):
        tokens, is_corrected = text.split(), False
        for i, elem in enumerate(tokens):
            if elem in cls.local_rules:
                tokens[i] = cls.local_rules[elem]
                is_corrected = True
        return ' '.join(tokens), is_corrected

    @classmethod
    def get_spellcheck(cls, text, local_rules=False):
        corrected_text, is_corrected = cls.get_misspell_spellcheck(text)
        if local_rules:
            corrected_text, local_is_corrected = cls.get_local_rules_spellcheck(corrected_text)
            is_corrected |= local_is_corrected
        return corrected_text, is_corrected


class SynonymAPI(BegemotAPI):

    request_params = {
        'wizclient': 'jobs',
        'action': 'markup',
        'markup': 'layers=Extensions,Tokens',
        'format': 'json',
        'rwr': 'on:CustomThesaurus/Jobs',
    }

    required_fields = {'Tokens'}
    batch_size = 15  # лимит 20, но слова с дефисом разбиваются на два слова. 15 должно хватить

    @classmethod
    def _parse_response(cls, json_response, min_weight) -> dict[str, list]:
        tokens = []
        for data in json_response['Tokens']:
            tokens.append(data['Text'])

        synonyms = defaultdict(list)
        for data in json_response.get('Extensions', {}):
            if data.get('Weight', 0.0) < min_weight:
                continue
            begin = data.get('Tokens', {}).get('Begin', 0)
            end = data.get('Tokens', {}).get('End', 0)
            token = ' '.join(tokens[begin:end])
            synonyms[token].append(data.get('ExtendTo', ''))

        return synonyms

    @classmethod
    def _get_synonyms_batch(cls, text, min_weight=0.0, timeout=None):
        user_params = {'text': text}
        try:
            begemot_json = cls._request(user_params, timeout)
        except BegemotAPIException as e:
            logger.exception(e.message)
            return {}

        return cls._parse_response(begemot_json, min_weight)

    @classmethod
    def get_synonyms(cls, words, min_weight=0.0, timeout=None):
        synonyms = {}
        for words_chunk in get_chunks(list(words), n=cls.batch_size):
            batch = cls._get_synonyms_batch(
                text=' '.join(words_chunk),
                min_weight=min_weight,
                timeout=timeout,
            )
            synonyms.update(batch)

        return synonyms
