import waffle

from uuid import uuid4

from django.utils.functional import cached_property


class CSPMixin:
    """
    Миксин, который добавляет возможность проставить
    CSP заголовок к ответу вьюхи
    """
    csp_report_service = 'femida'
    csp_default_sources = ["'none'"]
    csp_connect_sources = ["'self'"]

    csp_child_sources = []
    csp_font_sources = []
    csp_frame_sources = []
    csp_img_sources = []
    csp_manifest_sources = []
    csp_media_sources = []
    csp_object_sources = []
    csp_script_sources = []
    csp_style_sources = []
    csp_worker_sources = []

    csp_include_nonce = True

    @cached_property
    def csp_nonce(self):
        return uuid4().hex

    @cached_property
    def reqid(self):
        return self.request.META.get('HTTP_X_REQUEST_ID') or uuid4().hex

    def _get_csp_header_name(self):
        if waffle.switch_is_active('enable_csp_report_only'):
            return 'Content-Security-Policy-Report-Only'
        return 'Content-Security-Policy'

    def _get_directive(self, dir_name, dir_values):
        if isinstance(dir_values, (list, tuple)):
            dir_values = ' '.join(dir_values)
        return '{} {};'.format(dir_name, dir_values)

    def _get_report_uri(self):
        return 'https://csp.yandex.net/csp?from={}'.format(self.csp_report_service)

    def _get_csp_header_value(self):
        csp_script_sources = self.csp_script_sources[:]
        if self.csp_include_nonce:
            csp_script_sources.append("'nonce-{}'".format(self.csp_nonce))
        directives = (
            ('default-src', self.csp_default_sources),
            ('child-src', self.csp_child_sources),
            ('connect-src', self.csp_connect_sources),
            ('font-src', self.csp_font_sources),
            ('frame-src', self.csp_frame_sources),
            ('img-src', self.csp_img_sources),
            ('manifest-src', self.csp_manifest_sources),
            ('media-src', self.csp_media_sources),
            ('object-src', self.csp_object_sources),
            ('script-src', csp_script_sources),
            ('style-src', self.csp_style_sources),
            ('worker-src', self.csp_worker_sources),
        )
        directive_list = [self._get_directive(n, v) for n, v in directives if v]
        directive_list.append('report-to {};'.format(self._get_report_uri()))
        # deprecated, нужна для обратной совместимости
        # https://www.w3.org/TR/CSP3/#directive-report-uri
        directive_list.append('report-uri {};'.format(self._get_report_uri()))
        return ' '.join(directive_list)

    def set_csp_header(self, response):
        response[self._get_csp_header_name()] = self._get_csp_header_value()
        return response
