import logging
import requests

from requests.exceptions import ConnectionError, Timeout
from urllib.parse import unquote

from django.conf import settings
from yandex.maps.proto.common2.response_pb2 import Response
from yandex.maps.proto.search.geocoder_internal_pb2 import TOPONYM_INFO
from yandex.maps.proto.search.geocoder_pb2 import GEO_OBJECT_METADATA as GEOCODER_METADATA
from yandex.maps.proto.uri.uri_pb2 import GEO_OBJECT_METADATA as URI_METADATA

from intranet.femida.src.core.exceptions import FemidaError
from intranet.femida.src.utils.itertools import get_chunks
from intranet.femida.src.utils.tvm2_client import get_service_ticket


logger = logging.getLogger(__name__)


class GeosearchError(FemidaError):
    pass


class GeosearchAPI:
    """
    Класс для хождения в Геокодер
    https://a.yandex-team.ru/arc/trunk/arcadia/extsearch/geo
    """
    url = settings.GEOSEARCH_API_URL
    tvm_client_id = settings.TVM_GEOSEARCH_CLIENT_ID
    chunk_size = 10  # максимальное количество объектов в ответе от Геокодера

    @classmethod
    def _request(cls, method, params=None):
        headers = {'X-Ya-Service-Ticket': get_service_ticket(cls.tvm_client_id)}

        try:
            response = requests.request(
                method=method,
                url=cls.url,
                params=params,
                headers=headers,
                verify=settings.YANDEX_INTERNAL_CERT_PATH,
            )
        except (ConnectionError, Timeout):
            raise GeosearchError('Geosearch is not responding')

        if not response.ok:
            logger.error(
                'Geosearch error %s, response: %s',
                response.status_code,
                response.content.decode('utf-8'),
            )
            raise GeosearchError(f'Wrong status code from Geosearch: {response.status_code}')

        return response.content

    @classmethod
    def _parse_geo_object_metadata(cls, geocoder_metadata, uri_metadata):
        if not geocoder_metadata.HasExtension(TOPONYM_INFO):
            raise GeosearchError(f'No TOPONYM_INFO in geocoder_metadata: {geocoder_metadata}')
        if not uri_metadata.uri:
            raise GeosearchError(f'Empty uri_metadata: {uri_metadata}')
        return {
            'geo_id': geocoder_metadata.Extensions[TOPONYM_INFO].geoid,
            'name': geocoder_metadata.address.formatted_address,
            'country_code': geocoder_metadata.address.country_code,
            'geocoder_uri': unquote(uri_metadata.uri[0].uri),
        }

    @classmethod
    def _parse_response(cls, raw_response):
        """
        Парсинг protobuf ответа
        :return: [
            {
                'geo_id': <geo_id>,
                'name': <name>,
                'country_code': <country_code>,
                'geocoder_uri': <uri>,
            },
            ...
        ]
        """
        response = Response()
        response.ParseFromString(raw_response)

        results = []
        for geo_object in response.reply.geo_object:
            geocoder_metadata = None
            uri_metadata = None

            for metadata in geo_object.metadata:
                if metadata.HasExtension(GEOCODER_METADATA):
                    geocoder_metadata = metadata.Extensions[GEOCODER_METADATA]
                if metadata.HasExtension(URI_METADATA):
                    uri_metadata = metadata.Extensions[URI_METADATA]

            if geocoder_metadata and uri_metadata:
                geo_object_data = cls._parse_geo_object_metadata(geocoder_metadata, uri_metadata)
            else:
                raise GeosearchError(f'No GEO_OBJECT_METADATA for geo_object: {geo_object}')
            results.append(geo_object_data)

        return results

    @classmethod
    def get_geo_objects(cls, params=None):
        base_params = {
            'origin': 'femida',  # human readeable client and process description
            'type': 'geo',  # search results type
            'lang': 'ru',  # response language
            'ms': 'pb',  # turn protobuf on
        }
        params = base_params | (params or {})
        raw_response = cls._request(method='GET', params=params)
        return cls._parse_response(raw_response)

    @classmethod
    def get_geo_objects_by_uris(cls, uris, params=None):
        """
        Получаем от Геокодера список локаций по списку их uri.
        Список uri разбиваем на чанки, чтобы Геокодер не обрезал ответ.
        """
        geo_objects = []
        for uris_chunk in get_chunks(uris, cls.chunk_size):
            params = {'uri': uris_chunk} | (params or {})
            geo_objects_batch = cls.get_geo_objects(params=params)
            if len(geo_objects_batch) == len(uris_chunk):
                geo_objects.extend(geo_objects_batch)
            else:
                raise GeosearchError(
                    f'Could not get data for all uris: {uris_chunk}. Data: {geo_objects_batch}'
                )
        return geo_objects

    @classmethod
    def get_multilang_geo_objects(cls, uris):
        """
        Получаем от Геокодера список локаций с адресом на двух языках
        """
        ru_geo_objects = cls.get_geo_objects_by_uris(uris=uris, params={'lang': 'ru'})
        en_geo_objects = cls.get_geo_objects_by_uris(uris=uris, params={'lang': 'en'})
        geo_objects = []

        en_geo_objects_by_id = {obj['geo_id']: obj for obj in en_geo_objects}
        for ru_geo_object in ru_geo_objects:
            geo_id = ru_geo_object['geo_id']
            en_geo_object = en_geo_objects_by_id[geo_id]
            ru_geo_object['name_ru'] = ru_geo_object.pop('name')
            ru_geo_object['name_en'] = en_geo_object['name']
            geo_objects.append(ru_geo_object)
        return geo_objects
