import functools
import hashlib
import logging
import os
import time

import ylock

from contextlib import contextmanager

from celery.exceptions import Ignore
from django.core.cache import cache
from django.conf import settings


DEFAULT_CACHE_TIMEOUT = 60

logger = logging.getLogger(__name__)
manager = ylock.backends.create_manager(**settings.YLOCK)


def _get_lock_name(function, args, kwargs):
    lock_name = f'{function.__module__}.{function.__name__}'
    if args or kwargs:
        hashed_args = hashlib.sha1(f'{args}-{kwargs}'.encode('utf-8')).hexdigest()
        lock_name += f':{hashed_args}'
    return lock_name


@contextmanager
def cache_lock(name, timeout=None, *args, **kwargs):
    name = f'{settings.DEPLOY_STAGE_ID}.{name}'
    timeout = timeout or DEFAULT_CACHE_TIMEOUT
    acquired = cache.add(name, True, timeout)
    delay = kwargs.get('delay', 0)
    try:
        yield acquired
    finally:
        if acquired and cache.get(name):
            cache.set(name, True, delay)


def locked_task(task=None, *, timeout=10, block=False, delay=10):
    assert task is None or callable(task)
    if callable(task):
        return locked_task()(task)

    def decorator(function):
        @functools.wraps(function)
        def wrapper(*args, **kwargs):
            # Если выполняется не в celery-воркере, то не нужно брать лок
            if not int(os.getenv('CELERY_INSTANCE', 0)):
                return function(*args, **kwargs)

            is_cache_lock = int(os.getenv('ENABLE_CELERY_CACHE_LOCK', 0))
            lock_function = cache_lock if is_cache_lock else manager.lock

            lock_name = _get_lock_name(function, args, kwargs)
            logger.info('Acquiring lock `%s`', lock_name)
            with lock_function(lock_name, timeout, block, delay=delay) as acquired:
                if acquired:
                    logger.info('Lock `%s` acquired', lock_name)
                    result = function(*args, **kwargs)
                    if not is_cache_lock:
                        # Чтобы lock не освобождался слишком быстро
                        time.sleep(delay)
                    return result
                else:
                    logger.info('Lock `%s` NOT acquired', lock_name)
                    raise Ignore(f'Lock `{lock_name}` NOT acquired')
        return wrapper
    return decorator
