import logging
import requests

from requests.exceptions import ConnectionError, Timeout

from django.conf import settings

from intranet.femida.src.core.exceptions import FemidaError
from intranet.femida.src.utils.tvm2_client import get_service_ticket, get_robot_user_ticket


logger = logging.getLogger(__name__)


class NewhireError(FemidaError):

    pass


class NewhireAPI:

    @classmethod
    def _request(cls, method, url, params=None, data=None, use_tvm=False):
        if use_tvm:
            headers = {
                'X-Ya-Service-Ticket': get_service_ticket(settings.TVM_STAFF_CLIENT_ID),
                'X-Ya-User-Ticket': get_robot_user_ticket(),
            }
        else:
            headers = {
                'Authorization': 'OAuth %s' % settings.FEMIDA_ROBOT_TOKEN,
            }

        try:
            response = requests.request(
                method=method,
                url=settings.STAFF_URL + url,
                params=params,
                json=data,
                timeout=30,
                headers=headers,
                verify=settings.YANDEX_INTERNAL_CERT_PATH,
            )
        except (ConnectionError, Timeout):
            logger.exception('Newhire is not responding')
            return None, 500

        if not response.ok:
            logger.info(
                'Newhire error %s, response: %s',
                response.status_code,
                response.content.decode('utf-8'),
            )

        try:
            data = response.json() if response.content else {}
        except ValueError:
            logger.exception(
                'Error during parse Newhire response: %s',
                response.content.decode('utf-8'),
            )
            return None, 500

        return data, response.status_code

    @classmethod
    def _post(cls, url, data=None, use_tvm=False):
        return cls._request('POST', url, data=data, use_tvm=use_tvm)

    @classmethod
    def _get(cls, url, params=None):
        return cls._request('GET', url, params=params)

    @classmethod
    def check_login(cls, login, preprofile_id=None):
        data = {'login': login}
        if preprofile_id:
            data['preprofile_id'] = preprofile_id
        return cls._post('preprofile-api/check_login_availability/', data=data)

    @classmethod
    def create_preprofile(cls, data):
        return cls._post('preprofile-api/form/create_from_femida/', data=data)

    @classmethod
    def get_preprofile(cls, newhire_offer_id):
        url = 'preprofile-api/form/%s/femida_update/' % newhire_offer_id
        return cls._get(url)

    @classmethod
    def update_preprofile(cls, newhire_offer_id, data):
        url = 'preprofile-api/form/%s/femida_update/' % newhire_offer_id
        return cls._post(url, data=data)

    @classmethod
    def get_preprofiles(cls, date_from):
        params = {'date_from': date_from.isoformat()}
        return cls._get('preprofile-api/export/femida/', params=params)

    @classmethod
    def cancel_preprofile(cls, newhire_offer_id):
        url = 'preprofile-api/form/%s/cancel/' % newhire_offer_id
        return cls._post(url)

    @classmethod
    def accept_preprofile(cls, preprofile_id, data):
        url = 'preprofile-api/form/%s/submit_ext_form/' % preprofile_id
        return cls._post(url, data=data)

    @classmethod
    def attach_phone(cls, data):
        url = 'preprofile-api/digital_sign/attach/'
        return cls._post(url, data=data, use_tvm=True)

    @classmethod
    def verify_code(cls, data):
        url = 'preprofile-api/digital_sign/verify/'
        return cls._post(url, data=data, use_tvm=True)
