import re
import string
import unicodedata

from typing import Set


cyrillic_latin_map = {
    'а': 'a',
    'б': 'b',
    'в': 'v',
    'г': 'g',
    'д': 'd',
    'е': 'e',
    'ё': 'yo',
    'ж': 'zh',
    'з': 'z',
    'и': 'i',
    'й': 'y',
    'к': 'k',
    'л': 'l',
    'м': 'm',
    'н': 'n',
    'о': 'o',
    'п': 'p',
    'р': 'r',
    'с': 's',
    'т': 't',
    'у': 'u',
    'ф': 'f',
    'х': 'h',
    'ц': 'ts',
    'ч': 'ch',
    'ш': 'sh',
    'щ': 'sch',
    'ъ': '',
    'ь': '',
    'э': 'e',
    'ю': 'yu',
    'я': 'ya',
}


def _slugify_char(c):
    if c in cyrillic_latin_map:
        return cyrillic_latin_map[c]
    if c.lower() in cyrillic_latin_map:
        return cyrillic_latin_map[c.lower()].capitalize()
    if c in string.printable:
        return c
    return '_'


def slugify(text):
    """
    Подменяет кириллические символы в строке на латинские,
    незнакомые на "_".
    """
    return ''.join(_slugify_char(c) for c in text)


def unaccent(text):
    """
    Удаляет диакритические знаки из слов
    >>> assert unaccent('Все́волод Вели́чко') == 'Всеволод Величко'
    >>> assert unaccent('Алёна') == 'Алена'
    """
    norm_txt = unicodedata.normalize('NFD', text)
    shaved = ''.join(c for c in norm_txt if not unicodedata.combining(c))
    return unicodedata.normalize('NFC', shaved)


def fetch_comma_separated_integers(s: str) -> Set:
    s = s.strip()
    return {int(i.strip()) for i in s.split(',')} if s else set()


def camel_to_kebab(text):
    rgx1 = re.compile(r'[^A-Za-z]')
    rgx2 = re.compile(r'(.)([A-Z][a-z]+)')
    rgx3 = re.compile(r'([a-z])([A-Z])')

    text = re.sub(rgx1, '', text)
    text = re.sub(rgx2, r'\1-\2', text)
    text = re.sub(rgx3, r'\1-\2', text)
    return text.lower()
