from collections import namedtuple

from constance import config
from django import forms
from django.conf.urls import url
from django.contrib import admin
from django.template.response import TemplateResponse

from intranet.femida.src.core.models import City
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.skills.models import Skill
from intranet.femida.src.staff.choices import DEPARTMENT_KINDS
from intranet.femida.src.staff.models import Department
from intranet.femida.src.startrek.admin import get_st_changelog_url
from intranet.femida.src.vacancies.choices import ACTIVE_VACANCY_STATUSES, VACANCY_PRO_LEVELS
from intranet.femida.src.vacancies.helpers import get_suitable_vacancies
from intranet.femida.src.vacancies.ranking import VACANCY_FACTORS, rank_proposals, get_factors
from . import models


class VacancySkillInlineAdmin(admin.TabularInline):

    model = models.VacancySkill


class VacancyMembershipInlineAdmin(admin.TabularInline):

    model = models.VacancyMembership

    raw_id_fields = (
        'member',
        'department_user',
    )


class VacancyCityInlineAdmin(admin.TabularInline):

    model = models.VacancyCity


class VacancyLocationInlineAdmin(admin.TabularInline):

    model = models.VacancyLocation


class VacancyOfficeInlineAdmin(admin.TabularInline):

    model = models.VacancyOffice


class VacancyRankingAdminForm(forms.Form):

    cities = forms.ModelMultipleChoiceField(City.objects.all(), required=False)
    professions = forms.ModelMultipleChoiceField(Profession.objects.all(), required=False)
    skills = forms.ModelMultipleChoiceField(Skill.objects.all(), required=False)
    pro_level_min = forms.TypedChoiceField(choices=VACANCY_PRO_LEVELS, required=False, coerce=int)
    pro_level_max = forms.TypedChoiceField(choices=VACANCY_PRO_LEVELS, required=False, coerce=int)
    interview_directions = forms.ModelMultipleChoiceField(
        # В Стаффе меньше 70 активных направлений
        queryset=Department.objects.filter(
            is_deleted=False,
            kind=DEPARTMENT_KINDS.direction,
        ),
        required=False,
        help_text=(
            'Направления, в которых как будто бы проводились секции. '
            'В фильтрации не участвуют, нужны только для вычисления фактора '
            'interview_by_department (FEMIDA-5023)'
        ),
    )
    formula = forms.CharField(widget=forms.Textarea, required=False)

    def clean_formula(self):
        return self.cleaned_data.get('formula') or config.PROPOSALS_RANKING_FORMULA

    def clean(self):
        cleaned_data = super().clean()
        interview_directions = cleaned_data.get('interview_directions') or []
        cleaned_data['interview_direction_ids'] = {d.id for d in interview_directions}
        return cleaned_data


class VacancyAdmin(admin.ModelAdmin):

    raw_id_fields = (
        'created_by',
        'instead_of',
        'department',
        'abc_services',
    )
    readonly_fields = (
        'touched_at',
    )

    search_fields = (
        'id',
        'name',
        'startrek_key',
        'budget_position_id',
    )

    list_display = (
        'name',
        'status',
        'resolution',
        'startrek_url',
        'deadline',
        'budget_position_id',
    )

    list_filter = (
        'status',
        'resolution',
        'priority',
        'replacement_reason',
    )

    inlines = (
        VacancySkillInlineAdmin,
        VacancyMembershipInlineAdmin,
        VacancyCityInlineAdmin,
        VacancyLocationInlineAdmin,
        VacancyOfficeInlineAdmin,
    )

    def startrek_url(self, obj):
        return get_st_changelog_url(obj.startrek_key)

    def get_urls(self):
        url_name = '%s_%s_rank' % (self.model._meta.app_label, self.model._meta.model_name)
        urlpatterns = [
            url(r'^rank/$', self.admin_site.admin_view(self.ranking_view), name=url_name),
        ]
        urlpatterns.extend(super().get_urls())
        return urlpatterns

    def get_list_display(self, request):
        if request.user.is_publication_manager:
            return (
                'id',
                'name',
                'publication_title',
                'is_published',
            )
        return self.list_display

    def get_list_filter(self, request):
        if request.user.is_publication_manager:
            return ('is_published',)
        return self.list_filter

    def get_search_fields(self, request):
        if request.user.is_publication_manager:
            return (
                'id',
                'name',
                'publication_title',
            )
        return self.search_fields

    def get_inline_instances(self, request, obj=None):
        if request.user.is_publication_manager:
            return []
        return super().get_inline_instances(request, obj)

    def get_fields(self, request, obj=None):
        if request.user.is_publication_manager:
            return (
                'name',
                'publication_title',
                'publication_content',
                'is_published',
            )
        return super().get_fields(request, obj)

    def ranking_view(self, request, **kwargs):
        """
        Вьюха для тестирования формулы ранжирования массового предложения на вакансии
        """
        form = VacancyRankingAdminForm(data=request.GET)

        if form.is_valid():
            search_data = form.cleaned_data
            formula = search_data.pop('formula')

            qs = (
                models.Vacancy.objects
                .filter(
                    status__in=ACTIVE_VACANCY_STATUSES._db_values,
                    is_hidden=False,
                )
            )
            qs = get_suitable_vacancies(qs, **search_data)
            for factor in VACANCY_FACTORS:
                qs = factor.annotate_qs(qs)

            FakeApplication = namedtuple('Application', ['vacancy', 'proposal_factors'])
            applications = [
                FakeApplication(vacancy, get_factors(vacancy, search_data))
                for vacancy in qs
            ]
            applications = rank_proposals(applications, formula)
        else:
            applications = []
            formula = None

        context = {
            'opts': models.Vacancy._meta,
            'form': form,
            'applications': applications,
            'formula': formula,
            'factors': VACANCY_FACTORS,
        }
        return TemplateResponse(request, 'admin/ranking.html', context)


class VacancyGroupMembershipInlineAdmin(admin.StackedInline):

    model = models.VacancyGroupMembership

    raw_id_fields = (
        'member',
    )


class VacancyGroupAdmin(admin.ModelAdmin):

    list_display = (
        'name',
        'created_by',
        'is_active',
    )

    list_filter = (
        'is_active',
    )

    inlines = (
        VacancyGroupMembershipInlineAdmin,
    )


class VacancySkillAdmin(admin.ModelAdmin):

    search_fields = (
        'skill__name',
    )

    list_display = (
        'vacancy',
        'skill',
        'is_required',
    )


class VacancyMembershipAdmin(admin.ModelAdmin):

    raw_id_fields = (
        'member',
        'vacancy',
        'department_user',
    )

    search_fields = (
        'vacancy__name',
        'member__username',
    )

    list_display = (
        'vacancy',
        'member',
        'role',
    )

    list_filter = (
        'role',
    )


class VacancyCityAdmin(admin.ModelAdmin):

    raw_id_fields = (
        'vacancy',
    )

    search_fields = (
        'vacancy__id',
        'vacancy__name',
        'city__id',
        'city__name_ru',
        'city__name_en',
    )

    list_display = (
        'id',
        'vacancy',
        'city',
    )


class VacancyOfficeAdmin(admin.ModelAdmin):

    raw_id_fields = (
        'vacancy',
    )

    search_fields = (
        'vacancy__id',
        'vacancy__name',
        'office__id',
        'office__name_ru',
        'office__name_en',
    )

    list_display = (
        'id',
        'vacancy',
        'office',
    )


class VacancyLocationAdmin(admin.ModelAdmin):

    raw_id_fields = (
        'vacancy',
    )

    search_fields = (
        'vacancy__id',
        'vacancy__name',
        'location__id',
        'location__geo_id',
        'location__geocoder_uri',
        'location__name_ru',
        'location__name_en',
    )

    list_display = (
        'id',
        'vacancy',
        'location',
    )


class SubmissionFormAdmin(admin.ModelAdmin):

    search_fields = (
        '=id',
        '=forms_constructor_id',
        '=vacancies__id',
        'title',
        'url',
    )

    list_display = (
        'id',
        'forms_constructor_id',
        'title',
        'url',
        'vacancy_ids',
    )

    raw_id_fields = (
        'vacancies',
    )

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.prefetch_related('vacancies')

    def vacancy_ids(self, obj):
        return ', '.join(str(v.id) for v in obj.vacancies.all())


class VacancyHistoryAdmin(admin.ModelAdmin):

    search_fields = (
        'vacancy__id',
        'vacancy__name',
    )

    list_display = (
        'vacancy',
        'status',
        'resolution',
        'changed_at',
    )

    raw_id_fields = (
        'vacancy',
    )


class PublicationSubscriptionAdmin(admin.ModelAdmin):

    search_fields = (
        '=id',
        '=shown_vacancies__id',
        'created_by__username',
    )

    list_display = (
        'id',
        'created_by',
        'created',
    )

    raw_id_fields = (
        'created_by',
        'shown_vacancies',
        'abc_services',
        'department',
        'professions',
        'skills',
        'cities',
    )


class VacancyWorkModeAdmin(admin.ModelAdmin):
    list_display = ('vacancy', 'work_mode')
    search_fields = (
        'vacancy__id',
        'work_mode__slug',
        'vacancy__name',
        'work_mode__name_ru',
        'work_mode__name_en',
    )


admin.site.register(models.Vacancy, VacancyAdmin)
admin.site.register(models.VacancyGroup, VacancyGroupAdmin)
admin.site.register(models.VacancySkill, VacancySkillAdmin)
admin.site.register(models.VacancyMembership, VacancyMembershipAdmin)
admin.site.register(models.VacancyCity, VacancyCityAdmin)
admin.site.register(models.VacancyOffice, VacancyOfficeAdmin)
admin.site.register(models.VacancyLocation, VacancyLocationAdmin)
admin.site.register(models.SubmissionForm, SubmissionFormAdmin)
admin.site.register(models.VacancyHistory, VacancyHistoryAdmin)
admin.site.register(models.PublicationSubscription, PublicationSubscriptionAdmin)
admin.site.register(models.VacancyWorkMode, VacancyWorkModeAdmin)
