import re

from typing import Union

from django.db.models import F, Value, QuerySet
from django.db.models.functions import Coalesce

from intranet.femida.src.core.db import SearchQuery
from intranet.femida.src.core.models import City
from intranet.femida.src.professions.models import Profession
from intranet.femida.src.skills.models import Skill
from intranet.femida.src.staff.models import Department
from intranet.femida.src.staff.models import Service
from intranet.femida.src.vacancies.choices import (
    PUBLICATION_ACTIVE_STATUS_LIST,
    VACANCY_PRO_LEVELS,
    VACANCY_STATUSES,
)
from intranet.femida.src.vacancies.models import Vacancy


def get_vacancy_group_label(vacancy_group):

    return {
        'id': vacancy_group.id,
        'name': vacancy_group.name,
        'active_vacancies_count': (
            vacancy_group.vacancies
            .filter(status=VACANCY_STATUSES.in_progress)
            .count()
        ),
    }


def filter_vacancies_by_pro_level(queryset, pro_level_min=None, pro_level_max=None):
    pro_level_min = pro_level_min or 0
    pro_level_max = pro_level_max or VACANCY_PRO_LEVELS.expert

    return (
        queryset
        .annotate(nonnull_pro_level_min=Coalesce(F('pro_level_min'), Value(0)))
        .filter(
            pro_level_max__gte=pro_level_min,
            nonnull_pro_level_min__lte=pro_level_max,
        )
    )


def get_suitable_vacancies(
    queryset: 'QuerySet[Vacancy]',
    professions: 'QuerySet[Profession]' = None,
    cities: 'QuerySet[City]' = None,
    skills: 'QuerySet[Skill]' = None,
    abc_services: 'QuerySet[Service]' = None,
    pro_level_min: int = None,
    pro_level_max: int = None,
    only_active: bool = None,
    department: Union[int, Department] = None,
    departments: 'QuerySet[Department]' = None,
    text: str = None,
    external_url: str = None,
    **kwargs,
) -> 'QuerySet[Vacancy]':
    """
    Поиск пересекающихся вакансий
    Ищем вакансии, которые удовлетворяют всем условиям ниже:
    - профессия на вакансии - одна из professions;
    - есть хоть один город из cities;
    - есть хоть один навык из skills;
    - есть хоть один ABC-сервис из abc_services;
    - диапазон уровней пересекается с диапазоном
      от pro_level_min до pro_level_max
    - входит в дерево хотя бы одного из department
    """
    if professions:
        queryset = queryset.filter(profession__in=professions)

    if cities:
        queryset = queryset.filter(cities__in=cities)

    if skills:
        queryset = queryset.filter(skills__in=skills)

    if abc_services:
        queryset = queryset.filter(abc_services__in=abc_services)

    if pro_level_min or pro_level_max:
        queryset = filter_vacancies_by_pro_level(queryset, pro_level_min, pro_level_max)

    if only_active:
        queryset = queryset.filter(status__in=PUBLICATION_ACTIVE_STATUS_LIST)

    if department:
        department_id = department if isinstance(department, int) else department.id
        departments_qs = Department.objects.in_tree(department_id)
        queryset = queryset.filter(department__in=departments_qs)

    if departments:
        department_ids = list(departments.values_list('id', flat=True))
        departments_qs = Department.objects.in_trees(department_ids)
        queryset = queryset.filter(department__in=departments_qs)

    if text:
        search_query = SearchQuery(text)
        queryset = queryset.filter(search_vector=search_query)

    if external_url:
        # Примерная регулярка url объявления на новом СВ
        new_url_rgx = re.compile(r'/vacancies/(([\w-]+-)|)(?P<publication_id>\d+)([/?#]|$)')
        match = new_url_rgx.search(external_url)
        publication_id = match.group('publication_id') if match else None
        if publication_id:
            queryset = queryset.filter(publications=publication_id)
        else:
            queryset = queryset.filter(submission_forms__url__icontains=external_url)

    return queryset.distinct()
