from django.core.management.base import BaseCommand
from django.db import transaction

from intranet.femida.src.oebs.api import OebsAPI
from intranet.femida.src.staff.models import ValueStream
from intranet.femida.src.vacancies.choices import VACANCY_STATUSES
from intranet.femida.src.vacancies.models import Vacancy


class Command(BaseCommand):

    help = 'Migrate value streams from OEBS'

    def add_arguments(self, parser):
        parser.add_argument('--dry-run', action='store_true')
        parser.add_argument('--include-closed', action='store_true')
        parser.add_argument('--include-filled', action='store_true')

    def handle(self, *args, **options):
        value_streams = dict(
            ValueStream.objects
            .filter(oebs_product_id__isnull=False)
            .values_list('oebs_product_id', 'id')
        )
        vacancies = Vacancy.unsafe.filter(budget_position_id__isnull=False)
        if not options['include_closed']:
            vacancies = vacancies.exclude(status=VACANCY_STATUSES.closed)
        if not options['include_filled']:
            vacancies = vacancies.filter(value_stream__isnull=True)

        vacancies_to_update = []
        failed_bp_count = 0
        unknown_vs_count = 0

        for i, vacancy in enumerate(vacancies):
            if i % 100 == 0:
                print(f'Processed {i} vacancies')
            try:
                # TODO: надо на какую-то дату получать или там берётся уже на правильную/текущую?
                oebs_data = OebsAPI.get_budget_position_form_data(vacancy.budget_position_id)
            except Exception as exc:
                failed_bp_count += 1
                print('Failed to get BP', vacancy.budget_position_id, exc)
                continue
            oebs_product_id = oebs_data.get('positionAnalitics', {}).get('productID')
            if oebs_product_id in value_streams:
                vacancy.value_stream_id = value_streams[oebs_product_id]
                vacancies_to_update.append(vacancy)
            else:
                unknown_vs_count += 1

        if not options['dry_run']:
            with transaction.atomic():
                Vacancy.objects.bulk_update(vacancies_to_update, ['value_stream_id'], 1000)

        print('----')
        print(len(vacancies_to_update), 'vacancies were updated')
        print(f'Could not get BP info for {failed_bp_count} vacancies')
        print(f'Could not find VS for {unknown_vs_count} vacancies')
