from collections import defaultdict

from django.utils.translation import ugettext_noop as _
from rest_framework import serializers

from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.core.serializers import MapperField
from intranet.femida.src.utils.diffs import get_strings_diff
from intranet.femida.src.vacancies.choices import VACANCY_FIELDS_TRANSLATIONS
from . import models
from . import choices


class VacancyDiffEstimationSerializer(serializers.ModelSerializer):

    cities = serializers.SerializerMethodField()
    offices = serializers.SerializerMethodField()
    locations = serializers.SerializerMethodField()
    department = serializers.CharField(source='department.name', default=None)
    is_hidden = serializers.SerializerMethodField()
    profession = serializers.CharField(source='profession.name', default=None)
    professional_sphere = serializers.CharField(source='professional_sphere.name', default=None)
    pro_level_min = MapperField(choices.VACANCY_PRO_LEVELS_TRANSLATIONS)
    pro_level_max = MapperField(choices.VACANCY_PRO_LEVELS_TRANSLATIONS)
    skills = serializers.SerializerMethodField()
    is_published = MapperField({True: 'Да', False: 'Нет'})

    head = serializers.SerializerMethodField()
    hiring_manager = serializers.SerializerMethodField()
    main_recruiter = serializers.SerializerMethodField()
    recruiters = UserSerializer(many=True, source='researchers')
    interviewers = UserSerializer(many=True)
    responsibles = UserSerializer(many=True)
    observers = UserSerializer(many=True)

    def get_cities(self, obj):
        return ', '.join([c.name_ru for c in obj.cities.all()])

    def get_offices(self, obj):
        return ', '.join([office.name_ru for office in obj.offices.all()])

    def get_locations(self, obj):
        return ', '.join([location.name_ru for location in obj.locations.all()])

    def _get_user(self, user):
        return UserSerializer(instance=user).data if user else None

    def get_head(self, obj):
        return self._get_user(obj.head)

    def get_hiring_manager(self, obj):
        return self._get_user(obj.hiring_manager)

    def get_main_recruiter(self, obj):
        return self._get_user(obj.main_recruiter)

    def get_is_hidden(self, obj):
        return _('да') if obj.is_hidden else _('нет')

    def get_skills(self, obj):
        return ', '.join([s.name for s in obj.skills.all()])

    def to_representation(self, data):
        data = super().to_representation(data)
        for key in data:
            if data[key] is None:
                data[key] = ''
        return data

    class Meta:
        model = models.Vacancy
        fields = (
            'name',
            'is_hidden',
            'deadline',
            'department',
            'cities',
            'offices',
            'locations',
            'professional_sphere',
            'profession',
            'pro_level_min',
            'pro_level_max',
            'skills',
            'responsibilities_comment',
            'additional_qualities_comment',
            'recommendations_comment',
            'kpi_comment',
            'other_comment',
            'publication_title',
            'publication_content',
            'is_published',

            'head',
            'hiring_manager',
            'main_recruiter',
            'recruiters',
            'interviewers',
            'responsibles',
            'observers',
        )

    @classmethod
    def _get_user_list_diff(cls, a, b):
        usernames_a = set(i['username'] for i in a)
        usernames_b = set(i['username'] for i in b)

        if usernames_a == usernames_b:
            return None

        all_users = {i['username']: i for i in (a + b)}
        signed_usernames = (
            ('-', usernames_a - usernames_b),
            ('=', usernames_a & usernames_b),
            ('+', usernames_b - usernames_a),
        )

        result = []
        for sign, usernames in signed_usernames:
            for username in usernames:
                data = all_users.get(username)
                result.append([[sign, '{fullname} ({username}@)'.format(**data)]])

        return result

    @classmethod
    def _get_users_diff(cls, a, b):
        new_a = [a] if a else []
        new_b = [b] if b else []
        return cls._get_user_list_diff(new_a, new_b)

    @classmethod
    def get_diff(cls, a, b):
        compare_functions_map = defaultdict(lambda: get_strings_diff)
        for i in ('head', 'hiring_manager', 'main_recruiter'):
            compare_functions_map[i] = cls._get_users_diff
        for i in ('recruiters', 'interviewers', 'responsibles', 'observers'):
            compare_functions_map[i] = cls._get_user_list_diff

        diff = []
        for k in a:
            if a[k] == b[k]:
                continue
            compare_result = compare_functions_map[k](a[k], b[k])
            if not compare_result:
                continue

            diff.append((VACANCY_FIELDS_TRANSLATIONS.get(k), compare_result))

        return diff
