import random
import waffle

from django.conf import settings
from rest_framework import serializers

from intranet.femida.src.core.serializers import (
    MapperField,
    StartrekLocalField,
    StartrekUniqueField,
)
from intranet.femida.src.staff.helpers import get_hr_analysts, get_recruitment_partners
from intranet.femida.src.startrek.utils import ContractTypesEnum, ProLevelEnum, SalarySystemEnum
from intranet.femida.src.vacancies.choices import (
    VACANCY_CONTRACT_TYPES,
    VACANCY_REPLACEMENT_REASONS,
    WAGE_SYSTEM_TYPES,
    VACANCY_TYPES,
)


VACANCY_TYPES_MAP = {
    VACANCY_TYPES.new: 'новая',
    VACANCY_TYPES.pool: 'новая',
    VACANCY_TYPES.internship: 'новая',
    VACANCY_TYPES.replacement: 'замена',
    VACANCY_TYPES.autohire: 'масснайм',
}


REPLACEMENT_REASONS_MAP = {
    VACANCY_REPLACEMENT_REASONS.rotation: 'Сотрудник переводится в другое подразделение',
    VACANCY_REPLACEMENT_REASONS.dismissal: 'Сотрудник увольняется',
    VACANCY_REPLACEMENT_REASONS.maternity_leave: 'Сотрудник уходит в декретный отпуск',
}


CONTRACT_TYPES_MAP = {
    None: '',
    '': '',
    VACANCY_CONTRACT_TYPES.fixed: ContractTypesEnum.fixed,
    VACANCY_CONTRACT_TYPES.open: ContractTypesEnum.open,
}


PRO_LEVELS_MAP = {
    None: '',
    1: ProLevelEnum.intern,
    2: ProLevelEnum.junior,
    3: ProLevelEnum.middle,
    4: ProLevelEnum.senior,
    5: ProLevelEnum.lead,
    6: ProLevelEnum.expert,
}

PRO_LEVELS_MAP_EN = {
    None: '',
    1: ProLevelEnum.intern_en,
    2: ProLevelEnum.junior_en,
    3: ProLevelEnum.middle_en,
    4: ProLevelEnum.senior_en,
    5: ProLevelEnum.lead_en,
    6: ProLevelEnum.expert_en,
}


WAGE_SYSTEMS_MAP = {
    WAGE_SYSTEM_TYPES.fixed: SalarySystemEnum.fixed,
    WAGE_SYSTEM_TYPES.piecework: SalarySystemEnum.piecework,
    WAGE_SYSTEM_TYPES.hourly: SalarySystemEnum.hourly,
}

WAGE_SYSTEMS_MAP_EN = {
    WAGE_SYSTEM_TYPES.fixed: SalarySystemEnum.fixed_en,
    WAGE_SYSTEM_TYPES.piecework: SalarySystemEnum.piecework_en,
    WAGE_SYSTEM_TYPES.hourly: SalarySystemEnum.hourly_en,
}


_pseudo_bool = lambda x: 'yes' if bool(x) else 'no'


class IssueFieldsSerializer(serializers.Serializer):
    """
    Сериализатор полей JOB-тикета
    """

    # Note: до релиза FEMIDA-3497 для JOB-тикетов в качестве unique
    # использовался формат femida_vacancy_<id>
    unique = StartrekUniqueField(queue=settings.STARTREK_JOB_QUEUE)

    analyst = serializers.SerializerMethodField()
    assignee = serializers.SerializerMethodField(method_name='get_analyst')
    recruitmentPartner = serializers.SerializerMethodField()
    activity = serializers.SerializerMethodField()
    bpNumber = serializers.ReadOnlyField(source='budget_position_id')
    fixVersions = serializers.SerializerMethodField()
    professionalSphere = serializers.ReadOnlyField(source='profession.startrek_id')

    # 2 поля, которые несут в себе одну и ту же суть.
    # Но оставить какое-то одно пока не могут.
    # internship - стажировка
    # hrInternship - готовы взять стажера
    internship = serializers.SerializerMethodField()
    hrInternship = serializers.SerializerMethodField(method_name='get_internship')

    salarySystem = MapperField(WAGE_SYSTEMS_MAP, source='wage_system')
    maxSalary = serializers.SerializerMethodField()
    followers = serializers.SerializerMethodField()
    reason = serializers.ReadOnlyField()
    hrVacancyLevel = MapperField(PRO_LEVELS_MAP, source='pro_level_max')
    quitDate = serializers.DateField(source='quit_date', required=False)
    contractType = MapperField(CONTRACT_TYPES_MAP, source='contract_type')
    vacancyType = MapperField(VACANCY_TYPES_MAP, source='type')
    firedName = serializers.ReadOnlyField(source='instead_of.username', required=False)
    newDepartment = serializers.ReadOnlyField(source='department.name', required=False)
    priority = serializers.ReadOnlyField()
    oebs_product_id = StartrekLocalField.new(
        settings.STARTREK_JOB_OEBS_PRODUCT_ID_FIELD,
        source='value_stream.oebs_product_id',
    )
    geography2 = serializers.ReadOnlyField(source='geography.startrek_id', required=False)
    countryTo = serializers.SerializerMethodField()
    mainProduct = StartrekLocalField.new(
        settings.STARTREK_JOB_MAIN_PRODUCT_ID_FIELD,
        field_class=serializers.SerializerMethodField,
        method_name='get_mainProduct',
    )
    workMode = StartrekLocalField.new(
        settings.STARTREK_JOB_WORK_MODE_ID_FIELD,
        field_class=serializers.SerializerMethodField,
        method_name='get_workMode',
    )

    def get_workMode(self, obj):
        return [mode.name_en for mode in obj.get('work_mode', [])]

    def get_mainProduct(self, obj):
        if obj.get('value_stream'):
            vs = obj.get('value_stream')
            return vs.service.id if vs.service else None

        # Временный костыль, чтобы из hire_order abc_services уходили в поле primaryProduct
        # Уберем в новой версии формы, когда все переедут на новую схему ручки
        if obj.get('abc_services'):
            return obj.get('abc_services')[0].id

    def get_countryTo(self, obj):
        geography = obj.get('geography')
        return geography and geography.get_kind_display()

    def get_analyst(self, obj):
        # Случайный HR-аналитик из списка аналитиков данного подразделения
        if not hasattr(self, '_cached_analyst'):
            hr_analysts = get_hr_analysts(obj['department'])
            self._cached_analyst = random.choice(hr_analysts).username if hr_analysts else None
        return self._cached_analyst

    def get_recruitmentPartner(self, obj):
        # Случайный человек из списка рекрутмент-партнёров данного подразделения
        partners = get_recruitment_partners(obj['department'])
        return random.choice(partners).username if partners else None

    def get_activity(self, obj):
        # Note: Трекер в тестинге использует продовые ABC-сервисы,
        # из-за чего многие сервисы отличаются, и в тестинге Фемиды
        # постоянно возникают проблемы с созданием JOB-тикетов.
        # Хочется иметь возможность отключить прокидывание ABC-сервисов в тикет.
        if waffle.switch_is_active('ignore_vacancy_abc_services'):
            return []
        return [s.id for s in obj.get('abc_services', [])]

    def get_fixVersions(self, obj):
        if obj.get('cities'):
            return [city.startrek_id for city in obj['cities']]
        return []

    def get_internship(self, obj):
        return _pseudo_bool(obj['type'] == VACANCY_TYPES.internship)

    def get_maxSalary(self, obj):
        if obj.get('max_salary'):
            if obj.get('currency'):
                return '%s %s' % (obj['max_salary'], obj['currency'])
            else:
                return str(obj['max_salary'])

    def get_followers(self, obj):
        return [user.username for user in obj.get('followers', [])]

    def to_representation(self, data):
        """
        Если в итоге какие-то значения пустые, то не нужно их отправлять в ST
        """
        data = super().to_representation(data)
        for key in list(data):
            if not data[key]:
                data.pop(key)
        return data


class IssueContextSerializer(serializers.Serializer):
    """
    Сериализатор контекста, которым будет отрендерено описание JOB-тикета
    """
    id = serializers.ReadOnlyField()
    type = serializers.ReadOnlyField()
    name = serializers.ReadOnlyField()
    profession = serializers.SerializerMethodField()
    pro_level = serializers.SerializerMethodField()
    department = serializers.SerializerMethodField()
    value_stream = serializers.SerializerMethodField()
    locations = serializers.SerializerMethodField()
    geography = serializers.SerializerMethodField()
    replacement_department = serializers.SerializerMethodField()
    instead_of = serializers.ReadOnlyField(source='instead_of.username')
    quit_date = serializers.DateField(required=False)
    contract_type = MapperField(CONTRACT_TYPES_MAP)
    replacement_reason = MapperField(REPLACEMENT_REASONS_MAP)
    budget_source = serializers.ReadOnlyField()
    budget_position_id = serializers.ReadOnlyField()
    max_salary = serializers.ReadOnlyField()
    currency = serializers.ReadOnlyField(source='currency.code')
    wage_system = serializers.SerializerMethodField()
    work_mode = serializers.SerializerMethodField()
    followers = serializers.SerializerMethodField()
    reason = serializers.ReadOnlyField(required=False)

    def _get_department(self, obj, key):
        if obj.get(key):
            return {
                'url': obj[key].url,
                'name': obj[key].name,
                'name_en': obj[key].name_en,
            }

    def _flatten_name(self, obj, key, sep=', '):
        if obj.get(key):
            en_list = list(map(lambda x: x.name_en, obj[key]))
            ru_list = list(map(lambda x: x.name_ru, obj[key]))
            return f'{sep.join(ru_list)} / {sep.join(en_list)}'

    def get_profession(self, obj):
        return f'{obj["profession"].name}/{obj["profession"].name_en}'

    def get_pro_level(self, obj):
        pro_level_min_verbose, pro_level_min_verbose_en = (
            PRO_LEVELS_MAP[obj.get('pro_level_min')],
            PRO_LEVELS_MAP_EN[obj.get('pro_level_min')],
        )
        pro_level_max_verbose, pro_level_max_verbose_en = (
            PRO_LEVELS_MAP[obj.get('pro_level_max')],
            PRO_LEVELS_MAP_EN[obj.get('pro_level_max')]
        )
        if not pro_level_min_verbose or pro_level_min_verbose == pro_level_max_verbose:
            return f'{pro_level_max_verbose}/{pro_level_max_verbose_en}'
        else:
            return f'{pro_level_min_verbose} — {pro_level_max_verbose} /' \
                   + f' {pro_level_min_verbose_en} — {pro_level_max_verbose_en}'

    def get_department(self, obj):
        return self._get_department(obj, 'department')

    def get_locations(self, obj):
        # why this separator https://st.yandex-team.ru/FEMIDA-7244#624ec5daf09bb43dab604894
        return self._flatten_name(obj, 'locations', sep='; ')

    def get_value_stream(self, obj):
        if obj.get('value_stream'):
            return {
                'slug': obj['value_stream'].slug,
                'name': obj['value_stream'].name_ru,
                'name_en': obj['value_stream'].name_en,
            }

    def get_geography(self, obj):
        if obj.get('geography'):
            return {
                'name': obj['geography'].name_ru,
            }

    def get_replacement_department(self, obj):
        return self._get_department(obj, 'replacement_department')

    def get_followers(self, obj):
        return [u.username for u in obj.get('followers', [])]

    def get_wage_system(self, obj):
        if obj.get('wage_system'):
            return f'{WAGE_SYSTEMS_MAP[obj["wage_system"]]} / {WAGE_SYSTEMS_MAP_EN[obj["wage_system"]]} '

    def get_work_mode(self, obj):
        return self._flatten_name(obj, 'work_mode')
